#line 1
##############################################################################
#      $URL: http://perlcritic.tigris.org/svn/perlcritic/trunk/distributions/Perl-Critic/lib/Perl/Critic/Policy/RegularExpressions/ProhibitEnumeratedClasses.pm $
#     $Date: 2011-12-21 14:40:10 -0800 (Wed, 21 Dec 2011) $
#   $Author: thaljef $
# $Revision: 4106 $
##############################################################################

package Perl::Critic::Policy::RegularExpressions::ProhibitEnumeratedClasses;

use 5.006001;
use strict;
use warnings;

use Carp qw(carp);
use English qw(-no_match_vars);
use List::MoreUtils qw(all);
use Readonly;

use Perl::Critic::Utils qw{ :booleans :severities hashify };

use base 'Perl::Critic::Policy';

our $VERSION = '1.117';

#-----------------------------------------------------------------------------

Readonly::Scalar my $DESC => q{Use named character classes};
Readonly::Scalar my $EXPL => [248];

Readonly::Array my @PATTERNS => (  # order matters: most to least specific
   [q{ },'\\t','\\r','\\n']      => ['\\s', '\\S'],
   ['A-Z','a-z','0-9','_']       => ['\\w', '\\W'], # RT 69322
   ['A-Z','a-z']                 => ['[[:alpha:]]','[[:^alpha:]]'],
   ['A-Z']                       => ['[[:upper:]]','[[:^upper:]]'],
   ['a-z']                       => ['[[:lower:]]','[[:^lower:]]'],
   ['0-9']                       => ['\\d','\\D'],
   ['\w']                        => [undef, '\\W'],
   ['\s']                        => [undef, '\\S'],
);

#-----------------------------------------------------------------------------

sub supported_parameters { return qw()                    }
sub default_severity     { return $SEVERITY_LOWEST        }
sub default_themes       { return qw( core pbp cosmetic unicode ) }
sub applies_to           { return qw(PPI::Token::Regexp::Match
                                     PPI::Token::Regexp::Substitute
                                     PPI::Token::QuoteLike::Regexp) }

#-----------------------------------------------------------------------------


sub violates {
    my ( $self, $elem, $document ) = @_;

    # optimization: don't bother parsing the regexp if there are no character classes
    return if $elem !~ m/\[/xms;

    my $re = $document->ppix_regexp_from_element( $elem ) or return;
    $re->failures() and return;

    my $anyofs = $re->find( 'PPIx::Regexp::Structure::CharClass' )
        or return;
    foreach my $anyof ( @{ $anyofs } ) {
        my $violation;
        $violation = $self->_get_character_class_violations( $elem, $anyof )
            and return $violation;
    }

    return;  # OK
}

sub _get_character_class_violations {
    my ($self, $elem, $anyof) = @_;

    my %elements;
    foreach my $element ( $anyof->children() ) {
        $elements{ _fixup( $element ) } = 1;
    }

    for (my $i = 0; $i < @PATTERNS; $i += 2) {  ##no critic (CStyleForLoop)
        if (all { exists $elements{$_} } @{$PATTERNS[$i]}) {
            my $neg = $anyof->negated();
            my $improvement = $PATTERNS[$i + 1]->[$neg ? 1 : 0];
            next if !defined $improvement;

            if ($neg && ! defined $PATTERNS[$i + 1]->[0]) {
                # the [^\w] => \W rule only applies if \w is the only token.
                # that is it does not apply to [^\w\s]
                next if 1 != scalar keys %elements;
            }

            my $orig = join q{}, '[', ($neg ? q{^} : ()), @{$PATTERNS[$i]}, ']';
            return $self->violation( $DESC . " ($orig vs. $improvement)", $EXPL, $elem );
        }
    }

    return;  # OK
}

Readonly::Hash my %ORDINALS => (
    ord "\n"    => '\\n',
    ord "\f"    => '\\f',
    ord "\r"    => '\\r',
    ord q< >    => q< >,
);

sub _fixup {
    my ( $element ) = @_;
    if ( $element->isa( 'PPIx::Regexp::Token::Literal' ) ) {
        my $ord = $element->ordinal();
        exists $ORDINALS{$ord} and return $ORDINALS{$ord};
        return $element->content();
    } elsif ( $element->isa( 'PPIx::Regexp::Node' ) ) {
        return join q{}, map{ _fixup( $_ ) } $element->elements();
    } else {
        return $element->content();
    }
}

1;

__END__

#-----------------------------------------------------------------------------

#line 196

# Local Variables:
#   mode: cperl
#   cperl-indent-level: 4
#   fill-column: 78
#   indent-tabs-mode: nil
#   c-indentation-style: bsd
# End:
# ex: set ts=8 sts=4 sw=4 tw=78 ft=perl expandtab shiftround :
