#!/usr/bin/env python3

import importlib.util
import os

from fenrirscreenreader.core.i18n import _

# Load base configuration class
_base_path = os.path.join(os.path.dirname(__file__), "..", "config_base.py")
_spec = importlib.util.spec_from_file_location("config_base", _base_path)
_module = importlib.util.module_from_spec(_spec)
_spec.loader.exec_module(_module)
config_command = _module.config_command


class command(config_command):
    def __init__(self):
        super().__init__()

    def get_description(self):
        return "Set character echo mode"

    def run(self):
        current_mode = self.get_setting("keyboard", "charEchoMode", "1")

        # Present current mode
        mode_descriptions = {
            "0": "None - no character echo",
            "1": "Always - echo all typed characters",
            "2": "Caps Lock - echo only when caps lock is on",
        }

        current_description = mode_descriptions.get(current_mode, "Unknown")
        self.present_text(
            f"Current character echo mode: {current_description}"
        )

        # Cycle through the three modes
        modes = ["0", "1", "2"]
        try:
            current_index = modes.index(current_mode)
            next_index = (current_index + 1) % len(modes)
            new_mode = modes[next_index]
        except ValueError:
            new_mode = "1"  # Default to always

        success = self.set_setting("keyboard", "charEchoMode", new_mode)

        if success:
            new_description = mode_descriptions[new_mode]
            self.present_text(f"Character echo mode set to: {new_description}")
            self.play_sound("Accept")
        else:
            self.present_text("Failed to change character echo mode")
            self.play_sound("Error")
