#load "./paths.cake"
#load "./packages.cake"
#load "./version.cake"
#load "./credentials.cake"

public class BuildParameters
{
    public string Target { get; }
    public string Configuration { get; }
    public bool IsLocalBuild { get; }
    public bool IsRunningOnUnix { get; }
    public bool IsRunningOnWindows { get; }
    public bool IsRunningOnAppVeyor { get; }
    public bool IsRunningOnGitHubActions { get; }
    public bool IsPullRequest { get; }
    public bool IsMainCakeRepo { get; }
    public bool IsMainCakeBranch { get; }
    public bool IsDevelopCakeBranch { get; }
    public bool IsTagged { get; }
    public bool IsPublishBuild { get; }
    public bool IsReleaseBuild { get; }
    public bool SkipGitVersion { get; }
    public bool SkipSigning { get; }
    public BuildCredentials GitHub { get; }
    public TwitterCredentials Twitter { get; }
    public ReleaseNotes ReleaseNotes { get; }
    public BuildVersion Version { get; set; }
    public BuildPaths Paths { get; }
    public BuildPackages Packages { get; }
    public bool PublishingError { get; set; }
    public DotNetMSBuildSettings MSBuildSettings { get; }
    public CodeSigningCredentials CodeSigning { get; }

    public bool ShouldPublish
    {
        get
        {
            return IsRunningOnGitHubActions
                && IsRunningOnWindows
                && !IsLocalBuild
                && !IsPullRequest
                && IsMainCakeRepo
                && IsTagged;
        }
    }

    public bool ShouldPublishToAzureDevOps
    {
        get
        {
            return IsRunningOnGitHubActions
                && IsRunningOnWindows
                && !IsLocalBuild
                && !IsPullRequest
                && IsMainCakeRepo
                && (IsTagged || IsDevelopCakeBranch);
        }
    }


    public bool ShouldSignPackages { get; }
    public bool CanPostToTwitter
    {
        get
        {
            return !string.IsNullOrEmpty(Twitter.ConsumerKey) &&
                !string.IsNullOrEmpty(Twitter.ConsumerSecret) &&
                !string.IsNullOrEmpty(Twitter.AccessToken) &&
                !string.IsNullOrEmpty(Twitter.AccessTokenSecret);
        }
    }

    public BuildParameters (ISetupContext context)
    {
        if (context == null)
        {
            throw new ArgumentNullException("context");
        }

        var buildSystem = context.BuildSystem();

        Target = context.TargetTask.Name;
        Configuration = context.Argument("configuration", "Release");
        IsLocalBuild = buildSystem.IsLocalBuild;
        IsRunningOnUnix = context.IsRunningOnUnix();
        IsRunningOnWindows = context.IsRunningOnWindows();
        IsRunningOnAppVeyor = buildSystem.AppVeyor.IsRunningOnAppVeyor;
        IsRunningOnGitHubActions = buildSystem.GitHubActions.IsRunningOnGitHubActions;
        IsPullRequest = buildSystem.AppVeyor.Environment.PullRequest.IsPullRequest || buildSystem.GitHubActions.Environment.PullRequest.IsPullRequest;
        IsTagged = IsBuildTagged(buildSystem);
        GitHub = BuildCredentials.GetGitHubCredentials(context);
        Twitter = TwitterCredentials.GetTwitterCredentials(context);
        CodeSigning = CodeSigningCredentials.GetCodeSigningCredentials(context);
        ReleaseNotes = context.ParseReleaseNotes("./ReleaseNotes.md");
        IsPublishBuild = IsPublishing(context.TargetTask.Name);
        IsReleaseBuild = IsReleasing(context.TargetTask.Name);
        SkipSigning = StringComparer.OrdinalIgnoreCase.Equals("True", context.Argument("skipsigning", "False"));
        SkipGitVersion = StringComparer.OrdinalIgnoreCase.Equals("True", context.EnvironmentVariable("CAKE_SKIP_GITVERSION"));
        Version = BuildVersion.Calculate(context, this);
        IsMainCakeBranch = StringComparer.OrdinalIgnoreCase.Equals("main", buildSystem.AppVeyor.Environment.Repository.Branch) || StringComparer.OrdinalIgnoreCase.Equals("main", Version.BranchName);
        IsDevelopCakeBranch = StringComparer.OrdinalIgnoreCase.Equals("develop", buildSystem.AppVeyor.Environment.Repository.Branch) || StringComparer.OrdinalIgnoreCase.Equals("develop", Version.BranchName);
        IsMainCakeRepo = StringComparer.OrdinalIgnoreCase.Equals("cake-build/cake", buildSystem.AppVeyor.Environment.Repository.Name) || StringComparer.OrdinalIgnoreCase.Equals("cake-build", buildSystem.GitHubActions.Environment.Workflow.RepositoryOwner);
        Paths = BuildPaths.GetPaths(context, Configuration, Version.SemVersion);
        Packages = BuildPackages.GetPackages(
            Paths.Directories.NuGetRoot,
            Version.SemVersion,
            new [] {
                "Cake.Core",
                "Cake.Common",
                "Cake.Testing",
                "Cake.Testing.Xunit",
                "Cake.Testing.Xunit.v3",
                "Cake.NuGet",
                "Cake.Tool",
                "Cake.Frosting",
                "Cake.Frosting.Template",
                "Cake.Cli",
                "Cake.DotNetTool.Module"
                });

        var releaseNotes = string.Join(
                                System.Environment.NewLine,
                                ReleaseNotes.Notes.ToArray()
                            );

        MSBuildSettings = new DotNetMSBuildSettings {
                                Version = Version.SemVersion,
                                AssemblyVersion = Version.Version,
                                FileVersion = Version.Version,
                                PackageReleaseNotes =  releaseNotes
                            };

        if (!IsLocalBuild)
        {
            MSBuildSettings.WithProperty("TemplateVersion", Version.SemVersion);
        }


        ShouldSignPackages = (!SkipSigning && ShouldPublish)
                                ||
                                StringComparer.OrdinalIgnoreCase.Equals(
                                    context.EnvironmentVariable("SIGNING_TEST"),
                                    "True"
                                );
    }

    private static bool IsBuildTagged(BuildSystem buildSystem)
    {
        return buildSystem.GitHubActions.IsRunningOnGitHubActions && buildSystem.GitHubActions.Environment.Workflow.RefType == GitHubActionsRefType.Tag;
    }

    private static bool IsReleasing(string target)
    {
        var targets = new [] { "Publish", "Publish-NuGet", "Publish-GitHub-Release" };
        return targets.Any(t => StringComparer.OrdinalIgnoreCase.Equals(t, target));
    }

    private static bool IsPublishing(string target)
    {
        var targets = new [] { "ReleaseNotes", "Create-Release-Notes" };
        return targets.Any(t => StringComparer.OrdinalIgnoreCase.Equals(t, target));
    }
}

