//  Copyright (c) 2011-present, Facebook, Inc.  All rights reserved.
//  This source code is licensed under both the GPLv2 (found in the
//  COPYING file in the root directory) and Apache 2.0 License
//  (found in the LICENSE.Apache file in the root directory).

#pragma once
#include <string>

#include "db/table_properties_collector.h"
#include "rocksdb/types.h"
#include "util/coding.h"
#include "util/string_util.h"

namespace ROCKSDB_NAMESPACE {

// Table Properties that are specific to tables created by SstFileWriter.
struct ExternalSstFilePropertyNames {
  // value of this property is a fixed uint32 number.
  static const std::string kVersion;
  // value of this property is a fixed uint64 number.
  static const std::string kGlobalSeqno;
};

// PropertiesCollector used to add properties specific to tables
// generated by SstFileWriter
class SstFileWriterPropertiesCollector : public InternalTblPropColl {
 public:
  explicit SstFileWriterPropertiesCollector(int32_t version,
                                            SequenceNumber global_seqno)
      : version_(version), global_seqno_(global_seqno) {}

  Status InternalAdd(const Slice& /*key*/, const Slice& /*value*/,
                     uint64_t /*file_size*/) override {
    // Intentionally left blank. Have no interest in collecting stats for
    // individual key/value pairs.
    return Status::OK();
  }

  void BlockAdd(uint64_t /* block_uncomp_bytes */,
                uint64_t /* block_compressed_bytes_fast */,
                uint64_t /* block_compressed_bytes_slow */) override {
    // Intentionally left blank. No interest in collecting stats for
    // blocks.
    return;
  }

  Status Finish(UserCollectedProperties* properties) override {
    // File version
    std::string version_val;
    PutFixed32(&version_val, static_cast<uint32_t>(version_));
    properties->insert({ExternalSstFilePropertyNames::kVersion, version_val});

    // Global Sequence number
    std::string seqno_val;
    PutFixed64(&seqno_val, static_cast<uint64_t>(global_seqno_));
    properties->insert({ExternalSstFilePropertyNames::kGlobalSeqno, seqno_val});

    return Status::OK();
  }

  const char* Name() const override {
    return "SstFileWriterPropertiesCollector";
  }

  UserCollectedProperties GetReadableProperties() const override {
    return {{ExternalSstFilePropertyNames::kVersion, std::to_string(version_)}};
  }

 private:
  int32_t version_;
  SequenceNumber global_seqno_;
};

class SstFileWriterPropertiesCollectorFactory
    : public InternalTblPropCollFactory {
 public:
  explicit SstFileWriterPropertiesCollectorFactory(int32_t version,
                                                   SequenceNumber global_seqno)
      : version_(version), global_seqno_(global_seqno) {}

  InternalTblPropColl* CreateInternalTblPropColl(
      uint32_t /*column_family_id*/, int /* level_at_creation */,
      int /* num_levels */,
      SequenceNumber /* last_level_inclusive_max_seqno_threshold */) override {
    return new SstFileWriterPropertiesCollector(version_, global_seqno_);
  }

  const char* Name() const override {
    return "SstFileWriterPropertiesCollector";
  }

 private:
  int32_t version_;
  SequenceNumber global_seqno_;
};

}  // namespace ROCKSDB_NAMESPACE
