// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief 
  class BESIII_2024_I2806455 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BESIII_2024_I2806455);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");
      declare(UnstableParticles(), "UFS");
      for(unsigned int ix=0;ix<2;++ix)
	book(_sigma[ix],ix+1, 1, 1);

      for (const string& en : _sigma[0].binning().edges<0>()) {
        const double end = std::stod(en)*GeV;
        if (isCompatibleWithSqrtS(end)) {
          _ecms = en;
          break;
        }
      }
      if(_ecms.empty()) MSG_ERROR("Beam energy incompatible with analysis.");
    }

    void findChildren(const Particle & p,map<long,int> & nRes, int &ncount) {
      for(const Particle &child : p.children()) {
	if(child.children().empty()) {
	  nRes[child.pid()]-=1;
	  --ncount;
	}
	else
	  findChildren(child,nRes,ncount);
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const FinalState& fs = apply<FinalState>(event, "FS");
      // total analyse final state
      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
      	nCount[p.pid()] += 1;
      	++ntotal;
      }
      // unstable charm analysis
      Particles ds = apply<UnstableParticles>(event, "UFS").particles(Cuts::abspid==431 or
                                                                      Cuts::abspid==10433 or
                                                                      Cuts::abspid==435);
      for(unsigned int ix=0;ix<ds.size();++ix) {
	const Particle& p1 = ds[ix];
       	int id1 = p1.abspid();
	// check fs
	bool fs = true;
	for (const Particle & child : p1.children()) {
	  if(child.pid()==p1.pid()) {
	    fs = false;
	    break;
	  }
	}
      	if(!fs) continue;
      	// find the children
      	map<long,int> nRes = nCount;
      	int ncount = ntotal;
      	findChildren(p1,nRes,ncount);
      	bool matched=false;
       	int sign = p1.pid()/id1;
       	for(unsigned int iy=ix+1;iy<ds.size();++iy) {
      	  const Particle& p2 = ds[iy];
      	  fs = true;
      	  for (const Particle & child : p2.children()) {
      	    if(child.pid()==p2.pid()) {
      	      fs = false;
      	      break;
      	    }
      	  }
      	  if(!fs) continue;
          // particle antiparticle
	  int id2 = p2.abspid();
      	  if(p2.pid()/id2==sign) continue;
	  if(!p2.parents().empty() && p2.parents()[0].pid()==p1.pid())
	    continue;
          // check not same
          if(p1.abspid()==p2.abspid()) continue;
          // at least 1 D_s
          if(p1.abspid()!=431 && p2.abspid()!=431) continue;
	  map<long,int> nRes2 = nRes;
	  int ncount2 = ncount;
	  findChildren(p2,nRes2,ncount2);
	  if(ncount2!=0) continue;
	  matched=true;
	  for(auto const & val : nRes2) {
	    if(val.second!=0) {
	      matched = false;
	      break;
	    }
	  }
	  if(matched) {
       	    if((id1==431 && id2==10433) || (id1==10433 && id2==431)) {
	      Particle Ds1 = id1==10433 ? p1 : p2;
	      matched = false;
	      if(Ds1.children().size()!=2) continue;
	      if(Ds1.pid()==10433 &&
		 ((Ds1.children()[0].pid()==423 && Ds1.children()[1].pid()==321) ||
		  (Ds1.children()[1].pid()==423 && Ds1.children()[0].pid()==321) ))
		matched = true;
	      else if (Ds1.pid()==-10433 &&
		 ((Ds1.children()[0].pid()==-423 && Ds1.children()[1].pid()==-321) ||
		  (Ds1.children()[1].pid()==-423 && Ds1.children()[0].pid()==-321) ))
		matched = true;
	      if(matched) {
		_sigma[0]->fill(_ecms);
		break;
	      }
            }
            else if ((id1==431 && id2==435) || (id1==435 && id2==431)) {
	      Particle Ds2 = id1==435 ? p1 : p2;
	      matched = false;
	      if(Ds2.children().size()!=2) continue;
	      if(Ds2.pid()==435 &&
		 ((Ds2.children()[0].pid()==421 && Ds2.children()[1].pid()==321) ||
		  (Ds2.children()[1].pid()==421 && Ds2.children()[0].pid()==321) ))
		matched = true;
	      else if (Ds2.pid()==-435 &&
		 ((Ds2.children()[0].pid()==-421 && Ds2.children()[1].pid()==-321) ||
		  (Ds2.children()[1].pid()==-421 && Ds2.children()[0].pid()==-321) ))
		matched = true;
	      if(matched) {
		_sigma[1]->fill(_ecms);
		break;
	      }
            }
          }
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_sigma,crossSection()/ sumOfWeights()/picobarn);
    }

    /// @}


    /// @name Histograms
    /// @{
    BinnedHistoPtr<string> _sigma[2];
    string _ecms;
    /// @}


  };


  RIVET_DECLARE_PLUGIN(BESIII_2024_I2806455);

}
