#ifndef CAPSTONE_LOONGARCH_H
#define CAPSTONE_LOONGARCH_H

/* Capstone Disassembly Engine */
/* By Jiajie Chen <c@jia.je> 2024 */
/*    Yanglin Xun <1109673069@qq.com>, 2024 */

#ifdef __cplusplus
extern "C" {
#endif

#if !defined(_MSC_VER) || !defined(_KERNEL_MODE)
#include <stdint.h>
#endif

#include "platform.h"
#include "cs_operand.h"

/// Operand type for instruction's operands
typedef enum loongarch_op_type {
	LOONGARCH_OP_INVALID = CS_OP_INVALID, ///< Invalid
	LOONGARCH_OP_REG = CS_OP_REG,	      ///< Register operand
	LOONGARCH_OP_IMM = CS_OP_IMM,	      ///< Immediate operand
	LOONGARCH_OP_MEM = CS_OP_MEM,	      ///< Memory operand
} loongarch_op_type;

/// Instruction's operand referring to memory
/// This is associated with LOONGARCH_OP_MEM operand type above
typedef struct loongarch_op_mem {
	unsigned int base;  ///< base register
	unsigned int index; ///< index register
	int64_t disp;	    ///< displacement/offset value
} loongarch_op_mem;

/// Instruction operand
typedef struct cs_loongarch_op {
	loongarch_op_type type;	      //< operand type
	union {
		unsigned int reg;     /// register value for REG operand
		int64_t imm;	      /// immediate value for IMM operand
		loongarch_op_mem mem; /// base/disp value for MEM operand
	};

	/// How is this operand accessed? (READ, WRITE or READ|WRITE)
	/// NOTE: this field is irrelevant if engine is compiled in DIET mode.
	cs_ac_type access;
} cs_loongarch_op;

/// LoongArch instruction formats. To get details about them please
/// refer to `LoongArchInstrFormats.td` in LLVM.
typedef enum {
	// generated content <LoongArchGenCSInsnFormatsEnum.inc> begin
	// clang-format off

	LOONGARCH_INSN_FORM_PSEUDO,
	LOONGARCH_INSN_FORM_FMT3R,
	LOONGARCH_INSN_FORM_FMT2RI12,
	LOONGARCH_INSN_FORM_FMT2RI5,
	LOONGARCH_INSN_FORM_FMT2RI16,
	LOONGARCH_INSN_FORM_FMT3RI2,
	LOONGARCH_INSN_FORM_NODSTFMT2RI4,
	LOONGARCH_INSN_FORM_FMT1RI8,
	LOONGARCH_INSN_FORM_FMT2RI4,
	LOONGARCH_INSN_FORM_NODSTFMT1RI4,
	LOONGARCH_INSN_FORM_NODSTFMT1RI5I4,
	LOONGARCH_INSN_FORM_FMTASRT,
	LOONGARCH_INSN_FORM_FMTI26,
	LOONGARCH_INSN_FORM_FPFMTBR,
	LOONGARCH_INSN_FORM_FMT1RI21,
	LOONGARCH_INSN_FORM_FMT2R,
	LOONGARCH_INSN_FORM_FMTI15,
	LOONGARCH_INSN_FORM_FMTBSTR_D,
	LOONGARCH_INSN_FORM_FMTBSTR_W,
	LOONGARCH_INSN_FORM_FMT3RI3,
	LOONGARCH_INSN_FORM_FMTCACOP,
	LOONGARCH_INSN_FORM_FMTCSR,
	LOONGARCH_INSN_FORM_FMTCSRXCHG,
	LOONGARCH_INSN_FORM_FMTI32,
	LOONGARCH_INSN_FORM_FPFMT2R,
	LOONGARCH_INSN_FORM_FPFMT3R,
	LOONGARCH_INSN_FORM_FPFMTFCMP,
	LOONGARCH_INSN_FORM_FPFMTMEM,
	LOONGARCH_INSN_FORM_FPFMT2RI12,
	LOONGARCH_INSN_FORM_FPFMT4R,
	LOONGARCH_INSN_FORM_FPFMTMOV,
	LOONGARCH_INSN_FORM_FPFMTFSEL,
	LOONGARCH_INSN_FORM_FMTINVTLB,
	LOONGARCH_INSN_FORM_FMTJISCR,
	LOONGARCH_INSN_FORM_FMT2RI8,
	LOONGARCH_INSN_FORM_FMTLDPTE,
	LOONGARCH_INSN_FORM_FMT2RI14,
	LOONGARCH_INSN_FORM_FMT1RI20,
	LOONGARCH_INSN_FORM_FMTGR2SCR,
	LOONGARCH_INSN_FORM_FMTSCR2GR,
	LOONGARCH_INSN_FORM_FMTPRELD,
	LOONGARCH_INSN_FORM_FMTPRELDX,
	LOONGARCH_INSN_FORM_FMT2RI3,
	LOONGARCH_INSN_FORM_FMT2RI6,
	LOONGARCH_INSN_FORM_FMT1RI4,
	LOONGARCH_INSN_FORM_FMT3R_VVV,
	LOONGARCH_INSN_FORM_FMT2RI5_VVI,
	LOONGARCH_INSN_FORM_FMT2RI8_VVI,
	LOONGARCH_INSN_FORM_FMT2RI3_VVI,
	LOONGARCH_INSN_FORM_FMT2RI6_VVI,
	LOONGARCH_INSN_FORM_FMT2RI4_VVI,
	LOONGARCH_INSN_FORM_FMT4R_VVVV,
	LOONGARCH_INSN_FORM_FMT2R_VV,
	LOONGARCH_INSN_FORM_FMT2R_XX,
	LOONGARCH_INSN_FORM_FMT2RI4_VRI,
	LOONGARCH_INSN_FORM_FMT2RI1_VRI,
	LOONGARCH_INSN_FORM_FMT2RI3_VRI,
	LOONGARCH_INSN_FORM_FMT2RI2_VRI,
	LOONGARCH_INSN_FORM_FMT2RI12_VRI,
	LOONGARCH_INSN_FORM_FMT1RI13_VI,
	LOONGARCH_INSN_FORM_FMT2RI9_VRI,
	LOONGARCH_INSN_FORM_FMT2RI11_VRI,
	LOONGARCH_INSN_FORM_FMT2RI10_VRI,
	LOONGARCH_INSN_FORM_FMT3R_VRR,
	LOONGARCH_INSN_FORM_FMT2RI4_RVI,
	LOONGARCH_INSN_FORM_FMT2RI1_RVI,
	LOONGARCH_INSN_FORM_FMT2RI3_RVI,
	LOONGARCH_INSN_FORM_FMT2RI2_RVI,
	LOONGARCH_INSN_FORM_FMT2R_VR,
	LOONGARCH_INSN_FORM_FMT2RI1_VVI,
	LOONGARCH_INSN_FORM_FMT2RI2_VVI,
	LOONGARCH_INSN_FORM_FMT3R_VVR,
	LOONGARCH_INSN_FORM_FMT2R_CV,
	LOONGARCH_INSN_FORM_FMT2RI7_VVI,
	LOONGARCH_INSN_FORM_FMT2RI8I4_VRII,
	LOONGARCH_INSN_FORM_FMT2RI8I1_VRII,
	LOONGARCH_INSN_FORM_FMT2RI8I3_VRII,
	LOONGARCH_INSN_FORM_FMT2RI8I2_VRII,
	LOONGARCH_INSN_FORM_NODSTFMT2R,
	LOONGARCH_INSN_FORM_NODSTFMT1R,
	LOONGARCH_INSN_FORM_FMTMFTOP,
	LOONGARCH_INSN_FORM_FMTMTTOP,
	LOONGARCH_INSN_FORM_NODSTFMT1RI3,
	LOONGARCH_INSN_FORM_NODSTFMT1RI6,
	LOONGARCH_INSN_FORM_NODSTFMT1RI5,
	LOONGARCH_INSN_FORM_FMT1RI5I8,
	LOONGARCH_INSN_FORM_FMT3R_XXX,
	LOONGARCH_INSN_FORM_FMT2RI5_XXI,
	LOONGARCH_INSN_FORM_FMT2RI8_XXI,
	LOONGARCH_INSN_FORM_FMT2RI3_XXI,
	LOONGARCH_INSN_FORM_FMT2RI6_XXI,
	LOONGARCH_INSN_FORM_FMT2RI4_XXI,
	LOONGARCH_INSN_FORM_FMT4R_XXXX,
	LOONGARCH_INSN_FORM_FMT2RI2_XRI,
	LOONGARCH_INSN_FORM_FMT2RI3_XRI,
	LOONGARCH_INSN_FORM_FMT2RI2_XXI,
	LOONGARCH_INSN_FORM_FMT2RI12_XRI,
	LOONGARCH_INSN_FORM_FMT1RI13_XI,
	LOONGARCH_INSN_FORM_FMT2RI9_XRI,
	LOONGARCH_INSN_FORM_FMT2RI11_XRI,
	LOONGARCH_INSN_FORM_FMT2RI10_XRI,
	LOONGARCH_INSN_FORM_FMT3R_XRR,
	LOONGARCH_INSN_FORM_FMT2RI2_RXI,
	LOONGARCH_INSN_FORM_FMT2RI3_RXI,
	LOONGARCH_INSN_FORM_FMT2RI1_XXI,
	LOONGARCH_INSN_FORM_FMT2R_XR,
	LOONGARCH_INSN_FORM_FMT3R_XXR,
	LOONGARCH_INSN_FORM_FMT2R_CX,
	LOONGARCH_INSN_FORM_FMT2RI7_XXI,
	LOONGARCH_INSN_FORM_FMT2RI8I5_XRII,
	LOONGARCH_INSN_FORM_FMT2RI8I2_XRII,
	LOONGARCH_INSN_FORM_FMT2RI8I4_XRII,
	LOONGARCH_INSN_FORM_FMT2RI8I3_XRII,

	// clang-format on
	// generated content <LoongArchGenCSInsnFormatsEnum.inc> end
} loongarch_insn_form;

/// Supplemental info for LoongArch instructions
typedef struct {
	loongarch_insn_form form; ///< Instruction form
	enum cs_ac_type
		memory_access; ///< Memory access (none/read/write/read+write)
} loongarch_suppl_info;

#define NUM_LOONGARCH_OPS 8

/// Instruction structure
typedef struct cs_loongarch {
	/// The instruction format. Can be use to determine the bit encoding of the instruction.
	loongarch_insn_form format;

	/// Number of operands of this instruction,
	/// or 0 when instruction has no operand.
	uint8_t op_count;
	cs_loongarch_op operands[NUM_LOONGARCH_OPS]; ///< operands for this instruction.
} cs_loongarch;

/// LoongArch registers
typedef enum loongarch_reg {
	// generated content <LoongArchGenCSRegEnum.inc> begin
	// clang-format off

	LOONGARCH_REG_INVALID = 0,
	LOONGARCH_REG_F0 = 1,
	LOONGARCH_REG_F1 = 2,
	LOONGARCH_REG_F2 = 3,
	LOONGARCH_REG_F3 = 4,
	LOONGARCH_REG_F4 = 5,
	LOONGARCH_REG_F5 = 6,
	LOONGARCH_REG_F6 = 7,
	LOONGARCH_REG_F7 = 8,
	LOONGARCH_REG_F8 = 9,
	LOONGARCH_REG_F9 = 10,
	LOONGARCH_REG_F10 = 11,
	LOONGARCH_REG_F11 = 12,
	LOONGARCH_REG_F12 = 13,
	LOONGARCH_REG_F13 = 14,
	LOONGARCH_REG_F14 = 15,
	LOONGARCH_REG_F15 = 16,
	LOONGARCH_REG_F16 = 17,
	LOONGARCH_REG_F17 = 18,
	LOONGARCH_REG_F18 = 19,
	LOONGARCH_REG_F19 = 20,
	LOONGARCH_REG_F20 = 21,
	LOONGARCH_REG_F21 = 22,
	LOONGARCH_REG_F22 = 23,
	LOONGARCH_REG_F23 = 24,
	LOONGARCH_REG_F24 = 25,
	LOONGARCH_REG_F25 = 26,
	LOONGARCH_REG_F26 = 27,
	LOONGARCH_REG_F27 = 28,
	LOONGARCH_REG_F28 = 29,
	LOONGARCH_REG_F29 = 30,
	LOONGARCH_REG_F30 = 31,
	LOONGARCH_REG_F31 = 32,
	LOONGARCH_REG_FCC0 = 33,
	LOONGARCH_REG_FCC1 = 34,
	LOONGARCH_REG_FCC2 = 35,
	LOONGARCH_REG_FCC3 = 36,
	LOONGARCH_REG_FCC4 = 37,
	LOONGARCH_REG_FCC5 = 38,
	LOONGARCH_REG_FCC6 = 39,
	LOONGARCH_REG_FCC7 = 40,
	LOONGARCH_REG_FCSR0 = 41,
	LOONGARCH_REG_FCSR1 = 42,
	LOONGARCH_REG_FCSR2 = 43,
	LOONGARCH_REG_FCSR3 = 44,
	LOONGARCH_REG_R0 = 45,
	LOONGARCH_REG_R1 = 46,
	LOONGARCH_REG_R2 = 47,
	LOONGARCH_REG_R3 = 48,
	LOONGARCH_REG_R4 = 49,
	LOONGARCH_REG_R5 = 50,
	LOONGARCH_REG_R6 = 51,
	LOONGARCH_REG_R7 = 52,
	LOONGARCH_REG_R8 = 53,
	LOONGARCH_REG_R9 = 54,
	LOONGARCH_REG_R10 = 55,
	LOONGARCH_REG_R11 = 56,
	LOONGARCH_REG_R12 = 57,
	LOONGARCH_REG_R13 = 58,
	LOONGARCH_REG_R14 = 59,
	LOONGARCH_REG_R15 = 60,
	LOONGARCH_REG_R16 = 61,
	LOONGARCH_REG_R17 = 62,
	LOONGARCH_REG_R18 = 63,
	LOONGARCH_REG_R19 = 64,
	LOONGARCH_REG_R20 = 65,
	LOONGARCH_REG_R21 = 66,
	LOONGARCH_REG_R22 = 67,
	LOONGARCH_REG_R23 = 68,
	LOONGARCH_REG_R24 = 69,
	LOONGARCH_REG_R25 = 70,
	LOONGARCH_REG_R26 = 71,
	LOONGARCH_REG_R27 = 72,
	LOONGARCH_REG_R28 = 73,
	LOONGARCH_REG_R29 = 74,
	LOONGARCH_REG_R30 = 75,
	LOONGARCH_REG_R31 = 76,
	LOONGARCH_REG_SCR0 = 77,
	LOONGARCH_REG_SCR1 = 78,
	LOONGARCH_REG_SCR2 = 79,
	LOONGARCH_REG_SCR3 = 80,
	LOONGARCH_REG_VR0 = 81,
	LOONGARCH_REG_VR1 = 82,
	LOONGARCH_REG_VR2 = 83,
	LOONGARCH_REG_VR3 = 84,
	LOONGARCH_REG_VR4 = 85,
	LOONGARCH_REG_VR5 = 86,
	LOONGARCH_REG_VR6 = 87,
	LOONGARCH_REG_VR7 = 88,
	LOONGARCH_REG_VR8 = 89,
	LOONGARCH_REG_VR9 = 90,
	LOONGARCH_REG_VR10 = 91,
	LOONGARCH_REG_VR11 = 92,
	LOONGARCH_REG_VR12 = 93,
	LOONGARCH_REG_VR13 = 94,
	LOONGARCH_REG_VR14 = 95,
	LOONGARCH_REG_VR15 = 96,
	LOONGARCH_REG_VR16 = 97,
	LOONGARCH_REG_VR17 = 98,
	LOONGARCH_REG_VR18 = 99,
	LOONGARCH_REG_VR19 = 100,
	LOONGARCH_REG_VR20 = 101,
	LOONGARCH_REG_VR21 = 102,
	LOONGARCH_REG_VR22 = 103,
	LOONGARCH_REG_VR23 = 104,
	LOONGARCH_REG_VR24 = 105,
	LOONGARCH_REG_VR25 = 106,
	LOONGARCH_REG_VR26 = 107,
	LOONGARCH_REG_VR27 = 108,
	LOONGARCH_REG_VR28 = 109,
	LOONGARCH_REG_VR29 = 110,
	LOONGARCH_REG_VR30 = 111,
	LOONGARCH_REG_VR31 = 112,
	LOONGARCH_REG_XR0 = 113,
	LOONGARCH_REG_XR1 = 114,
	LOONGARCH_REG_XR2 = 115,
	LOONGARCH_REG_XR3 = 116,
	LOONGARCH_REG_XR4 = 117,
	LOONGARCH_REG_XR5 = 118,
	LOONGARCH_REG_XR6 = 119,
	LOONGARCH_REG_XR7 = 120,
	LOONGARCH_REG_XR8 = 121,
	LOONGARCH_REG_XR9 = 122,
	LOONGARCH_REG_XR10 = 123,
	LOONGARCH_REG_XR11 = 124,
	LOONGARCH_REG_XR12 = 125,
	LOONGARCH_REG_XR13 = 126,
	LOONGARCH_REG_XR14 = 127,
	LOONGARCH_REG_XR15 = 128,
	LOONGARCH_REG_XR16 = 129,
	LOONGARCH_REG_XR17 = 130,
	LOONGARCH_REG_XR18 = 131,
	LOONGARCH_REG_XR19 = 132,
	LOONGARCH_REG_XR20 = 133,
	LOONGARCH_REG_XR21 = 134,
	LOONGARCH_REG_XR22 = 135,
	LOONGARCH_REG_XR23 = 136,
	LOONGARCH_REG_XR24 = 137,
	LOONGARCH_REG_XR25 = 138,
	LOONGARCH_REG_XR26 = 139,
	LOONGARCH_REG_XR27 = 140,
	LOONGARCH_REG_XR28 = 141,
	LOONGARCH_REG_XR29 = 142,
	LOONGARCH_REG_XR30 = 143,
	LOONGARCH_REG_XR31 = 144,
	LOONGARCH_REG_F0_64 = 145,
	LOONGARCH_REG_F1_64 = 146,
	LOONGARCH_REG_F2_64 = 147,
	LOONGARCH_REG_F3_64 = 148,
	LOONGARCH_REG_F4_64 = 149,
	LOONGARCH_REG_F5_64 = 150,
	LOONGARCH_REG_F6_64 = 151,
	LOONGARCH_REG_F7_64 = 152,
	LOONGARCH_REG_F8_64 = 153,
	LOONGARCH_REG_F9_64 = 154,
	LOONGARCH_REG_F10_64 = 155,
	LOONGARCH_REG_F11_64 = 156,
	LOONGARCH_REG_F12_64 = 157,
	LOONGARCH_REG_F13_64 = 158,
	LOONGARCH_REG_F14_64 = 159,
	LOONGARCH_REG_F15_64 = 160,
	LOONGARCH_REG_F16_64 = 161,
	LOONGARCH_REG_F17_64 = 162,
	LOONGARCH_REG_F18_64 = 163,
	LOONGARCH_REG_F19_64 = 164,
	LOONGARCH_REG_F20_64 = 165,
	LOONGARCH_REG_F21_64 = 166,
	LOONGARCH_REG_F22_64 = 167,
	LOONGARCH_REG_F23_64 = 168,
	LOONGARCH_REG_F24_64 = 169,
	LOONGARCH_REG_F25_64 = 170,
	LOONGARCH_REG_F26_64 = 171,
	LOONGARCH_REG_F27_64 = 172,
	LOONGARCH_REG_F28_64 = 173,
	LOONGARCH_REG_F29_64 = 174,
	LOONGARCH_REG_F30_64 = 175,
	LOONGARCH_REG_F31_64 = 176,
	LOONGARCH_REG_ENDING, // 177

	// clang-format on
	// generated content <LoongArchGenCSRegEnum.inc> end

	// alias registers
	LOONGARCH_REG_ZERO = LOONGARCH_REG_R0,
	LOONGARCH_REG_RA = LOONGARCH_REG_R1,
	LOONGARCH_REG_TP = LOONGARCH_REG_R2,
	LOONGARCH_REG_SP = LOONGARCH_REG_R3,
	LOONGARCH_REG_A0 = LOONGARCH_REG_R4,
	LOONGARCH_REG_A1 = LOONGARCH_REG_R5,
	LOONGARCH_REG_A2 = LOONGARCH_REG_R6,
	LOONGARCH_REG_A3 = LOONGARCH_REG_R7,
	LOONGARCH_REG_A4 = LOONGARCH_REG_R8,
	LOONGARCH_REG_A5 = LOONGARCH_REG_R9,
	LOONGARCH_REG_A6 = LOONGARCH_REG_R10,
	LOONGARCH_REG_A7 = LOONGARCH_REG_R11,
	LOONGARCH_REG_T0 = LOONGARCH_REG_R12,
	LOONGARCH_REG_T1 = LOONGARCH_REG_R13,
	LOONGARCH_REG_T2 = LOONGARCH_REG_R14,
	LOONGARCH_REG_T3 = LOONGARCH_REG_R15,
	LOONGARCH_REG_T4 = LOONGARCH_REG_R16,
	LOONGARCH_REG_T5 = LOONGARCH_REG_R17,
	LOONGARCH_REG_T6 = LOONGARCH_REG_R18,
	LOONGARCH_REG_T7 = LOONGARCH_REG_R19,
	LOONGARCH_REG_T8 = LOONGARCH_REG_R20,
	LOONGARCH_REG_FP = LOONGARCH_REG_R22,
	LOONGARCH_REG_S9 = LOONGARCH_REG_R22,
	LOONGARCH_REG_S0 = LOONGARCH_REG_R23,
	LOONGARCH_REG_S1 = LOONGARCH_REG_R24,
	LOONGARCH_REG_S2 = LOONGARCH_REG_R25,
	LOONGARCH_REG_S3 = LOONGARCH_REG_R26,
	LOONGARCH_REG_S4 = LOONGARCH_REG_R27,
	LOONGARCH_REG_S5 = LOONGARCH_REG_R28,
	LOONGARCH_REG_S6 = LOONGARCH_REG_R29,
	LOONGARCH_REG_S7 = LOONGARCH_REG_R30,
	LOONGARCH_REG_S8 = LOONGARCH_REG_R31,
} loongarch_reg;

/// LoongArch instruction
typedef enum loongarch_insn {
	// generated content <LoongArchGenCSInsnEnum.inc> begin
	// clang-format off

	LOONGARCH_INS_INVALID,
	LOONGARCH_INS_CALL36,
	LOONGARCH_INS_LA_ABS,
	LOONGARCH_INS_LA_GOT,
	LOONGARCH_INS_LA_PCREL,
	LOONGARCH_INS_LA_TLS_GD,
	LOONGARCH_INS_LA_TLS_IE,
	LOONGARCH_INS_LA_TLS_LD,
	LOONGARCH_INS_LA_TLS_LE,
	LOONGARCH_INS_LI_D,
	LOONGARCH_INS_LI_W,
	LOONGARCH_INS_TAIL36,
	LOONGARCH_INS_VREPLI_B,
	LOONGARCH_INS_VREPLI_D,
	LOONGARCH_INS_VREPLI_H,
	LOONGARCH_INS_VREPLI_W,
	LOONGARCH_INS_XVREPLI_B,
	LOONGARCH_INS_XVREPLI_D,
	LOONGARCH_INS_XVREPLI_H,
	LOONGARCH_INS_XVREPLI_W,
	LOONGARCH_INS_ADC_B,
	LOONGARCH_INS_ADC_D,
	LOONGARCH_INS_ADC_H,
	LOONGARCH_INS_ADC_W,
	LOONGARCH_INS_ADDI_D,
	LOONGARCH_INS_ADDI_W,
	LOONGARCH_INS_ADDU12I_D,
	LOONGARCH_INS_ADDU12I_W,
	LOONGARCH_INS_ADDU16I_D,
	LOONGARCH_INS_ADD_D,
	LOONGARCH_INS_ADD_W,
	LOONGARCH_INS_ALSL_D,
	LOONGARCH_INS_ALSL_W,
	LOONGARCH_INS_ALSL_WU,
	LOONGARCH_INS_AMADD_B,
	LOONGARCH_INS_AMADD_D,
	LOONGARCH_INS_AMADD_H,
	LOONGARCH_INS_AMADD_W,
	LOONGARCH_INS_AMADD_DB_B,
	LOONGARCH_INS_AMADD_DB_D,
	LOONGARCH_INS_AMADD_DB_H,
	LOONGARCH_INS_AMADD_DB_W,
	LOONGARCH_INS_AMAND_D,
	LOONGARCH_INS_AMAND_W,
	LOONGARCH_INS_AMAND_DB_D,
	LOONGARCH_INS_AMAND_DB_W,
	LOONGARCH_INS_AMCAS_B,
	LOONGARCH_INS_AMCAS_D,
	LOONGARCH_INS_AMCAS_H,
	LOONGARCH_INS_AMCAS_W,
	LOONGARCH_INS_AMCAS_DB_B,
	LOONGARCH_INS_AMCAS_DB_D,
	LOONGARCH_INS_AMCAS_DB_H,
	LOONGARCH_INS_AMCAS_DB_W,
	LOONGARCH_INS_AMMAX_D,
	LOONGARCH_INS_AMMAX_DU,
	LOONGARCH_INS_AMMAX_W,
	LOONGARCH_INS_AMMAX_WU,
	LOONGARCH_INS_AMMAX_DB_D,
	LOONGARCH_INS_AMMAX_DB_DU,
	LOONGARCH_INS_AMMAX_DB_W,
	LOONGARCH_INS_AMMAX_DB_WU,
	LOONGARCH_INS_AMMIN_D,
	LOONGARCH_INS_AMMIN_DU,
	LOONGARCH_INS_AMMIN_W,
	LOONGARCH_INS_AMMIN_WU,
	LOONGARCH_INS_AMMIN_DB_D,
	LOONGARCH_INS_AMMIN_DB_DU,
	LOONGARCH_INS_AMMIN_DB_W,
	LOONGARCH_INS_AMMIN_DB_WU,
	LOONGARCH_INS_AMOR_D,
	LOONGARCH_INS_AMOR_W,
	LOONGARCH_INS_AMOR_DB_D,
	LOONGARCH_INS_AMOR_DB_W,
	LOONGARCH_INS_AMSWAP_B,
	LOONGARCH_INS_AMSWAP_D,
	LOONGARCH_INS_AMSWAP_H,
	LOONGARCH_INS_AMSWAP_W,
	LOONGARCH_INS_AMSWAP_DB_B,
	LOONGARCH_INS_AMSWAP_DB_D,
	LOONGARCH_INS_AMSWAP_DB_H,
	LOONGARCH_INS_AMSWAP_DB_W,
	LOONGARCH_INS_AMXOR_D,
	LOONGARCH_INS_AMXOR_W,
	LOONGARCH_INS_AMXOR_DB_D,
	LOONGARCH_INS_AMXOR_DB_W,
	LOONGARCH_INS_AND,
	LOONGARCH_INS_ANDI,
	LOONGARCH_INS_ANDN,
	LOONGARCH_INS_ARMADC_W,
	LOONGARCH_INS_ARMADD_W,
	LOONGARCH_INS_ARMAND_W,
	LOONGARCH_INS_ARMMFFLAG,
	LOONGARCH_INS_ARMMOVE,
	LOONGARCH_INS_ARMMOV_D,
	LOONGARCH_INS_ARMMOV_W,
	LOONGARCH_INS_ARMMTFLAG,
	LOONGARCH_INS_ARMNOT_W,
	LOONGARCH_INS_ARMOR_W,
	LOONGARCH_INS_ARMROTRI_W,
	LOONGARCH_INS_ARMROTR_W,
	LOONGARCH_INS_ARMRRX_W,
	LOONGARCH_INS_ARMSBC_W,
	LOONGARCH_INS_ARMSLLI_W,
	LOONGARCH_INS_ARMSLL_W,
	LOONGARCH_INS_ARMSRAI_W,
	LOONGARCH_INS_ARMSRA_W,
	LOONGARCH_INS_ARMSRLI_W,
	LOONGARCH_INS_ARMSRL_W,
	LOONGARCH_INS_ARMSUB_W,
	LOONGARCH_INS_ARMXOR_W,
	LOONGARCH_INS_ASRTGT_D,
	LOONGARCH_INS_ASRTLE_D,
	LOONGARCH_INS_B,
	LOONGARCH_INS_BCEQZ,
	LOONGARCH_INS_BCNEZ,
	LOONGARCH_INS_BEQ,
	LOONGARCH_INS_BEQZ,
	LOONGARCH_INS_BGE,
	LOONGARCH_INS_BGEU,
	LOONGARCH_INS_BITREV_4B,
	LOONGARCH_INS_BITREV_8B,
	LOONGARCH_INS_BITREV_D,
	LOONGARCH_INS_BITREV_W,
	LOONGARCH_INS_BL,
	LOONGARCH_INS_BLT,
	LOONGARCH_INS_BLTU,
	LOONGARCH_INS_BNE,
	LOONGARCH_INS_BNEZ,
	LOONGARCH_INS_BREAK,
	LOONGARCH_INS_BSTRINS_D,
	LOONGARCH_INS_BSTRINS_W,
	LOONGARCH_INS_BSTRPICK_D,
	LOONGARCH_INS_BSTRPICK_W,
	LOONGARCH_INS_BYTEPICK_D,
	LOONGARCH_INS_BYTEPICK_W,
	LOONGARCH_INS_CACOP,
	LOONGARCH_INS_CLO_D,
	LOONGARCH_INS_CLO_W,
	LOONGARCH_INS_CLZ_D,
	LOONGARCH_INS_CLZ_W,
	LOONGARCH_INS_CPUCFG,
	LOONGARCH_INS_CRCC_W_B_W,
	LOONGARCH_INS_CRCC_W_D_W,
	LOONGARCH_INS_CRCC_W_H_W,
	LOONGARCH_INS_CRCC_W_W_W,
	LOONGARCH_INS_CRC_W_B_W,
	LOONGARCH_INS_CRC_W_D_W,
	LOONGARCH_INS_CRC_W_H_W,
	LOONGARCH_INS_CRC_W_W_W,
	LOONGARCH_INS_CSRRD,
	LOONGARCH_INS_CSRWR,
	LOONGARCH_INS_CSRXCHG,
	LOONGARCH_INS_CTO_D,
	LOONGARCH_INS_CTO_W,
	LOONGARCH_INS_CTZ_D,
	LOONGARCH_INS_CTZ_W,
	LOONGARCH_INS_DBAR,
	LOONGARCH_INS_DBCL,
	LOONGARCH_INS_DIV_D,
	LOONGARCH_INS_DIV_DU,
	LOONGARCH_INS_DIV_W,
	LOONGARCH_INS_DIV_WU,
	LOONGARCH_INS_ERTN,
	LOONGARCH_INS_EXT_W_B,
	LOONGARCH_INS_EXT_W_H,
	LOONGARCH_INS_FABS_D,
	LOONGARCH_INS_FABS_S,
	LOONGARCH_INS_FADD_D,
	LOONGARCH_INS_FADD_S,
	LOONGARCH_INS_FCLASS_D,
	LOONGARCH_INS_FCLASS_S,
	LOONGARCH_INS_FCMP_CAF_D,
	LOONGARCH_INS_FCMP_CAF_S,
	LOONGARCH_INS_FCMP_CEQ_D,
	LOONGARCH_INS_FCMP_CEQ_S,
	LOONGARCH_INS_FCMP_CLE_D,
	LOONGARCH_INS_FCMP_CLE_S,
	LOONGARCH_INS_FCMP_CLT_D,
	LOONGARCH_INS_FCMP_CLT_S,
	LOONGARCH_INS_FCMP_CNE_D,
	LOONGARCH_INS_FCMP_CNE_S,
	LOONGARCH_INS_FCMP_COR_D,
	LOONGARCH_INS_FCMP_COR_S,
	LOONGARCH_INS_FCMP_CUEQ_D,
	LOONGARCH_INS_FCMP_CUEQ_S,
	LOONGARCH_INS_FCMP_CULE_D,
	LOONGARCH_INS_FCMP_CULE_S,
	LOONGARCH_INS_FCMP_CULT_D,
	LOONGARCH_INS_FCMP_CULT_S,
	LOONGARCH_INS_FCMP_CUNE_D,
	LOONGARCH_INS_FCMP_CUNE_S,
	LOONGARCH_INS_FCMP_CUN_D,
	LOONGARCH_INS_FCMP_CUN_S,
	LOONGARCH_INS_FCMP_SAF_D,
	LOONGARCH_INS_FCMP_SAF_S,
	LOONGARCH_INS_FCMP_SEQ_D,
	LOONGARCH_INS_FCMP_SEQ_S,
	LOONGARCH_INS_FCMP_SLE_D,
	LOONGARCH_INS_FCMP_SLE_S,
	LOONGARCH_INS_FCMP_SLT_D,
	LOONGARCH_INS_FCMP_SLT_S,
	LOONGARCH_INS_FCMP_SNE_D,
	LOONGARCH_INS_FCMP_SNE_S,
	LOONGARCH_INS_FCMP_SOR_D,
	LOONGARCH_INS_FCMP_SOR_S,
	LOONGARCH_INS_FCMP_SUEQ_D,
	LOONGARCH_INS_FCMP_SUEQ_S,
	LOONGARCH_INS_FCMP_SULE_D,
	LOONGARCH_INS_FCMP_SULE_S,
	LOONGARCH_INS_FCMP_SULT_D,
	LOONGARCH_INS_FCMP_SULT_S,
	LOONGARCH_INS_FCMP_SUNE_D,
	LOONGARCH_INS_FCMP_SUNE_S,
	LOONGARCH_INS_FCMP_SUN_D,
	LOONGARCH_INS_FCMP_SUN_S,
	LOONGARCH_INS_FCOPYSIGN_D,
	LOONGARCH_INS_FCOPYSIGN_S,
	LOONGARCH_INS_FCVT_D_LD,
	LOONGARCH_INS_FCVT_D_S,
	LOONGARCH_INS_FCVT_LD_D,
	LOONGARCH_INS_FCVT_S_D,
	LOONGARCH_INS_FCVT_UD_D,
	LOONGARCH_INS_FDIV_D,
	LOONGARCH_INS_FDIV_S,
	LOONGARCH_INS_FFINT_D_L,
	LOONGARCH_INS_FFINT_D_W,
	LOONGARCH_INS_FFINT_S_L,
	LOONGARCH_INS_FFINT_S_W,
	LOONGARCH_INS_FLDGT_D,
	LOONGARCH_INS_FLDGT_S,
	LOONGARCH_INS_FLDLE_D,
	LOONGARCH_INS_FLDLE_S,
	LOONGARCH_INS_FLDX_D,
	LOONGARCH_INS_FLDX_S,
	LOONGARCH_INS_FLD_D,
	LOONGARCH_INS_FLD_S,
	LOONGARCH_INS_FLOGB_D,
	LOONGARCH_INS_FLOGB_S,
	LOONGARCH_INS_FMADD_D,
	LOONGARCH_INS_FMADD_S,
	LOONGARCH_INS_FMAXA_D,
	LOONGARCH_INS_FMAXA_S,
	LOONGARCH_INS_FMAX_D,
	LOONGARCH_INS_FMAX_S,
	LOONGARCH_INS_FMINA_D,
	LOONGARCH_INS_FMINA_S,
	LOONGARCH_INS_FMIN_D,
	LOONGARCH_INS_FMIN_S,
	LOONGARCH_INS_FMOV_D,
	LOONGARCH_INS_FMOV_S,
	LOONGARCH_INS_FMSUB_D,
	LOONGARCH_INS_FMSUB_S,
	LOONGARCH_INS_FMUL_D,
	LOONGARCH_INS_FMUL_S,
	LOONGARCH_INS_FNEG_D,
	LOONGARCH_INS_FNEG_S,
	LOONGARCH_INS_FNMADD_D,
	LOONGARCH_INS_FNMADD_S,
	LOONGARCH_INS_FNMSUB_D,
	LOONGARCH_INS_FNMSUB_S,
	LOONGARCH_INS_FRECIPE_D,
	LOONGARCH_INS_FRECIPE_S,
	LOONGARCH_INS_FRECIP_D,
	LOONGARCH_INS_FRECIP_S,
	LOONGARCH_INS_FRINT_D,
	LOONGARCH_INS_FRINT_S,
	LOONGARCH_INS_FRSQRTE_D,
	LOONGARCH_INS_FRSQRTE_S,
	LOONGARCH_INS_FRSQRT_D,
	LOONGARCH_INS_FRSQRT_S,
	LOONGARCH_INS_FSCALEB_D,
	LOONGARCH_INS_FSCALEB_S,
	LOONGARCH_INS_FSEL,
	LOONGARCH_INS_FSQRT_D,
	LOONGARCH_INS_FSQRT_S,
	LOONGARCH_INS_FSTGT_D,
	LOONGARCH_INS_FSTGT_S,
	LOONGARCH_INS_FSTLE_D,
	LOONGARCH_INS_FSTLE_S,
	LOONGARCH_INS_FSTX_D,
	LOONGARCH_INS_FSTX_S,
	LOONGARCH_INS_FST_D,
	LOONGARCH_INS_FST_S,
	LOONGARCH_INS_FSUB_D,
	LOONGARCH_INS_FSUB_S,
	LOONGARCH_INS_FTINTRM_L_D,
	LOONGARCH_INS_FTINTRM_L_S,
	LOONGARCH_INS_FTINTRM_W_D,
	LOONGARCH_INS_FTINTRM_W_S,
	LOONGARCH_INS_FTINTRNE_L_D,
	LOONGARCH_INS_FTINTRNE_L_S,
	LOONGARCH_INS_FTINTRNE_W_D,
	LOONGARCH_INS_FTINTRNE_W_S,
	LOONGARCH_INS_FTINTRP_L_D,
	LOONGARCH_INS_FTINTRP_L_S,
	LOONGARCH_INS_FTINTRP_W_D,
	LOONGARCH_INS_FTINTRP_W_S,
	LOONGARCH_INS_FTINTRZ_L_D,
	LOONGARCH_INS_FTINTRZ_L_S,
	LOONGARCH_INS_FTINTRZ_W_D,
	LOONGARCH_INS_FTINTRZ_W_S,
	LOONGARCH_INS_FTINT_L_D,
	LOONGARCH_INS_FTINT_L_S,
	LOONGARCH_INS_FTINT_W_D,
	LOONGARCH_INS_FTINT_W_S,
	LOONGARCH_INS_GCSRRD,
	LOONGARCH_INS_GCSRWR,
	LOONGARCH_INS_GCSRXCHG,
	LOONGARCH_INS_GTLBFLUSH,
	LOONGARCH_INS_HVCL,
	LOONGARCH_INS_IBAR,
	LOONGARCH_INS_IDLE,
	LOONGARCH_INS_INVTLB,
	LOONGARCH_INS_IOCSRRD_B,
	LOONGARCH_INS_IOCSRRD_D,
	LOONGARCH_INS_IOCSRRD_H,
	LOONGARCH_INS_IOCSRRD_W,
	LOONGARCH_INS_IOCSRWR_B,
	LOONGARCH_INS_IOCSRWR_D,
	LOONGARCH_INS_IOCSRWR_H,
	LOONGARCH_INS_IOCSRWR_W,
	LOONGARCH_INS_JIRL,
	LOONGARCH_INS_JISCR0,
	LOONGARCH_INS_JISCR1,
	LOONGARCH_INS_LDDIR,
	LOONGARCH_INS_LDGT_B,
	LOONGARCH_INS_LDGT_D,
	LOONGARCH_INS_LDGT_H,
	LOONGARCH_INS_LDGT_W,
	LOONGARCH_INS_LDLE_B,
	LOONGARCH_INS_LDLE_D,
	LOONGARCH_INS_LDLE_H,
	LOONGARCH_INS_LDLE_W,
	LOONGARCH_INS_LDL_D,
	LOONGARCH_INS_LDL_W,
	LOONGARCH_INS_LDPTE,
	LOONGARCH_INS_LDPTR_D,
	LOONGARCH_INS_LDPTR_W,
	LOONGARCH_INS_LDR_D,
	LOONGARCH_INS_LDR_W,
	LOONGARCH_INS_LDX_B,
	LOONGARCH_INS_LDX_BU,
	LOONGARCH_INS_LDX_D,
	LOONGARCH_INS_LDX_H,
	LOONGARCH_INS_LDX_HU,
	LOONGARCH_INS_LDX_W,
	LOONGARCH_INS_LDX_WU,
	LOONGARCH_INS_LD_B,
	LOONGARCH_INS_LD_BU,
	LOONGARCH_INS_LD_D,
	LOONGARCH_INS_LD_H,
	LOONGARCH_INS_LD_HU,
	LOONGARCH_INS_LD_W,
	LOONGARCH_INS_LD_WU,
	LOONGARCH_INS_LLACQ_D,
	LOONGARCH_INS_LLACQ_W,
	LOONGARCH_INS_LL_D,
	LOONGARCH_INS_LL_W,
	LOONGARCH_INS_LU12I_W,
	LOONGARCH_INS_LU32I_D,
	LOONGARCH_INS_LU52I_D,
	LOONGARCH_INS_MASKEQZ,
	LOONGARCH_INS_MASKNEZ,
	LOONGARCH_INS_MOD_D,
	LOONGARCH_INS_MOD_DU,
	LOONGARCH_INS_MOD_W,
	LOONGARCH_INS_MOD_WU,
	LOONGARCH_INS_MOVCF2FR,
	LOONGARCH_INS_MOVCF2GR,
	LOONGARCH_INS_MOVFCSR2GR,
	LOONGARCH_INS_MOVFR2CF,
	LOONGARCH_INS_MOVFR2GR_D,
	LOONGARCH_INS_MOVFR2GR_S,
	LOONGARCH_INS_MOVFRH2GR_S,
	LOONGARCH_INS_MOVGR2CF,
	LOONGARCH_INS_MOVGR2FCSR,
	LOONGARCH_INS_MOVGR2FRH_W,
	LOONGARCH_INS_MOVGR2FR_D,
	LOONGARCH_INS_MOVGR2FR_W,
	LOONGARCH_INS_MOVGR2SCR,
	LOONGARCH_INS_MOVSCR2GR,
	LOONGARCH_INS_MULH_D,
	LOONGARCH_INS_MULH_DU,
	LOONGARCH_INS_MULH_W,
	LOONGARCH_INS_MULH_WU,
	LOONGARCH_INS_MULW_D_W,
	LOONGARCH_INS_MULW_D_WU,
	LOONGARCH_INS_MUL_D,
	LOONGARCH_INS_MUL_W,
	LOONGARCH_INS_NOR,
	LOONGARCH_INS_OR,
	LOONGARCH_INS_ORI,
	LOONGARCH_INS_ORN,
	LOONGARCH_INS_PCADDI,
	LOONGARCH_INS_PCADDU12I,
	LOONGARCH_INS_PCADDU18I,
	LOONGARCH_INS_PCALAU12I,
	LOONGARCH_INS_PRELD,
	LOONGARCH_INS_PRELDX,
	LOONGARCH_INS_RCRI_B,
	LOONGARCH_INS_RCRI_D,
	LOONGARCH_INS_RCRI_H,
	LOONGARCH_INS_RCRI_W,
	LOONGARCH_INS_RCR_B,
	LOONGARCH_INS_RCR_D,
	LOONGARCH_INS_RCR_H,
	LOONGARCH_INS_RCR_W,
	LOONGARCH_INS_RDTIMEH_W,
	LOONGARCH_INS_RDTIMEL_W,
	LOONGARCH_INS_RDTIME_D,
	LOONGARCH_INS_REVB_2H,
	LOONGARCH_INS_REVB_2W,
	LOONGARCH_INS_REVB_4H,
	LOONGARCH_INS_REVB_D,
	LOONGARCH_INS_REVH_2W,
	LOONGARCH_INS_REVH_D,
	LOONGARCH_INS_ROTRI_B,
	LOONGARCH_INS_ROTRI_D,
	LOONGARCH_INS_ROTRI_H,
	LOONGARCH_INS_ROTRI_W,
	LOONGARCH_INS_ROTR_B,
	LOONGARCH_INS_ROTR_D,
	LOONGARCH_INS_ROTR_H,
	LOONGARCH_INS_ROTR_W,
	LOONGARCH_INS_SBC_B,
	LOONGARCH_INS_SBC_D,
	LOONGARCH_INS_SBC_H,
	LOONGARCH_INS_SBC_W,
	LOONGARCH_INS_SCREL_D,
	LOONGARCH_INS_SCREL_W,
	LOONGARCH_INS_SC_D,
	LOONGARCH_INS_SC_Q,
	LOONGARCH_INS_SC_W,
	LOONGARCH_INS_SETARMJ,
	LOONGARCH_INS_SETX86J,
	LOONGARCH_INS_SETX86LOOPE,
	LOONGARCH_INS_SETX86LOOPNE,
	LOONGARCH_INS_SLLI_D,
	LOONGARCH_INS_SLLI_W,
	LOONGARCH_INS_SLL_D,
	LOONGARCH_INS_SLL_W,
	LOONGARCH_INS_SLT,
	LOONGARCH_INS_SLTI,
	LOONGARCH_INS_SLTU,
	LOONGARCH_INS_SLTUI,
	LOONGARCH_INS_SRAI_D,
	LOONGARCH_INS_SRAI_W,
	LOONGARCH_INS_SRA_D,
	LOONGARCH_INS_SRA_W,
	LOONGARCH_INS_SRLI_D,
	LOONGARCH_INS_SRLI_W,
	LOONGARCH_INS_SRL_D,
	LOONGARCH_INS_SRL_W,
	LOONGARCH_INS_STGT_B,
	LOONGARCH_INS_STGT_D,
	LOONGARCH_INS_STGT_H,
	LOONGARCH_INS_STGT_W,
	LOONGARCH_INS_STLE_B,
	LOONGARCH_INS_STLE_D,
	LOONGARCH_INS_STLE_H,
	LOONGARCH_INS_STLE_W,
	LOONGARCH_INS_STL_D,
	LOONGARCH_INS_STL_W,
	LOONGARCH_INS_STPTR_D,
	LOONGARCH_INS_STPTR_W,
	LOONGARCH_INS_STR_D,
	LOONGARCH_INS_STR_W,
	LOONGARCH_INS_STX_B,
	LOONGARCH_INS_STX_D,
	LOONGARCH_INS_STX_H,
	LOONGARCH_INS_STX_W,
	LOONGARCH_INS_ST_B,
	LOONGARCH_INS_ST_D,
	LOONGARCH_INS_ST_H,
	LOONGARCH_INS_ST_W,
	LOONGARCH_INS_SUB_D,
	LOONGARCH_INS_SUB_W,
	LOONGARCH_INS_SYSCALL,
	LOONGARCH_INS_TLBCLR,
	LOONGARCH_INS_TLBFILL,
	LOONGARCH_INS_TLBFLUSH,
	LOONGARCH_INS_TLBRD,
	LOONGARCH_INS_TLBSRCH,
	LOONGARCH_INS_TLBWR,
	LOONGARCH_INS_VABSD_B,
	LOONGARCH_INS_VABSD_BU,
	LOONGARCH_INS_VABSD_D,
	LOONGARCH_INS_VABSD_DU,
	LOONGARCH_INS_VABSD_H,
	LOONGARCH_INS_VABSD_HU,
	LOONGARCH_INS_VABSD_W,
	LOONGARCH_INS_VABSD_WU,
	LOONGARCH_INS_VADDA_B,
	LOONGARCH_INS_VADDA_D,
	LOONGARCH_INS_VADDA_H,
	LOONGARCH_INS_VADDA_W,
	LOONGARCH_INS_VADDI_BU,
	LOONGARCH_INS_VADDI_DU,
	LOONGARCH_INS_VADDI_HU,
	LOONGARCH_INS_VADDI_WU,
	LOONGARCH_INS_VADDWEV_D_W,
	LOONGARCH_INS_VADDWEV_D_WU,
	LOONGARCH_INS_VADDWEV_D_WU_W,
	LOONGARCH_INS_VADDWEV_H_B,
	LOONGARCH_INS_VADDWEV_H_BU,
	LOONGARCH_INS_VADDWEV_H_BU_B,
	LOONGARCH_INS_VADDWEV_Q_D,
	LOONGARCH_INS_VADDWEV_Q_DU,
	LOONGARCH_INS_VADDWEV_Q_DU_D,
	LOONGARCH_INS_VADDWEV_W_H,
	LOONGARCH_INS_VADDWEV_W_HU,
	LOONGARCH_INS_VADDWEV_W_HU_H,
	LOONGARCH_INS_VADDWOD_D_W,
	LOONGARCH_INS_VADDWOD_D_WU,
	LOONGARCH_INS_VADDWOD_D_WU_W,
	LOONGARCH_INS_VADDWOD_H_B,
	LOONGARCH_INS_VADDWOD_H_BU,
	LOONGARCH_INS_VADDWOD_H_BU_B,
	LOONGARCH_INS_VADDWOD_Q_D,
	LOONGARCH_INS_VADDWOD_Q_DU,
	LOONGARCH_INS_VADDWOD_Q_DU_D,
	LOONGARCH_INS_VADDWOD_W_H,
	LOONGARCH_INS_VADDWOD_W_HU,
	LOONGARCH_INS_VADDWOD_W_HU_H,
	LOONGARCH_INS_VADD_B,
	LOONGARCH_INS_VADD_D,
	LOONGARCH_INS_VADD_H,
	LOONGARCH_INS_VADD_Q,
	LOONGARCH_INS_VADD_W,
	LOONGARCH_INS_VANDI_B,
	LOONGARCH_INS_VANDN_V,
	LOONGARCH_INS_VAND_V,
	LOONGARCH_INS_VAVGR_B,
	LOONGARCH_INS_VAVGR_BU,
	LOONGARCH_INS_VAVGR_D,
	LOONGARCH_INS_VAVGR_DU,
	LOONGARCH_INS_VAVGR_H,
	LOONGARCH_INS_VAVGR_HU,
	LOONGARCH_INS_VAVGR_W,
	LOONGARCH_INS_VAVGR_WU,
	LOONGARCH_INS_VAVG_B,
	LOONGARCH_INS_VAVG_BU,
	LOONGARCH_INS_VAVG_D,
	LOONGARCH_INS_VAVG_DU,
	LOONGARCH_INS_VAVG_H,
	LOONGARCH_INS_VAVG_HU,
	LOONGARCH_INS_VAVG_W,
	LOONGARCH_INS_VAVG_WU,
	LOONGARCH_INS_VBITCLRI_B,
	LOONGARCH_INS_VBITCLRI_D,
	LOONGARCH_INS_VBITCLRI_H,
	LOONGARCH_INS_VBITCLRI_W,
	LOONGARCH_INS_VBITCLR_B,
	LOONGARCH_INS_VBITCLR_D,
	LOONGARCH_INS_VBITCLR_H,
	LOONGARCH_INS_VBITCLR_W,
	LOONGARCH_INS_VBITREVI_B,
	LOONGARCH_INS_VBITREVI_D,
	LOONGARCH_INS_VBITREVI_H,
	LOONGARCH_INS_VBITREVI_W,
	LOONGARCH_INS_VBITREV_B,
	LOONGARCH_INS_VBITREV_D,
	LOONGARCH_INS_VBITREV_H,
	LOONGARCH_INS_VBITREV_W,
	LOONGARCH_INS_VBITSELI_B,
	LOONGARCH_INS_VBITSEL_V,
	LOONGARCH_INS_VBITSETI_B,
	LOONGARCH_INS_VBITSETI_D,
	LOONGARCH_INS_VBITSETI_H,
	LOONGARCH_INS_VBITSETI_W,
	LOONGARCH_INS_VBITSET_B,
	LOONGARCH_INS_VBITSET_D,
	LOONGARCH_INS_VBITSET_H,
	LOONGARCH_INS_VBITSET_W,
	LOONGARCH_INS_VBSLL_V,
	LOONGARCH_INS_VBSRL_V,
	LOONGARCH_INS_VCLO_B,
	LOONGARCH_INS_VCLO_D,
	LOONGARCH_INS_VCLO_H,
	LOONGARCH_INS_VCLO_W,
	LOONGARCH_INS_VCLZ_B,
	LOONGARCH_INS_VCLZ_D,
	LOONGARCH_INS_VCLZ_H,
	LOONGARCH_INS_VCLZ_W,
	LOONGARCH_INS_VDIV_B,
	LOONGARCH_INS_VDIV_BU,
	LOONGARCH_INS_VDIV_D,
	LOONGARCH_INS_VDIV_DU,
	LOONGARCH_INS_VDIV_H,
	LOONGARCH_INS_VDIV_HU,
	LOONGARCH_INS_VDIV_W,
	LOONGARCH_INS_VDIV_WU,
	LOONGARCH_INS_VEXT2XV_DU_BU,
	LOONGARCH_INS_VEXT2XV_DU_HU,
	LOONGARCH_INS_VEXT2XV_DU_WU,
	LOONGARCH_INS_VEXT2XV_D_B,
	LOONGARCH_INS_VEXT2XV_D_H,
	LOONGARCH_INS_VEXT2XV_D_W,
	LOONGARCH_INS_VEXT2XV_HU_BU,
	LOONGARCH_INS_VEXT2XV_H_B,
	LOONGARCH_INS_VEXT2XV_WU_BU,
	LOONGARCH_INS_VEXT2XV_WU_HU,
	LOONGARCH_INS_VEXT2XV_W_B,
	LOONGARCH_INS_VEXT2XV_W_H,
	LOONGARCH_INS_VEXTH_DU_WU,
	LOONGARCH_INS_VEXTH_D_W,
	LOONGARCH_INS_VEXTH_HU_BU,
	LOONGARCH_INS_VEXTH_H_B,
	LOONGARCH_INS_VEXTH_QU_DU,
	LOONGARCH_INS_VEXTH_Q_D,
	LOONGARCH_INS_VEXTH_WU_HU,
	LOONGARCH_INS_VEXTH_W_H,
	LOONGARCH_INS_VEXTL_QU_DU,
	LOONGARCH_INS_VEXTL_Q_D,
	LOONGARCH_INS_VEXTRINS_B,
	LOONGARCH_INS_VEXTRINS_D,
	LOONGARCH_INS_VEXTRINS_H,
	LOONGARCH_INS_VEXTRINS_W,
	LOONGARCH_INS_VFADD_D,
	LOONGARCH_INS_VFADD_S,
	LOONGARCH_INS_VFCLASS_D,
	LOONGARCH_INS_VFCLASS_S,
	LOONGARCH_INS_VFCMP_CAF_D,
	LOONGARCH_INS_VFCMP_CAF_S,
	LOONGARCH_INS_VFCMP_CEQ_D,
	LOONGARCH_INS_VFCMP_CEQ_S,
	LOONGARCH_INS_VFCMP_CLE_D,
	LOONGARCH_INS_VFCMP_CLE_S,
	LOONGARCH_INS_VFCMP_CLT_D,
	LOONGARCH_INS_VFCMP_CLT_S,
	LOONGARCH_INS_VFCMP_CNE_D,
	LOONGARCH_INS_VFCMP_CNE_S,
	LOONGARCH_INS_VFCMP_COR_D,
	LOONGARCH_INS_VFCMP_COR_S,
	LOONGARCH_INS_VFCMP_CUEQ_D,
	LOONGARCH_INS_VFCMP_CUEQ_S,
	LOONGARCH_INS_VFCMP_CULE_D,
	LOONGARCH_INS_VFCMP_CULE_S,
	LOONGARCH_INS_VFCMP_CULT_D,
	LOONGARCH_INS_VFCMP_CULT_S,
	LOONGARCH_INS_VFCMP_CUNE_D,
	LOONGARCH_INS_VFCMP_CUNE_S,
	LOONGARCH_INS_VFCMP_CUN_D,
	LOONGARCH_INS_VFCMP_CUN_S,
	LOONGARCH_INS_VFCMP_SAF_D,
	LOONGARCH_INS_VFCMP_SAF_S,
	LOONGARCH_INS_VFCMP_SEQ_D,
	LOONGARCH_INS_VFCMP_SEQ_S,
	LOONGARCH_INS_VFCMP_SLE_D,
	LOONGARCH_INS_VFCMP_SLE_S,
	LOONGARCH_INS_VFCMP_SLT_D,
	LOONGARCH_INS_VFCMP_SLT_S,
	LOONGARCH_INS_VFCMP_SNE_D,
	LOONGARCH_INS_VFCMP_SNE_S,
	LOONGARCH_INS_VFCMP_SOR_D,
	LOONGARCH_INS_VFCMP_SOR_S,
	LOONGARCH_INS_VFCMP_SUEQ_D,
	LOONGARCH_INS_VFCMP_SUEQ_S,
	LOONGARCH_INS_VFCMP_SULE_D,
	LOONGARCH_INS_VFCMP_SULE_S,
	LOONGARCH_INS_VFCMP_SULT_D,
	LOONGARCH_INS_VFCMP_SULT_S,
	LOONGARCH_INS_VFCMP_SUNE_D,
	LOONGARCH_INS_VFCMP_SUNE_S,
	LOONGARCH_INS_VFCMP_SUN_D,
	LOONGARCH_INS_VFCMP_SUN_S,
	LOONGARCH_INS_VFCVTH_D_S,
	LOONGARCH_INS_VFCVTH_S_H,
	LOONGARCH_INS_VFCVTL_D_S,
	LOONGARCH_INS_VFCVTL_S_H,
	LOONGARCH_INS_VFCVT_H_S,
	LOONGARCH_INS_VFCVT_S_D,
	LOONGARCH_INS_VFDIV_D,
	LOONGARCH_INS_VFDIV_S,
	LOONGARCH_INS_VFFINTH_D_W,
	LOONGARCH_INS_VFFINTL_D_W,
	LOONGARCH_INS_VFFINT_D_L,
	LOONGARCH_INS_VFFINT_D_LU,
	LOONGARCH_INS_VFFINT_S_L,
	LOONGARCH_INS_VFFINT_S_W,
	LOONGARCH_INS_VFFINT_S_WU,
	LOONGARCH_INS_VFLOGB_D,
	LOONGARCH_INS_VFLOGB_S,
	LOONGARCH_INS_VFMADD_D,
	LOONGARCH_INS_VFMADD_S,
	LOONGARCH_INS_VFMAXA_D,
	LOONGARCH_INS_VFMAXA_S,
	LOONGARCH_INS_VFMAX_D,
	LOONGARCH_INS_VFMAX_S,
	LOONGARCH_INS_VFMINA_D,
	LOONGARCH_INS_VFMINA_S,
	LOONGARCH_INS_VFMIN_D,
	LOONGARCH_INS_VFMIN_S,
	LOONGARCH_INS_VFMSUB_D,
	LOONGARCH_INS_VFMSUB_S,
	LOONGARCH_INS_VFMUL_D,
	LOONGARCH_INS_VFMUL_S,
	LOONGARCH_INS_VFNMADD_D,
	LOONGARCH_INS_VFNMADD_S,
	LOONGARCH_INS_VFNMSUB_D,
	LOONGARCH_INS_VFNMSUB_S,
	LOONGARCH_INS_VFRECIPE_D,
	LOONGARCH_INS_VFRECIPE_S,
	LOONGARCH_INS_VFRECIP_D,
	LOONGARCH_INS_VFRECIP_S,
	LOONGARCH_INS_VFRINTRM_D,
	LOONGARCH_INS_VFRINTRM_S,
	LOONGARCH_INS_VFRINTRNE_D,
	LOONGARCH_INS_VFRINTRNE_S,
	LOONGARCH_INS_VFRINTRP_D,
	LOONGARCH_INS_VFRINTRP_S,
	LOONGARCH_INS_VFRINTRZ_D,
	LOONGARCH_INS_VFRINTRZ_S,
	LOONGARCH_INS_VFRINT_D,
	LOONGARCH_INS_VFRINT_S,
	LOONGARCH_INS_VFRSQRTE_D,
	LOONGARCH_INS_VFRSQRTE_S,
	LOONGARCH_INS_VFRSQRT_D,
	LOONGARCH_INS_VFRSQRT_S,
	LOONGARCH_INS_VFRSTPI_B,
	LOONGARCH_INS_VFRSTPI_H,
	LOONGARCH_INS_VFRSTP_B,
	LOONGARCH_INS_VFRSTP_H,
	LOONGARCH_INS_VFSQRT_D,
	LOONGARCH_INS_VFSQRT_S,
	LOONGARCH_INS_VFSUB_D,
	LOONGARCH_INS_VFSUB_S,
	LOONGARCH_INS_VFTINTH_L_S,
	LOONGARCH_INS_VFTINTL_L_S,
	LOONGARCH_INS_VFTINTRMH_L_S,
	LOONGARCH_INS_VFTINTRML_L_S,
	LOONGARCH_INS_VFTINTRM_L_D,
	LOONGARCH_INS_VFTINTRM_W_D,
	LOONGARCH_INS_VFTINTRM_W_S,
	LOONGARCH_INS_VFTINTRNEH_L_S,
	LOONGARCH_INS_VFTINTRNEL_L_S,
	LOONGARCH_INS_VFTINTRNE_L_D,
	LOONGARCH_INS_VFTINTRNE_W_D,
	LOONGARCH_INS_VFTINTRNE_W_S,
	LOONGARCH_INS_VFTINTRPH_L_S,
	LOONGARCH_INS_VFTINTRPL_L_S,
	LOONGARCH_INS_VFTINTRP_L_D,
	LOONGARCH_INS_VFTINTRP_W_D,
	LOONGARCH_INS_VFTINTRP_W_S,
	LOONGARCH_INS_VFTINTRZH_L_S,
	LOONGARCH_INS_VFTINTRZL_L_S,
	LOONGARCH_INS_VFTINTRZ_LU_D,
	LOONGARCH_INS_VFTINTRZ_L_D,
	LOONGARCH_INS_VFTINTRZ_WU_S,
	LOONGARCH_INS_VFTINTRZ_W_D,
	LOONGARCH_INS_VFTINTRZ_W_S,
	LOONGARCH_INS_VFTINT_LU_D,
	LOONGARCH_INS_VFTINT_L_D,
	LOONGARCH_INS_VFTINT_WU_S,
	LOONGARCH_INS_VFTINT_W_D,
	LOONGARCH_INS_VFTINT_W_S,
	LOONGARCH_INS_VHADDW_DU_WU,
	LOONGARCH_INS_VHADDW_D_W,
	LOONGARCH_INS_VHADDW_HU_BU,
	LOONGARCH_INS_VHADDW_H_B,
	LOONGARCH_INS_VHADDW_QU_DU,
	LOONGARCH_INS_VHADDW_Q_D,
	LOONGARCH_INS_VHADDW_WU_HU,
	LOONGARCH_INS_VHADDW_W_H,
	LOONGARCH_INS_VHSUBW_DU_WU,
	LOONGARCH_INS_VHSUBW_D_W,
	LOONGARCH_INS_VHSUBW_HU_BU,
	LOONGARCH_INS_VHSUBW_H_B,
	LOONGARCH_INS_VHSUBW_QU_DU,
	LOONGARCH_INS_VHSUBW_Q_D,
	LOONGARCH_INS_VHSUBW_WU_HU,
	LOONGARCH_INS_VHSUBW_W_H,
	LOONGARCH_INS_VILVH_B,
	LOONGARCH_INS_VILVH_D,
	LOONGARCH_INS_VILVH_H,
	LOONGARCH_INS_VILVH_W,
	LOONGARCH_INS_VILVL_B,
	LOONGARCH_INS_VILVL_D,
	LOONGARCH_INS_VILVL_H,
	LOONGARCH_INS_VILVL_W,
	LOONGARCH_INS_VINSGR2VR_B,
	LOONGARCH_INS_VINSGR2VR_D,
	LOONGARCH_INS_VINSGR2VR_H,
	LOONGARCH_INS_VINSGR2VR_W,
	LOONGARCH_INS_VLD,
	LOONGARCH_INS_VLDI,
	LOONGARCH_INS_VLDREPL_B,
	LOONGARCH_INS_VLDREPL_D,
	LOONGARCH_INS_VLDREPL_H,
	LOONGARCH_INS_VLDREPL_W,
	LOONGARCH_INS_VLDX,
	LOONGARCH_INS_VMADDWEV_D_W,
	LOONGARCH_INS_VMADDWEV_D_WU,
	LOONGARCH_INS_VMADDWEV_D_WU_W,
	LOONGARCH_INS_VMADDWEV_H_B,
	LOONGARCH_INS_VMADDWEV_H_BU,
	LOONGARCH_INS_VMADDWEV_H_BU_B,
	LOONGARCH_INS_VMADDWEV_Q_D,
	LOONGARCH_INS_VMADDWEV_Q_DU,
	LOONGARCH_INS_VMADDWEV_Q_DU_D,
	LOONGARCH_INS_VMADDWEV_W_H,
	LOONGARCH_INS_VMADDWEV_W_HU,
	LOONGARCH_INS_VMADDWEV_W_HU_H,
	LOONGARCH_INS_VMADDWOD_D_W,
	LOONGARCH_INS_VMADDWOD_D_WU,
	LOONGARCH_INS_VMADDWOD_D_WU_W,
	LOONGARCH_INS_VMADDWOD_H_B,
	LOONGARCH_INS_VMADDWOD_H_BU,
	LOONGARCH_INS_VMADDWOD_H_BU_B,
	LOONGARCH_INS_VMADDWOD_Q_D,
	LOONGARCH_INS_VMADDWOD_Q_DU,
	LOONGARCH_INS_VMADDWOD_Q_DU_D,
	LOONGARCH_INS_VMADDWOD_W_H,
	LOONGARCH_INS_VMADDWOD_W_HU,
	LOONGARCH_INS_VMADDWOD_W_HU_H,
	LOONGARCH_INS_VMADD_B,
	LOONGARCH_INS_VMADD_D,
	LOONGARCH_INS_VMADD_H,
	LOONGARCH_INS_VMADD_W,
	LOONGARCH_INS_VMAXI_B,
	LOONGARCH_INS_VMAXI_BU,
	LOONGARCH_INS_VMAXI_D,
	LOONGARCH_INS_VMAXI_DU,
	LOONGARCH_INS_VMAXI_H,
	LOONGARCH_INS_VMAXI_HU,
	LOONGARCH_INS_VMAXI_W,
	LOONGARCH_INS_VMAXI_WU,
	LOONGARCH_INS_VMAX_B,
	LOONGARCH_INS_VMAX_BU,
	LOONGARCH_INS_VMAX_D,
	LOONGARCH_INS_VMAX_DU,
	LOONGARCH_INS_VMAX_H,
	LOONGARCH_INS_VMAX_HU,
	LOONGARCH_INS_VMAX_W,
	LOONGARCH_INS_VMAX_WU,
	LOONGARCH_INS_VMINI_B,
	LOONGARCH_INS_VMINI_BU,
	LOONGARCH_INS_VMINI_D,
	LOONGARCH_INS_VMINI_DU,
	LOONGARCH_INS_VMINI_H,
	LOONGARCH_INS_VMINI_HU,
	LOONGARCH_INS_VMINI_W,
	LOONGARCH_INS_VMINI_WU,
	LOONGARCH_INS_VMIN_B,
	LOONGARCH_INS_VMIN_BU,
	LOONGARCH_INS_VMIN_D,
	LOONGARCH_INS_VMIN_DU,
	LOONGARCH_INS_VMIN_H,
	LOONGARCH_INS_VMIN_HU,
	LOONGARCH_INS_VMIN_W,
	LOONGARCH_INS_VMIN_WU,
	LOONGARCH_INS_VMOD_B,
	LOONGARCH_INS_VMOD_BU,
	LOONGARCH_INS_VMOD_D,
	LOONGARCH_INS_VMOD_DU,
	LOONGARCH_INS_VMOD_H,
	LOONGARCH_INS_VMOD_HU,
	LOONGARCH_INS_VMOD_W,
	LOONGARCH_INS_VMOD_WU,
	LOONGARCH_INS_VMSKGEZ_B,
	LOONGARCH_INS_VMSKLTZ_B,
	LOONGARCH_INS_VMSKLTZ_D,
	LOONGARCH_INS_VMSKLTZ_H,
	LOONGARCH_INS_VMSKLTZ_W,
	LOONGARCH_INS_VMSKNZ_B,
	LOONGARCH_INS_VMSUB_B,
	LOONGARCH_INS_VMSUB_D,
	LOONGARCH_INS_VMSUB_H,
	LOONGARCH_INS_VMSUB_W,
	LOONGARCH_INS_VMUH_B,
	LOONGARCH_INS_VMUH_BU,
	LOONGARCH_INS_VMUH_D,
	LOONGARCH_INS_VMUH_DU,
	LOONGARCH_INS_VMUH_H,
	LOONGARCH_INS_VMUH_HU,
	LOONGARCH_INS_VMUH_W,
	LOONGARCH_INS_VMUH_WU,
	LOONGARCH_INS_VMULWEV_D_W,
	LOONGARCH_INS_VMULWEV_D_WU,
	LOONGARCH_INS_VMULWEV_D_WU_W,
	LOONGARCH_INS_VMULWEV_H_B,
	LOONGARCH_INS_VMULWEV_H_BU,
	LOONGARCH_INS_VMULWEV_H_BU_B,
	LOONGARCH_INS_VMULWEV_Q_D,
	LOONGARCH_INS_VMULWEV_Q_DU,
	LOONGARCH_INS_VMULWEV_Q_DU_D,
	LOONGARCH_INS_VMULWEV_W_H,
	LOONGARCH_INS_VMULWEV_W_HU,
	LOONGARCH_INS_VMULWEV_W_HU_H,
	LOONGARCH_INS_VMULWOD_D_W,
	LOONGARCH_INS_VMULWOD_D_WU,
	LOONGARCH_INS_VMULWOD_D_WU_W,
	LOONGARCH_INS_VMULWOD_H_B,
	LOONGARCH_INS_VMULWOD_H_BU,
	LOONGARCH_INS_VMULWOD_H_BU_B,
	LOONGARCH_INS_VMULWOD_Q_D,
	LOONGARCH_INS_VMULWOD_Q_DU,
	LOONGARCH_INS_VMULWOD_Q_DU_D,
	LOONGARCH_INS_VMULWOD_W_H,
	LOONGARCH_INS_VMULWOD_W_HU,
	LOONGARCH_INS_VMULWOD_W_HU_H,
	LOONGARCH_INS_VMUL_B,
	LOONGARCH_INS_VMUL_D,
	LOONGARCH_INS_VMUL_H,
	LOONGARCH_INS_VMUL_W,
	LOONGARCH_INS_VNEG_B,
	LOONGARCH_INS_VNEG_D,
	LOONGARCH_INS_VNEG_H,
	LOONGARCH_INS_VNEG_W,
	LOONGARCH_INS_VNORI_B,
	LOONGARCH_INS_VNOR_V,
	LOONGARCH_INS_VORI_B,
	LOONGARCH_INS_VORN_V,
	LOONGARCH_INS_VOR_V,
	LOONGARCH_INS_VPACKEV_B,
	LOONGARCH_INS_VPACKEV_D,
	LOONGARCH_INS_VPACKEV_H,
	LOONGARCH_INS_VPACKEV_W,
	LOONGARCH_INS_VPACKOD_B,
	LOONGARCH_INS_VPACKOD_D,
	LOONGARCH_INS_VPACKOD_H,
	LOONGARCH_INS_VPACKOD_W,
	LOONGARCH_INS_VPCNT_B,
	LOONGARCH_INS_VPCNT_D,
	LOONGARCH_INS_VPCNT_H,
	LOONGARCH_INS_VPCNT_W,
	LOONGARCH_INS_VPERMI_W,
	LOONGARCH_INS_VPICKEV_B,
	LOONGARCH_INS_VPICKEV_D,
	LOONGARCH_INS_VPICKEV_H,
	LOONGARCH_INS_VPICKEV_W,
	LOONGARCH_INS_VPICKOD_B,
	LOONGARCH_INS_VPICKOD_D,
	LOONGARCH_INS_VPICKOD_H,
	LOONGARCH_INS_VPICKOD_W,
	LOONGARCH_INS_VPICKVE2GR_B,
	LOONGARCH_INS_VPICKVE2GR_BU,
	LOONGARCH_INS_VPICKVE2GR_D,
	LOONGARCH_INS_VPICKVE2GR_DU,
	LOONGARCH_INS_VPICKVE2GR_H,
	LOONGARCH_INS_VPICKVE2GR_HU,
	LOONGARCH_INS_VPICKVE2GR_W,
	LOONGARCH_INS_VPICKVE2GR_WU,
	LOONGARCH_INS_VREPLGR2VR_B,
	LOONGARCH_INS_VREPLGR2VR_D,
	LOONGARCH_INS_VREPLGR2VR_H,
	LOONGARCH_INS_VREPLGR2VR_W,
	LOONGARCH_INS_VREPLVEI_B,
	LOONGARCH_INS_VREPLVEI_D,
	LOONGARCH_INS_VREPLVEI_H,
	LOONGARCH_INS_VREPLVEI_W,
	LOONGARCH_INS_VREPLVE_B,
	LOONGARCH_INS_VREPLVE_D,
	LOONGARCH_INS_VREPLVE_H,
	LOONGARCH_INS_VREPLVE_W,
	LOONGARCH_INS_VROTRI_B,
	LOONGARCH_INS_VROTRI_D,
	LOONGARCH_INS_VROTRI_H,
	LOONGARCH_INS_VROTRI_W,
	LOONGARCH_INS_VROTR_B,
	LOONGARCH_INS_VROTR_D,
	LOONGARCH_INS_VROTR_H,
	LOONGARCH_INS_VROTR_W,
	LOONGARCH_INS_VSADD_B,
	LOONGARCH_INS_VSADD_BU,
	LOONGARCH_INS_VSADD_D,
	LOONGARCH_INS_VSADD_DU,
	LOONGARCH_INS_VSADD_H,
	LOONGARCH_INS_VSADD_HU,
	LOONGARCH_INS_VSADD_W,
	LOONGARCH_INS_VSADD_WU,
	LOONGARCH_INS_VSAT_B,
	LOONGARCH_INS_VSAT_BU,
	LOONGARCH_INS_VSAT_D,
	LOONGARCH_INS_VSAT_DU,
	LOONGARCH_INS_VSAT_H,
	LOONGARCH_INS_VSAT_HU,
	LOONGARCH_INS_VSAT_W,
	LOONGARCH_INS_VSAT_WU,
	LOONGARCH_INS_VSEQI_B,
	LOONGARCH_INS_VSEQI_D,
	LOONGARCH_INS_VSEQI_H,
	LOONGARCH_INS_VSEQI_W,
	LOONGARCH_INS_VSEQ_B,
	LOONGARCH_INS_VSEQ_D,
	LOONGARCH_INS_VSEQ_H,
	LOONGARCH_INS_VSEQ_W,
	LOONGARCH_INS_VSETALLNEZ_B,
	LOONGARCH_INS_VSETALLNEZ_D,
	LOONGARCH_INS_VSETALLNEZ_H,
	LOONGARCH_INS_VSETALLNEZ_W,
	LOONGARCH_INS_VSETANYEQZ_B,
	LOONGARCH_INS_VSETANYEQZ_D,
	LOONGARCH_INS_VSETANYEQZ_H,
	LOONGARCH_INS_VSETANYEQZ_W,
	LOONGARCH_INS_VSETEQZ_V,
	LOONGARCH_INS_VSETNEZ_V,
	LOONGARCH_INS_VSHUF4I_B,
	LOONGARCH_INS_VSHUF4I_D,
	LOONGARCH_INS_VSHUF4I_H,
	LOONGARCH_INS_VSHUF4I_W,
	LOONGARCH_INS_VSHUF_B,
	LOONGARCH_INS_VSHUF_D,
	LOONGARCH_INS_VSHUF_H,
	LOONGARCH_INS_VSHUF_W,
	LOONGARCH_INS_VSIGNCOV_B,
	LOONGARCH_INS_VSIGNCOV_D,
	LOONGARCH_INS_VSIGNCOV_H,
	LOONGARCH_INS_VSIGNCOV_W,
	LOONGARCH_INS_VSLEI_B,
	LOONGARCH_INS_VSLEI_BU,
	LOONGARCH_INS_VSLEI_D,
	LOONGARCH_INS_VSLEI_DU,
	LOONGARCH_INS_VSLEI_H,
	LOONGARCH_INS_VSLEI_HU,
	LOONGARCH_INS_VSLEI_W,
	LOONGARCH_INS_VSLEI_WU,
	LOONGARCH_INS_VSLE_B,
	LOONGARCH_INS_VSLE_BU,
	LOONGARCH_INS_VSLE_D,
	LOONGARCH_INS_VSLE_DU,
	LOONGARCH_INS_VSLE_H,
	LOONGARCH_INS_VSLE_HU,
	LOONGARCH_INS_VSLE_W,
	LOONGARCH_INS_VSLE_WU,
	LOONGARCH_INS_VSLLI_B,
	LOONGARCH_INS_VSLLI_D,
	LOONGARCH_INS_VSLLI_H,
	LOONGARCH_INS_VSLLI_W,
	LOONGARCH_INS_VSLLWIL_DU_WU,
	LOONGARCH_INS_VSLLWIL_D_W,
	LOONGARCH_INS_VSLLWIL_HU_BU,
	LOONGARCH_INS_VSLLWIL_H_B,
	LOONGARCH_INS_VSLLWIL_WU_HU,
	LOONGARCH_INS_VSLLWIL_W_H,
	LOONGARCH_INS_VSLL_B,
	LOONGARCH_INS_VSLL_D,
	LOONGARCH_INS_VSLL_H,
	LOONGARCH_INS_VSLL_W,
	LOONGARCH_INS_VSLTI_B,
	LOONGARCH_INS_VSLTI_BU,
	LOONGARCH_INS_VSLTI_D,
	LOONGARCH_INS_VSLTI_DU,
	LOONGARCH_INS_VSLTI_H,
	LOONGARCH_INS_VSLTI_HU,
	LOONGARCH_INS_VSLTI_W,
	LOONGARCH_INS_VSLTI_WU,
	LOONGARCH_INS_VSLT_B,
	LOONGARCH_INS_VSLT_BU,
	LOONGARCH_INS_VSLT_D,
	LOONGARCH_INS_VSLT_DU,
	LOONGARCH_INS_VSLT_H,
	LOONGARCH_INS_VSLT_HU,
	LOONGARCH_INS_VSLT_W,
	LOONGARCH_INS_VSLT_WU,
	LOONGARCH_INS_VSRAI_B,
	LOONGARCH_INS_VSRAI_D,
	LOONGARCH_INS_VSRAI_H,
	LOONGARCH_INS_VSRAI_W,
	LOONGARCH_INS_VSRANI_B_H,
	LOONGARCH_INS_VSRANI_D_Q,
	LOONGARCH_INS_VSRANI_H_W,
	LOONGARCH_INS_VSRANI_W_D,
	LOONGARCH_INS_VSRAN_B_H,
	LOONGARCH_INS_VSRAN_H_W,
	LOONGARCH_INS_VSRAN_W_D,
	LOONGARCH_INS_VSRARI_B,
	LOONGARCH_INS_VSRARI_D,
	LOONGARCH_INS_VSRARI_H,
	LOONGARCH_INS_VSRARI_W,
	LOONGARCH_INS_VSRARNI_B_H,
	LOONGARCH_INS_VSRARNI_D_Q,
	LOONGARCH_INS_VSRARNI_H_W,
	LOONGARCH_INS_VSRARNI_W_D,
	LOONGARCH_INS_VSRARN_B_H,
	LOONGARCH_INS_VSRARN_H_W,
	LOONGARCH_INS_VSRARN_W_D,
	LOONGARCH_INS_VSRAR_B,
	LOONGARCH_INS_VSRAR_D,
	LOONGARCH_INS_VSRAR_H,
	LOONGARCH_INS_VSRAR_W,
	LOONGARCH_INS_VSRA_B,
	LOONGARCH_INS_VSRA_D,
	LOONGARCH_INS_VSRA_H,
	LOONGARCH_INS_VSRA_W,
	LOONGARCH_INS_VSRLI_B,
	LOONGARCH_INS_VSRLI_D,
	LOONGARCH_INS_VSRLI_H,
	LOONGARCH_INS_VSRLI_W,
	LOONGARCH_INS_VSRLNI_B_H,
	LOONGARCH_INS_VSRLNI_D_Q,
	LOONGARCH_INS_VSRLNI_H_W,
	LOONGARCH_INS_VSRLNI_W_D,
	LOONGARCH_INS_VSRLN_B_H,
	LOONGARCH_INS_VSRLN_H_W,
	LOONGARCH_INS_VSRLN_W_D,
	LOONGARCH_INS_VSRLRI_B,
	LOONGARCH_INS_VSRLRI_D,
	LOONGARCH_INS_VSRLRI_H,
	LOONGARCH_INS_VSRLRI_W,
	LOONGARCH_INS_VSRLRNI_B_H,
	LOONGARCH_INS_VSRLRNI_D_Q,
	LOONGARCH_INS_VSRLRNI_H_W,
	LOONGARCH_INS_VSRLRNI_W_D,
	LOONGARCH_INS_VSRLRN_B_H,
	LOONGARCH_INS_VSRLRN_H_W,
	LOONGARCH_INS_VSRLRN_W_D,
	LOONGARCH_INS_VSRLR_B,
	LOONGARCH_INS_VSRLR_D,
	LOONGARCH_INS_VSRLR_H,
	LOONGARCH_INS_VSRLR_W,
	LOONGARCH_INS_VSRL_B,
	LOONGARCH_INS_VSRL_D,
	LOONGARCH_INS_VSRL_H,
	LOONGARCH_INS_VSRL_W,
	LOONGARCH_INS_VSSRANI_BU_H,
	LOONGARCH_INS_VSSRANI_B_H,
	LOONGARCH_INS_VSSRANI_DU_Q,
	LOONGARCH_INS_VSSRANI_D_Q,
	LOONGARCH_INS_VSSRANI_HU_W,
	LOONGARCH_INS_VSSRANI_H_W,
	LOONGARCH_INS_VSSRANI_WU_D,
	LOONGARCH_INS_VSSRANI_W_D,
	LOONGARCH_INS_VSSRAN_BU_H,
	LOONGARCH_INS_VSSRAN_B_H,
	LOONGARCH_INS_VSSRAN_HU_W,
	LOONGARCH_INS_VSSRAN_H_W,
	LOONGARCH_INS_VSSRAN_WU_D,
	LOONGARCH_INS_VSSRAN_W_D,
	LOONGARCH_INS_VSSRARNI_BU_H,
	LOONGARCH_INS_VSSRARNI_B_H,
	LOONGARCH_INS_VSSRARNI_DU_Q,
	LOONGARCH_INS_VSSRARNI_D_Q,
	LOONGARCH_INS_VSSRARNI_HU_W,
	LOONGARCH_INS_VSSRARNI_H_W,
	LOONGARCH_INS_VSSRARNI_WU_D,
	LOONGARCH_INS_VSSRARNI_W_D,
	LOONGARCH_INS_VSSRARN_BU_H,
	LOONGARCH_INS_VSSRARN_B_H,
	LOONGARCH_INS_VSSRARN_HU_W,
	LOONGARCH_INS_VSSRARN_H_W,
	LOONGARCH_INS_VSSRARN_WU_D,
	LOONGARCH_INS_VSSRARN_W_D,
	LOONGARCH_INS_VSSRLNI_BU_H,
	LOONGARCH_INS_VSSRLNI_B_H,
	LOONGARCH_INS_VSSRLNI_DU_Q,
	LOONGARCH_INS_VSSRLNI_D_Q,
	LOONGARCH_INS_VSSRLNI_HU_W,
	LOONGARCH_INS_VSSRLNI_H_W,
	LOONGARCH_INS_VSSRLNI_WU_D,
	LOONGARCH_INS_VSSRLNI_W_D,
	LOONGARCH_INS_VSSRLN_BU_H,
	LOONGARCH_INS_VSSRLN_B_H,
	LOONGARCH_INS_VSSRLN_HU_W,
	LOONGARCH_INS_VSSRLN_H_W,
	LOONGARCH_INS_VSSRLN_WU_D,
	LOONGARCH_INS_VSSRLN_W_D,
	LOONGARCH_INS_VSSRLRNI_BU_H,
	LOONGARCH_INS_VSSRLRNI_B_H,
	LOONGARCH_INS_VSSRLRNI_DU_Q,
	LOONGARCH_INS_VSSRLRNI_D_Q,
	LOONGARCH_INS_VSSRLRNI_HU_W,
	LOONGARCH_INS_VSSRLRNI_H_W,
	LOONGARCH_INS_VSSRLRNI_WU_D,
	LOONGARCH_INS_VSSRLRNI_W_D,
	LOONGARCH_INS_VSSRLRN_BU_H,
	LOONGARCH_INS_VSSRLRN_B_H,
	LOONGARCH_INS_VSSRLRN_HU_W,
	LOONGARCH_INS_VSSRLRN_H_W,
	LOONGARCH_INS_VSSRLRN_WU_D,
	LOONGARCH_INS_VSSRLRN_W_D,
	LOONGARCH_INS_VSSUB_B,
	LOONGARCH_INS_VSSUB_BU,
	LOONGARCH_INS_VSSUB_D,
	LOONGARCH_INS_VSSUB_DU,
	LOONGARCH_INS_VSSUB_H,
	LOONGARCH_INS_VSSUB_HU,
	LOONGARCH_INS_VSSUB_W,
	LOONGARCH_INS_VSSUB_WU,
	LOONGARCH_INS_VST,
	LOONGARCH_INS_VSTELM_B,
	LOONGARCH_INS_VSTELM_D,
	LOONGARCH_INS_VSTELM_H,
	LOONGARCH_INS_VSTELM_W,
	LOONGARCH_INS_VSTX,
	LOONGARCH_INS_VSUBI_BU,
	LOONGARCH_INS_VSUBI_DU,
	LOONGARCH_INS_VSUBI_HU,
	LOONGARCH_INS_VSUBI_WU,
	LOONGARCH_INS_VSUBWEV_D_W,
	LOONGARCH_INS_VSUBWEV_D_WU,
	LOONGARCH_INS_VSUBWEV_H_B,
	LOONGARCH_INS_VSUBWEV_H_BU,
	LOONGARCH_INS_VSUBWEV_Q_D,
	LOONGARCH_INS_VSUBWEV_Q_DU,
	LOONGARCH_INS_VSUBWEV_W_H,
	LOONGARCH_INS_VSUBWEV_W_HU,
	LOONGARCH_INS_VSUBWOD_D_W,
	LOONGARCH_INS_VSUBWOD_D_WU,
	LOONGARCH_INS_VSUBWOD_H_B,
	LOONGARCH_INS_VSUBWOD_H_BU,
	LOONGARCH_INS_VSUBWOD_Q_D,
	LOONGARCH_INS_VSUBWOD_Q_DU,
	LOONGARCH_INS_VSUBWOD_W_H,
	LOONGARCH_INS_VSUBWOD_W_HU,
	LOONGARCH_INS_VSUB_B,
	LOONGARCH_INS_VSUB_D,
	LOONGARCH_INS_VSUB_H,
	LOONGARCH_INS_VSUB_Q,
	LOONGARCH_INS_VSUB_W,
	LOONGARCH_INS_VXORI_B,
	LOONGARCH_INS_VXOR_V,
	LOONGARCH_INS_X86ADC_B,
	LOONGARCH_INS_X86ADC_D,
	LOONGARCH_INS_X86ADC_H,
	LOONGARCH_INS_X86ADC_W,
	LOONGARCH_INS_X86ADD_B,
	LOONGARCH_INS_X86ADD_D,
	LOONGARCH_INS_X86ADD_DU,
	LOONGARCH_INS_X86ADD_H,
	LOONGARCH_INS_X86ADD_W,
	LOONGARCH_INS_X86ADD_WU,
	LOONGARCH_INS_X86AND_B,
	LOONGARCH_INS_X86AND_D,
	LOONGARCH_INS_X86AND_H,
	LOONGARCH_INS_X86AND_W,
	LOONGARCH_INS_X86CLRTM,
	LOONGARCH_INS_X86DECTOP,
	LOONGARCH_INS_X86DEC_B,
	LOONGARCH_INS_X86DEC_D,
	LOONGARCH_INS_X86DEC_H,
	LOONGARCH_INS_X86DEC_W,
	LOONGARCH_INS_X86INCTOP,
	LOONGARCH_INS_X86INC_B,
	LOONGARCH_INS_X86INC_D,
	LOONGARCH_INS_X86INC_H,
	LOONGARCH_INS_X86INC_W,
	LOONGARCH_INS_X86MFFLAG,
	LOONGARCH_INS_X86MFTOP,
	LOONGARCH_INS_X86MTFLAG,
	LOONGARCH_INS_X86MTTOP,
	LOONGARCH_INS_X86MUL_B,
	LOONGARCH_INS_X86MUL_BU,
	LOONGARCH_INS_X86MUL_D,
	LOONGARCH_INS_X86MUL_DU,
	LOONGARCH_INS_X86MUL_H,
	LOONGARCH_INS_X86MUL_HU,
	LOONGARCH_INS_X86MUL_W,
	LOONGARCH_INS_X86MUL_WU,
	LOONGARCH_INS_X86OR_B,
	LOONGARCH_INS_X86OR_D,
	LOONGARCH_INS_X86OR_H,
	LOONGARCH_INS_X86OR_W,
	LOONGARCH_INS_X86RCLI_B,
	LOONGARCH_INS_X86RCLI_D,
	LOONGARCH_INS_X86RCLI_H,
	LOONGARCH_INS_X86RCLI_W,
	LOONGARCH_INS_X86RCL_B,
	LOONGARCH_INS_X86RCL_D,
	LOONGARCH_INS_X86RCL_H,
	LOONGARCH_INS_X86RCL_W,
	LOONGARCH_INS_X86RCRI_B,
	LOONGARCH_INS_X86RCRI_D,
	LOONGARCH_INS_X86RCRI_H,
	LOONGARCH_INS_X86RCRI_W,
	LOONGARCH_INS_X86RCR_B,
	LOONGARCH_INS_X86RCR_D,
	LOONGARCH_INS_X86RCR_H,
	LOONGARCH_INS_X86RCR_W,
	LOONGARCH_INS_X86ROTLI_B,
	LOONGARCH_INS_X86ROTLI_D,
	LOONGARCH_INS_X86ROTLI_H,
	LOONGARCH_INS_X86ROTLI_W,
	LOONGARCH_INS_X86ROTL_B,
	LOONGARCH_INS_X86ROTL_D,
	LOONGARCH_INS_X86ROTL_H,
	LOONGARCH_INS_X86ROTL_W,
	LOONGARCH_INS_X86ROTRI_B,
	LOONGARCH_INS_X86ROTRI_D,
	LOONGARCH_INS_X86ROTRI_H,
	LOONGARCH_INS_X86ROTRI_W,
	LOONGARCH_INS_X86ROTR_B,
	LOONGARCH_INS_X86ROTR_D,
	LOONGARCH_INS_X86ROTR_H,
	LOONGARCH_INS_X86ROTR_W,
	LOONGARCH_INS_X86SBC_B,
	LOONGARCH_INS_X86SBC_D,
	LOONGARCH_INS_X86SBC_H,
	LOONGARCH_INS_X86SBC_W,
	LOONGARCH_INS_X86SETTAG,
	LOONGARCH_INS_X86SETTM,
	LOONGARCH_INS_X86SLLI_B,
	LOONGARCH_INS_X86SLLI_D,
	LOONGARCH_INS_X86SLLI_H,
	LOONGARCH_INS_X86SLLI_W,
	LOONGARCH_INS_X86SLL_B,
	LOONGARCH_INS_X86SLL_D,
	LOONGARCH_INS_X86SLL_H,
	LOONGARCH_INS_X86SLL_W,
	LOONGARCH_INS_X86SRAI_B,
	LOONGARCH_INS_X86SRAI_D,
	LOONGARCH_INS_X86SRAI_H,
	LOONGARCH_INS_X86SRAI_W,
	LOONGARCH_INS_X86SRA_B,
	LOONGARCH_INS_X86SRA_D,
	LOONGARCH_INS_X86SRA_H,
	LOONGARCH_INS_X86SRA_W,
	LOONGARCH_INS_X86SRLI_B,
	LOONGARCH_INS_X86SRLI_D,
	LOONGARCH_INS_X86SRLI_H,
	LOONGARCH_INS_X86SRLI_W,
	LOONGARCH_INS_X86SRL_B,
	LOONGARCH_INS_X86SRL_D,
	LOONGARCH_INS_X86SRL_H,
	LOONGARCH_INS_X86SRL_W,
	LOONGARCH_INS_X86SUB_B,
	LOONGARCH_INS_X86SUB_D,
	LOONGARCH_INS_X86SUB_DU,
	LOONGARCH_INS_X86SUB_H,
	LOONGARCH_INS_X86SUB_W,
	LOONGARCH_INS_X86SUB_WU,
	LOONGARCH_INS_X86XOR_B,
	LOONGARCH_INS_X86XOR_D,
	LOONGARCH_INS_X86XOR_H,
	LOONGARCH_INS_X86XOR_W,
	LOONGARCH_INS_XOR,
	LOONGARCH_INS_XORI,
	LOONGARCH_INS_XVABSD_B,
	LOONGARCH_INS_XVABSD_BU,
	LOONGARCH_INS_XVABSD_D,
	LOONGARCH_INS_XVABSD_DU,
	LOONGARCH_INS_XVABSD_H,
	LOONGARCH_INS_XVABSD_HU,
	LOONGARCH_INS_XVABSD_W,
	LOONGARCH_INS_XVABSD_WU,
	LOONGARCH_INS_XVADDA_B,
	LOONGARCH_INS_XVADDA_D,
	LOONGARCH_INS_XVADDA_H,
	LOONGARCH_INS_XVADDA_W,
	LOONGARCH_INS_XVADDI_BU,
	LOONGARCH_INS_XVADDI_DU,
	LOONGARCH_INS_XVADDI_HU,
	LOONGARCH_INS_XVADDI_WU,
	LOONGARCH_INS_XVADDWEV_D_W,
	LOONGARCH_INS_XVADDWEV_D_WU,
	LOONGARCH_INS_XVADDWEV_D_WU_W,
	LOONGARCH_INS_XVADDWEV_H_B,
	LOONGARCH_INS_XVADDWEV_H_BU,
	LOONGARCH_INS_XVADDWEV_H_BU_B,
	LOONGARCH_INS_XVADDWEV_Q_D,
	LOONGARCH_INS_XVADDWEV_Q_DU,
	LOONGARCH_INS_XVADDWEV_Q_DU_D,
	LOONGARCH_INS_XVADDWEV_W_H,
	LOONGARCH_INS_XVADDWEV_W_HU,
	LOONGARCH_INS_XVADDWEV_W_HU_H,
	LOONGARCH_INS_XVADDWOD_D_W,
	LOONGARCH_INS_XVADDWOD_D_WU,
	LOONGARCH_INS_XVADDWOD_D_WU_W,
	LOONGARCH_INS_XVADDWOD_H_B,
	LOONGARCH_INS_XVADDWOD_H_BU,
	LOONGARCH_INS_XVADDWOD_H_BU_B,
	LOONGARCH_INS_XVADDWOD_Q_D,
	LOONGARCH_INS_XVADDWOD_Q_DU,
	LOONGARCH_INS_XVADDWOD_Q_DU_D,
	LOONGARCH_INS_XVADDWOD_W_H,
	LOONGARCH_INS_XVADDWOD_W_HU,
	LOONGARCH_INS_XVADDWOD_W_HU_H,
	LOONGARCH_INS_XVADD_B,
	LOONGARCH_INS_XVADD_D,
	LOONGARCH_INS_XVADD_H,
	LOONGARCH_INS_XVADD_Q,
	LOONGARCH_INS_XVADD_W,
	LOONGARCH_INS_XVANDI_B,
	LOONGARCH_INS_XVANDN_V,
	LOONGARCH_INS_XVAND_V,
	LOONGARCH_INS_XVAVGR_B,
	LOONGARCH_INS_XVAVGR_BU,
	LOONGARCH_INS_XVAVGR_D,
	LOONGARCH_INS_XVAVGR_DU,
	LOONGARCH_INS_XVAVGR_H,
	LOONGARCH_INS_XVAVGR_HU,
	LOONGARCH_INS_XVAVGR_W,
	LOONGARCH_INS_XVAVGR_WU,
	LOONGARCH_INS_XVAVG_B,
	LOONGARCH_INS_XVAVG_BU,
	LOONGARCH_INS_XVAVG_D,
	LOONGARCH_INS_XVAVG_DU,
	LOONGARCH_INS_XVAVG_H,
	LOONGARCH_INS_XVAVG_HU,
	LOONGARCH_INS_XVAVG_W,
	LOONGARCH_INS_XVAVG_WU,
	LOONGARCH_INS_XVBITCLRI_B,
	LOONGARCH_INS_XVBITCLRI_D,
	LOONGARCH_INS_XVBITCLRI_H,
	LOONGARCH_INS_XVBITCLRI_W,
	LOONGARCH_INS_XVBITCLR_B,
	LOONGARCH_INS_XVBITCLR_D,
	LOONGARCH_INS_XVBITCLR_H,
	LOONGARCH_INS_XVBITCLR_W,
	LOONGARCH_INS_XVBITREVI_B,
	LOONGARCH_INS_XVBITREVI_D,
	LOONGARCH_INS_XVBITREVI_H,
	LOONGARCH_INS_XVBITREVI_W,
	LOONGARCH_INS_XVBITREV_B,
	LOONGARCH_INS_XVBITREV_D,
	LOONGARCH_INS_XVBITREV_H,
	LOONGARCH_INS_XVBITREV_W,
	LOONGARCH_INS_XVBITSELI_B,
	LOONGARCH_INS_XVBITSEL_V,
	LOONGARCH_INS_XVBITSETI_B,
	LOONGARCH_INS_XVBITSETI_D,
	LOONGARCH_INS_XVBITSETI_H,
	LOONGARCH_INS_XVBITSETI_W,
	LOONGARCH_INS_XVBITSET_B,
	LOONGARCH_INS_XVBITSET_D,
	LOONGARCH_INS_XVBITSET_H,
	LOONGARCH_INS_XVBITSET_W,
	LOONGARCH_INS_XVBSLL_V,
	LOONGARCH_INS_XVBSRL_V,
	LOONGARCH_INS_XVCLO_B,
	LOONGARCH_INS_XVCLO_D,
	LOONGARCH_INS_XVCLO_H,
	LOONGARCH_INS_XVCLO_W,
	LOONGARCH_INS_XVCLZ_B,
	LOONGARCH_INS_XVCLZ_D,
	LOONGARCH_INS_XVCLZ_H,
	LOONGARCH_INS_XVCLZ_W,
	LOONGARCH_INS_XVDIV_B,
	LOONGARCH_INS_XVDIV_BU,
	LOONGARCH_INS_XVDIV_D,
	LOONGARCH_INS_XVDIV_DU,
	LOONGARCH_INS_XVDIV_H,
	LOONGARCH_INS_XVDIV_HU,
	LOONGARCH_INS_XVDIV_W,
	LOONGARCH_INS_XVDIV_WU,
	LOONGARCH_INS_XVEXTH_DU_WU,
	LOONGARCH_INS_XVEXTH_D_W,
	LOONGARCH_INS_XVEXTH_HU_BU,
	LOONGARCH_INS_XVEXTH_H_B,
	LOONGARCH_INS_XVEXTH_QU_DU,
	LOONGARCH_INS_XVEXTH_Q_D,
	LOONGARCH_INS_XVEXTH_WU_HU,
	LOONGARCH_INS_XVEXTH_W_H,
	LOONGARCH_INS_XVEXTL_QU_DU,
	LOONGARCH_INS_XVEXTL_Q_D,
	LOONGARCH_INS_XVEXTRINS_B,
	LOONGARCH_INS_XVEXTRINS_D,
	LOONGARCH_INS_XVEXTRINS_H,
	LOONGARCH_INS_XVEXTRINS_W,
	LOONGARCH_INS_XVFADD_D,
	LOONGARCH_INS_XVFADD_S,
	LOONGARCH_INS_XVFCLASS_D,
	LOONGARCH_INS_XVFCLASS_S,
	LOONGARCH_INS_XVFCMP_CAF_D,
	LOONGARCH_INS_XVFCMP_CAF_S,
	LOONGARCH_INS_XVFCMP_CEQ_D,
	LOONGARCH_INS_XVFCMP_CEQ_S,
	LOONGARCH_INS_XVFCMP_CLE_D,
	LOONGARCH_INS_XVFCMP_CLE_S,
	LOONGARCH_INS_XVFCMP_CLT_D,
	LOONGARCH_INS_XVFCMP_CLT_S,
	LOONGARCH_INS_XVFCMP_CNE_D,
	LOONGARCH_INS_XVFCMP_CNE_S,
	LOONGARCH_INS_XVFCMP_COR_D,
	LOONGARCH_INS_XVFCMP_COR_S,
	LOONGARCH_INS_XVFCMP_CUEQ_D,
	LOONGARCH_INS_XVFCMP_CUEQ_S,
	LOONGARCH_INS_XVFCMP_CULE_D,
	LOONGARCH_INS_XVFCMP_CULE_S,
	LOONGARCH_INS_XVFCMP_CULT_D,
	LOONGARCH_INS_XVFCMP_CULT_S,
	LOONGARCH_INS_XVFCMP_CUNE_D,
	LOONGARCH_INS_XVFCMP_CUNE_S,
	LOONGARCH_INS_XVFCMP_CUN_D,
	LOONGARCH_INS_XVFCMP_CUN_S,
	LOONGARCH_INS_XVFCMP_SAF_D,
	LOONGARCH_INS_XVFCMP_SAF_S,
	LOONGARCH_INS_XVFCMP_SEQ_D,
	LOONGARCH_INS_XVFCMP_SEQ_S,
	LOONGARCH_INS_XVFCMP_SLE_D,
	LOONGARCH_INS_XVFCMP_SLE_S,
	LOONGARCH_INS_XVFCMP_SLT_D,
	LOONGARCH_INS_XVFCMP_SLT_S,
	LOONGARCH_INS_XVFCMP_SNE_D,
	LOONGARCH_INS_XVFCMP_SNE_S,
	LOONGARCH_INS_XVFCMP_SOR_D,
	LOONGARCH_INS_XVFCMP_SOR_S,
	LOONGARCH_INS_XVFCMP_SUEQ_D,
	LOONGARCH_INS_XVFCMP_SUEQ_S,
	LOONGARCH_INS_XVFCMP_SULE_D,
	LOONGARCH_INS_XVFCMP_SULE_S,
	LOONGARCH_INS_XVFCMP_SULT_D,
	LOONGARCH_INS_XVFCMP_SULT_S,
	LOONGARCH_INS_XVFCMP_SUNE_D,
	LOONGARCH_INS_XVFCMP_SUNE_S,
	LOONGARCH_INS_XVFCMP_SUN_D,
	LOONGARCH_INS_XVFCMP_SUN_S,
	LOONGARCH_INS_XVFCVTH_D_S,
	LOONGARCH_INS_XVFCVTH_S_H,
	LOONGARCH_INS_XVFCVTL_D_S,
	LOONGARCH_INS_XVFCVTL_S_H,
	LOONGARCH_INS_XVFCVT_H_S,
	LOONGARCH_INS_XVFCVT_S_D,
	LOONGARCH_INS_XVFDIV_D,
	LOONGARCH_INS_XVFDIV_S,
	LOONGARCH_INS_XVFFINTH_D_W,
	LOONGARCH_INS_XVFFINTL_D_W,
	LOONGARCH_INS_XVFFINT_D_L,
	LOONGARCH_INS_XVFFINT_D_LU,
	LOONGARCH_INS_XVFFINT_S_L,
	LOONGARCH_INS_XVFFINT_S_W,
	LOONGARCH_INS_XVFFINT_S_WU,
	LOONGARCH_INS_XVFLOGB_D,
	LOONGARCH_INS_XVFLOGB_S,
	LOONGARCH_INS_XVFMADD_D,
	LOONGARCH_INS_XVFMADD_S,
	LOONGARCH_INS_XVFMAXA_D,
	LOONGARCH_INS_XVFMAXA_S,
	LOONGARCH_INS_XVFMAX_D,
	LOONGARCH_INS_XVFMAX_S,
	LOONGARCH_INS_XVFMINA_D,
	LOONGARCH_INS_XVFMINA_S,
	LOONGARCH_INS_XVFMIN_D,
	LOONGARCH_INS_XVFMIN_S,
	LOONGARCH_INS_XVFMSUB_D,
	LOONGARCH_INS_XVFMSUB_S,
	LOONGARCH_INS_XVFMUL_D,
	LOONGARCH_INS_XVFMUL_S,
	LOONGARCH_INS_XVFNMADD_D,
	LOONGARCH_INS_XVFNMADD_S,
	LOONGARCH_INS_XVFNMSUB_D,
	LOONGARCH_INS_XVFNMSUB_S,
	LOONGARCH_INS_XVFRECIPE_D,
	LOONGARCH_INS_XVFRECIPE_S,
	LOONGARCH_INS_XVFRECIP_D,
	LOONGARCH_INS_XVFRECIP_S,
	LOONGARCH_INS_XVFRINTRM_D,
	LOONGARCH_INS_XVFRINTRM_S,
	LOONGARCH_INS_XVFRINTRNE_D,
	LOONGARCH_INS_XVFRINTRNE_S,
	LOONGARCH_INS_XVFRINTRP_D,
	LOONGARCH_INS_XVFRINTRP_S,
	LOONGARCH_INS_XVFRINTRZ_D,
	LOONGARCH_INS_XVFRINTRZ_S,
	LOONGARCH_INS_XVFRINT_D,
	LOONGARCH_INS_XVFRINT_S,
	LOONGARCH_INS_XVFRSQRTE_D,
	LOONGARCH_INS_XVFRSQRTE_S,
	LOONGARCH_INS_XVFRSQRT_D,
	LOONGARCH_INS_XVFRSQRT_S,
	LOONGARCH_INS_XVFRSTPI_B,
	LOONGARCH_INS_XVFRSTPI_H,
	LOONGARCH_INS_XVFRSTP_B,
	LOONGARCH_INS_XVFRSTP_H,
	LOONGARCH_INS_XVFSQRT_D,
	LOONGARCH_INS_XVFSQRT_S,
	LOONGARCH_INS_XVFSUB_D,
	LOONGARCH_INS_XVFSUB_S,
	LOONGARCH_INS_XVFTINTH_L_S,
	LOONGARCH_INS_XVFTINTL_L_S,
	LOONGARCH_INS_XVFTINTRMH_L_S,
	LOONGARCH_INS_XVFTINTRML_L_S,
	LOONGARCH_INS_XVFTINTRM_L_D,
	LOONGARCH_INS_XVFTINTRM_W_D,
	LOONGARCH_INS_XVFTINTRM_W_S,
	LOONGARCH_INS_XVFTINTRNEH_L_S,
	LOONGARCH_INS_XVFTINTRNEL_L_S,
	LOONGARCH_INS_XVFTINTRNE_L_D,
	LOONGARCH_INS_XVFTINTRNE_W_D,
	LOONGARCH_INS_XVFTINTRNE_W_S,
	LOONGARCH_INS_XVFTINTRPH_L_S,
	LOONGARCH_INS_XVFTINTRPL_L_S,
	LOONGARCH_INS_XVFTINTRP_L_D,
	LOONGARCH_INS_XVFTINTRP_W_D,
	LOONGARCH_INS_XVFTINTRP_W_S,
	LOONGARCH_INS_XVFTINTRZH_L_S,
	LOONGARCH_INS_XVFTINTRZL_L_S,
	LOONGARCH_INS_XVFTINTRZ_LU_D,
	LOONGARCH_INS_XVFTINTRZ_L_D,
	LOONGARCH_INS_XVFTINTRZ_WU_S,
	LOONGARCH_INS_XVFTINTRZ_W_D,
	LOONGARCH_INS_XVFTINTRZ_W_S,
	LOONGARCH_INS_XVFTINT_LU_D,
	LOONGARCH_INS_XVFTINT_L_D,
	LOONGARCH_INS_XVFTINT_WU_S,
	LOONGARCH_INS_XVFTINT_W_D,
	LOONGARCH_INS_XVFTINT_W_S,
	LOONGARCH_INS_XVHADDW_DU_WU,
	LOONGARCH_INS_XVHADDW_D_W,
	LOONGARCH_INS_XVHADDW_HU_BU,
	LOONGARCH_INS_XVHADDW_H_B,
	LOONGARCH_INS_XVHADDW_QU_DU,
	LOONGARCH_INS_XVHADDW_Q_D,
	LOONGARCH_INS_XVHADDW_WU_HU,
	LOONGARCH_INS_XVHADDW_W_H,
	LOONGARCH_INS_XVHSELI_D,
	LOONGARCH_INS_XVHSUBW_DU_WU,
	LOONGARCH_INS_XVHSUBW_D_W,
	LOONGARCH_INS_XVHSUBW_HU_BU,
	LOONGARCH_INS_XVHSUBW_H_B,
	LOONGARCH_INS_XVHSUBW_QU_DU,
	LOONGARCH_INS_XVHSUBW_Q_D,
	LOONGARCH_INS_XVHSUBW_WU_HU,
	LOONGARCH_INS_XVHSUBW_W_H,
	LOONGARCH_INS_XVILVH_B,
	LOONGARCH_INS_XVILVH_D,
	LOONGARCH_INS_XVILVH_H,
	LOONGARCH_INS_XVILVH_W,
	LOONGARCH_INS_XVILVL_B,
	LOONGARCH_INS_XVILVL_D,
	LOONGARCH_INS_XVILVL_H,
	LOONGARCH_INS_XVILVL_W,
	LOONGARCH_INS_XVINSGR2VR_D,
	LOONGARCH_INS_XVINSGR2VR_W,
	LOONGARCH_INS_XVINSVE0_D,
	LOONGARCH_INS_XVINSVE0_W,
	LOONGARCH_INS_XVLD,
	LOONGARCH_INS_XVLDI,
	LOONGARCH_INS_XVLDREPL_B,
	LOONGARCH_INS_XVLDREPL_D,
	LOONGARCH_INS_XVLDREPL_H,
	LOONGARCH_INS_XVLDREPL_W,
	LOONGARCH_INS_XVLDX,
	LOONGARCH_INS_XVMADDWEV_D_W,
	LOONGARCH_INS_XVMADDWEV_D_WU,
	LOONGARCH_INS_XVMADDWEV_D_WU_W,
	LOONGARCH_INS_XVMADDWEV_H_B,
	LOONGARCH_INS_XVMADDWEV_H_BU,
	LOONGARCH_INS_XVMADDWEV_H_BU_B,
	LOONGARCH_INS_XVMADDWEV_Q_D,
	LOONGARCH_INS_XVMADDWEV_Q_DU,
	LOONGARCH_INS_XVMADDWEV_Q_DU_D,
	LOONGARCH_INS_XVMADDWEV_W_H,
	LOONGARCH_INS_XVMADDWEV_W_HU,
	LOONGARCH_INS_XVMADDWEV_W_HU_H,
	LOONGARCH_INS_XVMADDWOD_D_W,
	LOONGARCH_INS_XVMADDWOD_D_WU,
	LOONGARCH_INS_XVMADDWOD_D_WU_W,
	LOONGARCH_INS_XVMADDWOD_H_B,
	LOONGARCH_INS_XVMADDWOD_H_BU,
	LOONGARCH_INS_XVMADDWOD_H_BU_B,
	LOONGARCH_INS_XVMADDWOD_Q_D,
	LOONGARCH_INS_XVMADDWOD_Q_DU,
	LOONGARCH_INS_XVMADDWOD_Q_DU_D,
	LOONGARCH_INS_XVMADDWOD_W_H,
	LOONGARCH_INS_XVMADDWOD_W_HU,
	LOONGARCH_INS_XVMADDWOD_W_HU_H,
	LOONGARCH_INS_XVMADD_B,
	LOONGARCH_INS_XVMADD_D,
	LOONGARCH_INS_XVMADD_H,
	LOONGARCH_INS_XVMADD_W,
	LOONGARCH_INS_XVMAXI_B,
	LOONGARCH_INS_XVMAXI_BU,
	LOONGARCH_INS_XVMAXI_D,
	LOONGARCH_INS_XVMAXI_DU,
	LOONGARCH_INS_XVMAXI_H,
	LOONGARCH_INS_XVMAXI_HU,
	LOONGARCH_INS_XVMAXI_W,
	LOONGARCH_INS_XVMAXI_WU,
	LOONGARCH_INS_XVMAX_B,
	LOONGARCH_INS_XVMAX_BU,
	LOONGARCH_INS_XVMAX_D,
	LOONGARCH_INS_XVMAX_DU,
	LOONGARCH_INS_XVMAX_H,
	LOONGARCH_INS_XVMAX_HU,
	LOONGARCH_INS_XVMAX_W,
	LOONGARCH_INS_XVMAX_WU,
	LOONGARCH_INS_XVMINI_B,
	LOONGARCH_INS_XVMINI_BU,
	LOONGARCH_INS_XVMINI_D,
	LOONGARCH_INS_XVMINI_DU,
	LOONGARCH_INS_XVMINI_H,
	LOONGARCH_INS_XVMINI_HU,
	LOONGARCH_INS_XVMINI_W,
	LOONGARCH_INS_XVMINI_WU,
	LOONGARCH_INS_XVMIN_B,
	LOONGARCH_INS_XVMIN_BU,
	LOONGARCH_INS_XVMIN_D,
	LOONGARCH_INS_XVMIN_DU,
	LOONGARCH_INS_XVMIN_H,
	LOONGARCH_INS_XVMIN_HU,
	LOONGARCH_INS_XVMIN_W,
	LOONGARCH_INS_XVMIN_WU,
	LOONGARCH_INS_XVMOD_B,
	LOONGARCH_INS_XVMOD_BU,
	LOONGARCH_INS_XVMOD_D,
	LOONGARCH_INS_XVMOD_DU,
	LOONGARCH_INS_XVMOD_H,
	LOONGARCH_INS_XVMOD_HU,
	LOONGARCH_INS_XVMOD_W,
	LOONGARCH_INS_XVMOD_WU,
	LOONGARCH_INS_XVMSKGEZ_B,
	LOONGARCH_INS_XVMSKLTZ_B,
	LOONGARCH_INS_XVMSKLTZ_D,
	LOONGARCH_INS_XVMSKLTZ_H,
	LOONGARCH_INS_XVMSKLTZ_W,
	LOONGARCH_INS_XVMSKNZ_B,
	LOONGARCH_INS_XVMSUB_B,
	LOONGARCH_INS_XVMSUB_D,
	LOONGARCH_INS_XVMSUB_H,
	LOONGARCH_INS_XVMSUB_W,
	LOONGARCH_INS_XVMUH_B,
	LOONGARCH_INS_XVMUH_BU,
	LOONGARCH_INS_XVMUH_D,
	LOONGARCH_INS_XVMUH_DU,
	LOONGARCH_INS_XVMUH_H,
	LOONGARCH_INS_XVMUH_HU,
	LOONGARCH_INS_XVMUH_W,
	LOONGARCH_INS_XVMUH_WU,
	LOONGARCH_INS_XVMULWEV_D_W,
	LOONGARCH_INS_XVMULWEV_D_WU,
	LOONGARCH_INS_XVMULWEV_D_WU_W,
	LOONGARCH_INS_XVMULWEV_H_B,
	LOONGARCH_INS_XVMULWEV_H_BU,
	LOONGARCH_INS_XVMULWEV_H_BU_B,
	LOONGARCH_INS_XVMULWEV_Q_D,
	LOONGARCH_INS_XVMULWEV_Q_DU,
	LOONGARCH_INS_XVMULWEV_Q_DU_D,
	LOONGARCH_INS_XVMULWEV_W_H,
	LOONGARCH_INS_XVMULWEV_W_HU,
	LOONGARCH_INS_XVMULWEV_W_HU_H,
	LOONGARCH_INS_XVMULWOD_D_W,
	LOONGARCH_INS_XVMULWOD_D_WU,
	LOONGARCH_INS_XVMULWOD_D_WU_W,
	LOONGARCH_INS_XVMULWOD_H_B,
	LOONGARCH_INS_XVMULWOD_H_BU,
	LOONGARCH_INS_XVMULWOD_H_BU_B,
	LOONGARCH_INS_XVMULWOD_Q_D,
	LOONGARCH_INS_XVMULWOD_Q_DU,
	LOONGARCH_INS_XVMULWOD_Q_DU_D,
	LOONGARCH_INS_XVMULWOD_W_H,
	LOONGARCH_INS_XVMULWOD_W_HU,
	LOONGARCH_INS_XVMULWOD_W_HU_H,
	LOONGARCH_INS_XVMUL_B,
	LOONGARCH_INS_XVMUL_D,
	LOONGARCH_INS_XVMUL_H,
	LOONGARCH_INS_XVMUL_W,
	LOONGARCH_INS_XVNEG_B,
	LOONGARCH_INS_XVNEG_D,
	LOONGARCH_INS_XVNEG_H,
	LOONGARCH_INS_XVNEG_W,
	LOONGARCH_INS_XVNORI_B,
	LOONGARCH_INS_XVNOR_V,
	LOONGARCH_INS_XVORI_B,
	LOONGARCH_INS_XVORN_V,
	LOONGARCH_INS_XVOR_V,
	LOONGARCH_INS_XVPACKEV_B,
	LOONGARCH_INS_XVPACKEV_D,
	LOONGARCH_INS_XVPACKEV_H,
	LOONGARCH_INS_XVPACKEV_W,
	LOONGARCH_INS_XVPACKOD_B,
	LOONGARCH_INS_XVPACKOD_D,
	LOONGARCH_INS_XVPACKOD_H,
	LOONGARCH_INS_XVPACKOD_W,
	LOONGARCH_INS_XVPCNT_B,
	LOONGARCH_INS_XVPCNT_D,
	LOONGARCH_INS_XVPCNT_H,
	LOONGARCH_INS_XVPCNT_W,
	LOONGARCH_INS_XVPERMI_D,
	LOONGARCH_INS_XVPERMI_Q,
	LOONGARCH_INS_XVPERMI_W,
	LOONGARCH_INS_XVPERM_W,
	LOONGARCH_INS_XVPICKEV_B,
	LOONGARCH_INS_XVPICKEV_D,
	LOONGARCH_INS_XVPICKEV_H,
	LOONGARCH_INS_XVPICKEV_W,
	LOONGARCH_INS_XVPICKOD_B,
	LOONGARCH_INS_XVPICKOD_D,
	LOONGARCH_INS_XVPICKOD_H,
	LOONGARCH_INS_XVPICKOD_W,
	LOONGARCH_INS_XVPICKVE2GR_D,
	LOONGARCH_INS_XVPICKVE2GR_DU,
	LOONGARCH_INS_XVPICKVE2GR_W,
	LOONGARCH_INS_XVPICKVE2GR_WU,
	LOONGARCH_INS_XVPICKVE_D,
	LOONGARCH_INS_XVPICKVE_W,
	LOONGARCH_INS_XVREPL128VEI_B,
	LOONGARCH_INS_XVREPL128VEI_D,
	LOONGARCH_INS_XVREPL128VEI_H,
	LOONGARCH_INS_XVREPL128VEI_W,
	LOONGARCH_INS_XVREPLGR2VR_B,
	LOONGARCH_INS_XVREPLGR2VR_D,
	LOONGARCH_INS_XVREPLGR2VR_H,
	LOONGARCH_INS_XVREPLGR2VR_W,
	LOONGARCH_INS_XVREPLVE0_B,
	LOONGARCH_INS_XVREPLVE0_D,
	LOONGARCH_INS_XVREPLVE0_H,
	LOONGARCH_INS_XVREPLVE0_Q,
	LOONGARCH_INS_XVREPLVE0_W,
	LOONGARCH_INS_XVREPLVE_B,
	LOONGARCH_INS_XVREPLVE_D,
	LOONGARCH_INS_XVREPLVE_H,
	LOONGARCH_INS_XVREPLVE_W,
	LOONGARCH_INS_XVROTRI_B,
	LOONGARCH_INS_XVROTRI_D,
	LOONGARCH_INS_XVROTRI_H,
	LOONGARCH_INS_XVROTRI_W,
	LOONGARCH_INS_XVROTR_B,
	LOONGARCH_INS_XVROTR_D,
	LOONGARCH_INS_XVROTR_H,
	LOONGARCH_INS_XVROTR_W,
	LOONGARCH_INS_XVSADD_B,
	LOONGARCH_INS_XVSADD_BU,
	LOONGARCH_INS_XVSADD_D,
	LOONGARCH_INS_XVSADD_DU,
	LOONGARCH_INS_XVSADD_H,
	LOONGARCH_INS_XVSADD_HU,
	LOONGARCH_INS_XVSADD_W,
	LOONGARCH_INS_XVSADD_WU,
	LOONGARCH_INS_XVSAT_B,
	LOONGARCH_INS_XVSAT_BU,
	LOONGARCH_INS_XVSAT_D,
	LOONGARCH_INS_XVSAT_DU,
	LOONGARCH_INS_XVSAT_H,
	LOONGARCH_INS_XVSAT_HU,
	LOONGARCH_INS_XVSAT_W,
	LOONGARCH_INS_XVSAT_WU,
	LOONGARCH_INS_XVSEQI_B,
	LOONGARCH_INS_XVSEQI_D,
	LOONGARCH_INS_XVSEQI_H,
	LOONGARCH_INS_XVSEQI_W,
	LOONGARCH_INS_XVSEQ_B,
	LOONGARCH_INS_XVSEQ_D,
	LOONGARCH_INS_XVSEQ_H,
	LOONGARCH_INS_XVSEQ_W,
	LOONGARCH_INS_XVSETALLNEZ_B,
	LOONGARCH_INS_XVSETALLNEZ_D,
	LOONGARCH_INS_XVSETALLNEZ_H,
	LOONGARCH_INS_XVSETALLNEZ_W,
	LOONGARCH_INS_XVSETANYEQZ_B,
	LOONGARCH_INS_XVSETANYEQZ_D,
	LOONGARCH_INS_XVSETANYEQZ_H,
	LOONGARCH_INS_XVSETANYEQZ_W,
	LOONGARCH_INS_XVSETEQZ_V,
	LOONGARCH_INS_XVSETNEZ_V,
	LOONGARCH_INS_XVSHUF4I_B,
	LOONGARCH_INS_XVSHUF4I_D,
	LOONGARCH_INS_XVSHUF4I_H,
	LOONGARCH_INS_XVSHUF4I_W,
	LOONGARCH_INS_XVSHUF_B,
	LOONGARCH_INS_XVSHUF_D,
	LOONGARCH_INS_XVSHUF_H,
	LOONGARCH_INS_XVSHUF_W,
	LOONGARCH_INS_XVSIGNCOV_B,
	LOONGARCH_INS_XVSIGNCOV_D,
	LOONGARCH_INS_XVSIGNCOV_H,
	LOONGARCH_INS_XVSIGNCOV_W,
	LOONGARCH_INS_XVSLEI_B,
	LOONGARCH_INS_XVSLEI_BU,
	LOONGARCH_INS_XVSLEI_D,
	LOONGARCH_INS_XVSLEI_DU,
	LOONGARCH_INS_XVSLEI_H,
	LOONGARCH_INS_XVSLEI_HU,
	LOONGARCH_INS_XVSLEI_W,
	LOONGARCH_INS_XVSLEI_WU,
	LOONGARCH_INS_XVSLE_B,
	LOONGARCH_INS_XVSLE_BU,
	LOONGARCH_INS_XVSLE_D,
	LOONGARCH_INS_XVSLE_DU,
	LOONGARCH_INS_XVSLE_H,
	LOONGARCH_INS_XVSLE_HU,
	LOONGARCH_INS_XVSLE_W,
	LOONGARCH_INS_XVSLE_WU,
	LOONGARCH_INS_XVSLLI_B,
	LOONGARCH_INS_XVSLLI_D,
	LOONGARCH_INS_XVSLLI_H,
	LOONGARCH_INS_XVSLLI_W,
	LOONGARCH_INS_XVSLLWIL_DU_WU,
	LOONGARCH_INS_XVSLLWIL_D_W,
	LOONGARCH_INS_XVSLLWIL_HU_BU,
	LOONGARCH_INS_XVSLLWIL_H_B,
	LOONGARCH_INS_XVSLLWIL_WU_HU,
	LOONGARCH_INS_XVSLLWIL_W_H,
	LOONGARCH_INS_XVSLL_B,
	LOONGARCH_INS_XVSLL_D,
	LOONGARCH_INS_XVSLL_H,
	LOONGARCH_INS_XVSLL_W,
	LOONGARCH_INS_XVSLTI_B,
	LOONGARCH_INS_XVSLTI_BU,
	LOONGARCH_INS_XVSLTI_D,
	LOONGARCH_INS_XVSLTI_DU,
	LOONGARCH_INS_XVSLTI_H,
	LOONGARCH_INS_XVSLTI_HU,
	LOONGARCH_INS_XVSLTI_W,
	LOONGARCH_INS_XVSLTI_WU,
	LOONGARCH_INS_XVSLT_B,
	LOONGARCH_INS_XVSLT_BU,
	LOONGARCH_INS_XVSLT_D,
	LOONGARCH_INS_XVSLT_DU,
	LOONGARCH_INS_XVSLT_H,
	LOONGARCH_INS_XVSLT_HU,
	LOONGARCH_INS_XVSLT_W,
	LOONGARCH_INS_XVSLT_WU,
	LOONGARCH_INS_XVSRAI_B,
	LOONGARCH_INS_XVSRAI_D,
	LOONGARCH_INS_XVSRAI_H,
	LOONGARCH_INS_XVSRAI_W,
	LOONGARCH_INS_XVSRANI_B_H,
	LOONGARCH_INS_XVSRANI_D_Q,
	LOONGARCH_INS_XVSRANI_H_W,
	LOONGARCH_INS_XVSRANI_W_D,
	LOONGARCH_INS_XVSRAN_B_H,
	LOONGARCH_INS_XVSRAN_H_W,
	LOONGARCH_INS_XVSRAN_W_D,
	LOONGARCH_INS_XVSRARI_B,
	LOONGARCH_INS_XVSRARI_D,
	LOONGARCH_INS_XVSRARI_H,
	LOONGARCH_INS_XVSRARI_W,
	LOONGARCH_INS_XVSRARNI_B_H,
	LOONGARCH_INS_XVSRARNI_D_Q,
	LOONGARCH_INS_XVSRARNI_H_W,
	LOONGARCH_INS_XVSRARNI_W_D,
	LOONGARCH_INS_XVSRARN_B_H,
	LOONGARCH_INS_XVSRARN_H_W,
	LOONGARCH_INS_XVSRARN_W_D,
	LOONGARCH_INS_XVSRAR_B,
	LOONGARCH_INS_XVSRAR_D,
	LOONGARCH_INS_XVSRAR_H,
	LOONGARCH_INS_XVSRAR_W,
	LOONGARCH_INS_XVSRA_B,
	LOONGARCH_INS_XVSRA_D,
	LOONGARCH_INS_XVSRA_H,
	LOONGARCH_INS_XVSRA_W,
	LOONGARCH_INS_XVSRLI_B,
	LOONGARCH_INS_XVSRLI_D,
	LOONGARCH_INS_XVSRLI_H,
	LOONGARCH_INS_XVSRLI_W,
	LOONGARCH_INS_XVSRLNI_B_H,
	LOONGARCH_INS_XVSRLNI_D_Q,
	LOONGARCH_INS_XVSRLNI_H_W,
	LOONGARCH_INS_XVSRLNI_W_D,
	LOONGARCH_INS_XVSRLN_B_H,
	LOONGARCH_INS_XVSRLN_H_W,
	LOONGARCH_INS_XVSRLN_W_D,
	LOONGARCH_INS_XVSRLRI_B,
	LOONGARCH_INS_XVSRLRI_D,
	LOONGARCH_INS_XVSRLRI_H,
	LOONGARCH_INS_XVSRLRI_W,
	LOONGARCH_INS_XVSRLRNI_B_H,
	LOONGARCH_INS_XVSRLRNI_D_Q,
	LOONGARCH_INS_XVSRLRNI_H_W,
	LOONGARCH_INS_XVSRLRNI_W_D,
	LOONGARCH_INS_XVSRLRN_B_H,
	LOONGARCH_INS_XVSRLRN_H_W,
	LOONGARCH_INS_XVSRLRN_W_D,
	LOONGARCH_INS_XVSRLR_B,
	LOONGARCH_INS_XVSRLR_D,
	LOONGARCH_INS_XVSRLR_H,
	LOONGARCH_INS_XVSRLR_W,
	LOONGARCH_INS_XVSRL_B,
	LOONGARCH_INS_XVSRL_D,
	LOONGARCH_INS_XVSRL_H,
	LOONGARCH_INS_XVSRL_W,
	LOONGARCH_INS_XVSSRANI_BU_H,
	LOONGARCH_INS_XVSSRANI_B_H,
	LOONGARCH_INS_XVSSRANI_DU_Q,
	LOONGARCH_INS_XVSSRANI_D_Q,
	LOONGARCH_INS_XVSSRANI_HU_W,
	LOONGARCH_INS_XVSSRANI_H_W,
	LOONGARCH_INS_XVSSRANI_WU_D,
	LOONGARCH_INS_XVSSRANI_W_D,
	LOONGARCH_INS_XVSSRAN_BU_H,
	LOONGARCH_INS_XVSSRAN_B_H,
	LOONGARCH_INS_XVSSRAN_HU_W,
	LOONGARCH_INS_XVSSRAN_H_W,
	LOONGARCH_INS_XVSSRAN_WU_D,
	LOONGARCH_INS_XVSSRAN_W_D,
	LOONGARCH_INS_XVSSRARNI_BU_H,
	LOONGARCH_INS_XVSSRARNI_B_H,
	LOONGARCH_INS_XVSSRARNI_DU_Q,
	LOONGARCH_INS_XVSSRARNI_D_Q,
	LOONGARCH_INS_XVSSRARNI_HU_W,
	LOONGARCH_INS_XVSSRARNI_H_W,
	LOONGARCH_INS_XVSSRARNI_WU_D,
	LOONGARCH_INS_XVSSRARNI_W_D,
	LOONGARCH_INS_XVSSRARN_BU_H,
	LOONGARCH_INS_XVSSRARN_B_H,
	LOONGARCH_INS_XVSSRARN_HU_W,
	LOONGARCH_INS_XVSSRARN_H_W,
	LOONGARCH_INS_XVSSRARN_WU_D,
	LOONGARCH_INS_XVSSRARN_W_D,
	LOONGARCH_INS_XVSSRLNI_BU_H,
	LOONGARCH_INS_XVSSRLNI_B_H,
	LOONGARCH_INS_XVSSRLNI_DU_Q,
	LOONGARCH_INS_XVSSRLNI_D_Q,
	LOONGARCH_INS_XVSSRLNI_HU_W,
	LOONGARCH_INS_XVSSRLNI_H_W,
	LOONGARCH_INS_XVSSRLNI_WU_D,
	LOONGARCH_INS_XVSSRLNI_W_D,
	LOONGARCH_INS_XVSSRLN_BU_H,
	LOONGARCH_INS_XVSSRLN_B_H,
	LOONGARCH_INS_XVSSRLN_HU_W,
	LOONGARCH_INS_XVSSRLN_H_W,
	LOONGARCH_INS_XVSSRLN_WU_D,
	LOONGARCH_INS_XVSSRLN_W_D,
	LOONGARCH_INS_XVSSRLRNI_BU_H,
	LOONGARCH_INS_XVSSRLRNI_B_H,
	LOONGARCH_INS_XVSSRLRNI_DU_Q,
	LOONGARCH_INS_XVSSRLRNI_D_Q,
	LOONGARCH_INS_XVSSRLRNI_HU_W,
	LOONGARCH_INS_XVSSRLRNI_H_W,
	LOONGARCH_INS_XVSSRLRNI_WU_D,
	LOONGARCH_INS_XVSSRLRNI_W_D,
	LOONGARCH_INS_XVSSRLRN_BU_H,
	LOONGARCH_INS_XVSSRLRN_B_H,
	LOONGARCH_INS_XVSSRLRN_HU_W,
	LOONGARCH_INS_XVSSRLRN_H_W,
	LOONGARCH_INS_XVSSRLRN_WU_D,
	LOONGARCH_INS_XVSSRLRN_W_D,
	LOONGARCH_INS_XVSSUB_B,
	LOONGARCH_INS_XVSSUB_BU,
	LOONGARCH_INS_XVSSUB_D,
	LOONGARCH_INS_XVSSUB_DU,
	LOONGARCH_INS_XVSSUB_H,
	LOONGARCH_INS_XVSSUB_HU,
	LOONGARCH_INS_XVSSUB_W,
	LOONGARCH_INS_XVSSUB_WU,
	LOONGARCH_INS_XVST,
	LOONGARCH_INS_XVSTELM_B,
	LOONGARCH_INS_XVSTELM_D,
	LOONGARCH_INS_XVSTELM_H,
	LOONGARCH_INS_XVSTELM_W,
	LOONGARCH_INS_XVSTX,
	LOONGARCH_INS_XVSUBI_BU,
	LOONGARCH_INS_XVSUBI_DU,
	LOONGARCH_INS_XVSUBI_HU,
	LOONGARCH_INS_XVSUBI_WU,
	LOONGARCH_INS_XVSUBWEV_D_W,
	LOONGARCH_INS_XVSUBWEV_D_WU,
	LOONGARCH_INS_XVSUBWEV_H_B,
	LOONGARCH_INS_XVSUBWEV_H_BU,
	LOONGARCH_INS_XVSUBWEV_Q_D,
	LOONGARCH_INS_XVSUBWEV_Q_DU,
	LOONGARCH_INS_XVSUBWEV_W_H,
	LOONGARCH_INS_XVSUBWEV_W_HU,
	LOONGARCH_INS_XVSUBWOD_D_W,
	LOONGARCH_INS_XVSUBWOD_D_WU,
	LOONGARCH_INS_XVSUBWOD_H_B,
	LOONGARCH_INS_XVSUBWOD_H_BU,
	LOONGARCH_INS_XVSUBWOD_Q_D,
	LOONGARCH_INS_XVSUBWOD_Q_DU,
	LOONGARCH_INS_XVSUBWOD_W_H,
	LOONGARCH_INS_XVSUBWOD_W_HU,
	LOONGARCH_INS_XVSUB_B,
	LOONGARCH_INS_XVSUB_D,
	LOONGARCH_INS_XVSUB_H,
	LOONGARCH_INS_XVSUB_Q,
	LOONGARCH_INS_XVSUB_W,
	LOONGARCH_INS_XVXORI_B,
	LOONGARCH_INS_XVXOR_V,

	// clang-format on
	// generated content <LoongArchGenCSInsnEnum.inc> end

	LOONGARCH_INS_ENDING,

	LOONGARCH_INS_ALIAS_BEGIN,
	// generated content <LoongArchGenCSAliasEnum.inc> begin
	// clang-format off

	LOONGARCH_INS_ALIAS_LA, // Real instr.: LOONGARCH_PseudoLA_GOT
	LOONGARCH_INS_ALIAS_LA_GLOBAL, // Real instr.: LOONGARCH_PseudoLA_GOT
	LOONGARCH_INS_ALIAS_LA_LOCAL, // Real instr.: LOONGARCH_PseudoLA_PCREL
	LOONGARCH_INS_ALIAS_NOP, // Real instr.: LOONGARCH_ANDI
	LOONGARCH_INS_ALIAS_MOVE, // Real instr.: LOONGARCH_OR
	LOONGARCH_INS_ALIAS_RET, // Real instr.: LOONGARCH_JIRL
	LOONGARCH_INS_ALIAS_JR, // Real instr.: LOONGARCH_JIRL

	// clang-format on
	// generated content <LoongArchGenCSInsnEnum.inc> end
	LOONGARCH_INS_ALIAS_END,

} loongarch_insn;

//> Group of LOONGARCH instructions
typedef enum loongarch_insn_group {
	LOONGARCH_GRP_INVALID = 0, ///< = CS_GRP_INVALID

	/// Generic groups
	/// all jump instructions (conditional+direct+indirect jumps)
	LOONGARCH_GRP_JUMP, ///< = CS_GRP_JUMP
	/// all call instructions
	LOONGARCH_GRP_CALL, ///< = CS_GRP_CALL
	/// all return instructions
	LOONGARCH_GRP_RET, ///< = CS_GRP_RET
	/// all interrupt instructions (int+syscall)
	LOONGARCH_GRP_INT, ///< = CS_GRP_INT
	/// all interrupt return instructions
	LOONGARCH_GRP_IRET, ///< = CS_GRP_IRET
	/// all privileged instructions
	LOONGARCH_GRP_PRIVILEGE, ///< = CS_GRP_PRIVILEGE
	/// all relative branching instructions
	LOONGARCH_GRP_BRANCH_RELATIVE, ///< = CS_GRP_BRANCH_RELATIVE

	// Architecture-specific groups
	// generated content <LoongArchGenCSFeatureEnum.inc> begin
	// clang-format off

	LOONGARCH_FEATURE_ISLA64 = 128,
	LOONGARCH_FEATURE_ISLA32,
	LOONGARCH_FEATURE_HASLAGLOBALWITHPCREL,
	LOONGARCH_FEATURE_HASLAGLOBALWITHABS,
	LOONGARCH_FEATURE_HASLALOCALWITHABS,

	// clang-format on
	// generated content <LoongArchGenCSFeatureEnum.inc> end

	LOONGARCH_GRP_ENDING,
} loongarch_insn_group;

#ifdef __cplusplus
}
#endif

#endif
