"use strict";
// noinspection JSBitwiseOperatorUsage
Object.defineProperty(exports, "__esModule", { value: true });
exports.areTypesMutuallyAssignable = exports.getTypeProperties = exports.getSymbolType = exports.getElementTypeByOffsets = exports.getResolvedSignature = exports.getElementType = void 0;
const utils_1 = require("./utils");
let lastIdeTypeCheckerId = 0;
function getElementType(ts, ideProjectId, program, sourceFile, range, typeRequestKind, forceReturnType, cancellationToken, reverseMapper) {
    let startOffset = ts.getPositionOfLineAndCharacter(sourceFile, range.start.line, range.start.character);
    let endOffset = ts.getPositionOfLineAndCharacter(sourceFile, range.end.line, range.end.character);
    return getElementTypeByOffsets(ts, ideProjectId, program, sourceFile, startOffset, endOffset, typeRequestKind, forceReturnType, cancellationToken, reverseMapper);
}
exports.getElementType = getElementType;
function getResolvedSignature(ts, ideProjectId, program, sourceFile, range, cancellationToken, reverseMapper) {
    let startOffset = ts.getPositionOfLineAndCharacter(sourceFile, range.start.line, range.start.character);
    let endOffset = ts.getPositionOfLineAndCharacter(sourceFile, range.end.line, range.end.character);
    const typeChecker = program.getTypeChecker();
    // Find the node at the given position
    let node = ts.getTokenAtPosition(sourceFile, startOffset);
    while (node && node.getEnd() < endOffset) {
        node = node.parent;
    }
    if (!node || node === sourceFile) {
        return undefined;
    }
    // Find the call expression
    while (node && !ts.isCallLikeExpression(node)) {
        node = node.parent;
        if (!node || node === sourceFile) {
            return undefined;
        }
    }
    // Get the resolved signature
    const signature = typeChecker.getResolvedSignature(node);
    if (!signature) {
        return undefined;
    }
    // Return the signature information
    let ctx = new ConvertContext(ts, typeChecker, reverseMapper, cancellationToken);
    if (!typeChecker.webStormCacheInfo) {
        typeChecker.webStormCacheInfo = {
            ideTypeCheckerId: ++lastIdeTypeCheckerId,
            ideProjectId: ideProjectId,
            requestedTypeIds: new Set(),
            seenTypeIds: new Map(),
            seenSymbolIds: new Map()
        };
    }
    else if (typeChecker.webStormCacheInfo.ideProjectId != ideProjectId) {
        console.error(`getElementTypeByOffsets - wrong ideProjectId. Cached: ${typeChecker.webStormCacheInfo.ideProjectId}, requested ${ideProjectId} `);
        return undefined;
    }
    const cacheInfo = typeChecker.webStormCacheInfo;
    const prepared = convertSignature(signature, ctx);
    prepared.ideTypeCheckerId = cacheInfo.ideTypeCheckerId;
    prepared.ideProjectId = cacheInfo.ideProjectId;
    prepared.ideObjectType = "SignatureObject";
    return {
        responseRequired: true,
        response: prepared
    };
}
exports.getResolvedSignature = getResolvedSignature;
function getElementTypeByOffsets(ts, ideProjectId, program, sourceFile, startOffset, endOffset, typeRequestKind, forceReturnType, cancellationToken, reverseMapper) {
    var _a, _b;
    let node = ts.getTokenAtPosition(sourceFile, startOffset);
    while (node && node.getEnd() < endOffset) {
        node = node.parent;
    }
    if (!node || node === sourceFile) {
        return undefined;
    }
    const contextFlags = typeRequestKindToContextFlags(typeRequestKind);
    const isContextual = contextFlags >= 0;
    if ((isContextual
        ? !ts.isExpression(node)
        : ts.isStringLiteral(node) || ts.isNumericLiteral(node)) &&
        node.pos === ((_a = node.parent) === null || _a === void 0 ? void 0 : _a.pos) && node.end === ((_b = node.parent) === null || _b === void 0 ? void 0 : _b.end)) {
        node = node.parent;
    }
    const typeChecker = program.getTypeChecker();
    if (!typeChecker.webStormCacheInfo) {
        typeChecker.webStormCacheInfo = {
            ideTypeCheckerId: ++lastIdeTypeCheckerId,
            ideProjectId: ideProjectId,
            requestedTypeIds: new Set(),
            seenTypeIds: new Map(),
            seenSymbolIds: new Map()
        };
    }
    else if (typeChecker.webStormCacheInfo.ideProjectId != ideProjectId) {
        console.error(`getElementTypeByOffsets - wrong ideProjectId. Cached: ${typeChecker.webStormCacheInfo.ideProjectId}, requested ${ideProjectId} `);
        return undefined;
    }
    const cacheInfo = typeChecker.webStormCacheInfo;
    let type = contextFlags >= 0 ? typeChecker.getContextualType(node, contextFlags) : typeChecker.getTypeAtLocation(node);
    if (!type && isContextual && ts.isBinaryExpression(node.parent)) {
        // from getContextualType in services/completions.ts
        const { left, operatorToken, right } = node.parent;
        if (ts.isEqualityOperatorKind(operatorToken.kind)) {
            type = typeChecker.getTypeAtLocation(node === right ? left : right);
        }
    }
    if (!type)
        return undefined;
    let prepared;
    if (forceReturnType || type.id == null || !cacheInfo.requestedTypeIds.has(type.id)) {
        const ctx = new ConvertContext(ts, typeChecker, reverseMapper, cancellationToken);
        prepared = convertType(type, ctx);
        prepared.ideTypeCheckerId = cacheInfo.ideTypeCheckerId;
        prepared.ideProjectId = cacheInfo.ideProjectId;
        if (type.id != null) {
            cacheInfo.requestedTypeIds.add(type.id);
        }
    }
    else {
        prepared = {
            id: type.id,
            ideTypeCheckerId: cacheInfo.ideTypeCheckerId,
            ideProjectId: cacheInfo.ideProjectId,
            ideObjectType: "TypeObject",
        };
    }
    return { responseRequired: true, response: prepared };
}
exports.getElementTypeByOffsets = getElementTypeByOffsets;
function typeRequestKindToContextFlags(typeRequestKind) {
    switch (typeRequestKind) {
        case "Default":
            return -1;
        case "Contextual":
            return 0;
        case "ContextualCompletions":
            return 4;
    }
    throw Error("Unexpected typeRequestKind " + typeRequestKind);
}
class ConvertContext {
    constructor(ts, checker, reverseMapper, cancellationToken) {
        this.ts = ts;
        this.checker = checker;
        this.reverseMapper = reverseMapper;
        this.cancellationToken = cancellationToken;
        this.lastCancelCheck = 0;
        this.nextId = 0;
        this.createdObjectsIdeIds = new Map();
    }
    checkCancelled() {
        // Cancellation check might be costly, check only every 1ms
        if (this.cancellationToken && new Date().getTime() > this.lastCancelCheck) {
            this.lastCancelCheck = new Date().getTime();
            if (this.cancellationToken.isCancellationRequested()) {
                (0, utils_1.throwIdeError)("OperationCancelledException");
            }
        }
        return undefined;
    }
    getIdeObjectId(obj) {
        return this.createdObjectsIdeIds.get(obj);
    }
    registerIdeObject(obj) {
        this.checkCancelled();
        const id = this.nextId++;
        this.createdObjectsIdeIds.set(obj, id);
        return id;
    }
}
function convertType(type, ctx) {
    return findReferenceOrConvert(type, ideObjectId => {
        var _a, _b, _c, _d;
        const tscType = {
            ideObjectId,
            ideObjectType: "TypeObject",
            flags: type.flags,
        };
        // First, we map only types -- to get them on shallower levels, allowing them more room for nested items.
        // Then we map everything else -- these entries will likely get references to already mapped types.
        tscType.aliasTypeArguments = (_b = (_a = type.aliasTypeArguments) === null || _a === void 0 ? void 0 : _a.map(t => convertType(t, ctx))) === null || _b === void 0 ? void 0 : _b.filter(isNotNull);
        if (type.flags & ctx.ts.TypeFlags.Object) {
            if (type.target)
                // FIXME internal API
                // FIXME Whole target is not needed, only target.objectFlags needed
                tscType.target = convertType(type.target, ctx);
            if (type.objectFlags & ctx.ts.ObjectFlags.Reference)
                tscType.resolvedTypeArguments = ctx.checker.getTypeArguments(type)
                    // TS 4 returns 'this'-type as one of class type arguments, which we don't need
                    // FIXME internal API
                    .filter(t => !t.isThisType)
                    .map((t) => convertType(t, ctx))
                    .filter(isNotNull);
        }
        if (type.flags & (ctx.ts.TypeFlags.UnionOrIntersection | ctx.ts.TypeFlags.TemplateLiteral))
            tscType.types = type.types
                .map(t => convertType(t, ctx))
                .filter(isNotNull);
        if (type.flags & ctx.ts.TypeFlags.Literal && type.freshType)
            tscType.freshType = convertType(type.freshType, ctx);
        if (type.flags & ctx.ts.TypeFlags.TypeParameter) {
            const constraint = ctx.checker.getBaseConstraintOfType(type);
            if (constraint)
                tscType.constraint = convertType(constraint, ctx);
        }
        if (type.flags & ctx.ts.TypeFlags.Index)
            tscType.type = convertType(type.type, ctx);
        if (type.flags & ctx.ts.TypeFlags.IndexedAccess) {
            tscType.objectType = convertType(type.objectType, ctx);
            tscType.indexType = convertType(type.indexType, ctx);
        }
        if (type.flags & ctx.ts.TypeFlags.Conditional) {
            tscType.checkType = convertType(type.checkType, ctx);
            tscType.extendsType = convertType(type.extendsType, ctx);
        }
        if (type.flags & ctx.ts.TypeFlags.Substitution) {
            tscType.baseType = convertType(type.baseType, ctx);
        }
        // Now map everything else but types
        if (type.symbol)
            tscType.symbol = convertSymbol(type.symbol, ctx);
        if (type.aliasSymbol)
            tscType.aliasSymbol = convertSymbol(type.aliasSymbol, ctx);
        if (type.flags & ctx.ts.TypeFlags.Object) {
            tscType.objectFlags = type.objectFlags;
        }
        if (type.flags & ctx.ts.TypeFlags.Literal) {
            if (type.flags & ctx.ts.TypeFlags.BigIntLiteral)
                tscType.value = convertPseudoBigInt(type.value, ctx);
            else
                tscType.value = type.value;
        }
        if (type.flags & ctx.ts.TypeFlags.EnumLiteral)
            // FIXME 'nameType' is just some random name from generated Kotlin TypeObjectProperty.
            // FIXME This field should have its own name.
            tscType.nameType = getEnumQualifiedName(type, ctx);
        if (type.flags & ctx.ts.TypeFlags.TemplateLiteral)
            tscType.texts = type.texts;
        // FIXME internal API
        if (type.flags & ctx.ts.TypeFlags.TypeParameter && type.isThisType)
            tscType.isThisType = true;
        // FIXME internal API
        if (typeof type.intrinsicName === 'string')
            tscType.intrinsicName = type.intrinsicName;
        if (type.elementFlags)
            tscType.elementFlags = type.elementFlags;
        let typeId = type.id;
        tscType.id = typeId;
        if (typeId) {
            (_d = (_c = ctx.checker.webStormCacheInfo) === null || _c === void 0 ? void 0 : _c.seenTypeIds) === null || _d === void 0 ? void 0 : _d.set(typeId, type);
        }
        return tscType;
    }, ctx);
}
function getEnumQualifiedName(type, ctx) {
    let qName = '';
    // FIXME internal API. Maybe use Node.parent instead?
    let current = type.symbol.parent;
    while (current && !(current.valueDeclaration && ctx.ts.isSourceFile(current.valueDeclaration))) {
        qName = current.escapedName + (qName ? '.' + qName : '');
        current = current.parent;
    }
    return qName || undefined;
}
function convertSymbol(symbol, ctx) {
    return findReferenceOrConvert(symbol, ideObjectId => {
        var _a, _b, _c, _d, _e, _f;
        const tscSymbol = {
            ideObjectId,
            ideObjectType: "SymbolObject",
            flags: symbol.flags,
            escapedName: symbol.escapedName,
        };
        tscSymbol.declarations = (_b = (_a = symbol.declarations) === null || _a === void 0 ? void 0 : _a.map(d => convertNode(d, ctx))) === null || _b === void 0 ? void 0 : _b.filter(isNotNull);
        if (symbol.valueDeclaration)
            tscSymbol.valueDeclaration = convertNode(symbol.valueDeclaration, ctx);
        // FIXME internal API
        if ((_c = symbol.links) === null || _c === void 0 ? void 0 : _c.type)
            tscSymbol.type = convertType((_d = symbol.links) === null || _d === void 0 ? void 0 : _d.type, ctx); // TS 5
        else if (symbol.type)
            tscSymbol.type = convertType(symbol.type, ctx); // TS 4
        const symbolId = ctx.ts.getSymbolId(symbol);
        (_f = (_e = ctx.checker.webStormCacheInfo) === null || _e === void 0 ? void 0 : _e.seenSymbolIds) === null || _f === void 0 ? void 0 : _f.set(symbolId, symbol);
        tscSymbol.id = symbolId;
        return tscSymbol;
    }, ctx);
}
function convertSignature(signature, ctx) {
    return findReferenceOrConvert(signature, ideObjectId => {
        var _a, _b;
        return ({
            ideObjectId,
            ideObjectType: "SignatureObject",
            declaration: signature.declaration && ((_a = signature.declaration) === null || _a === void 0 ? void 0 : _a.kind) != ctx.ts.SyntaxKind.JSDocSignature ? convertNode(signature.declaration, ctx) : undefined,
            parameters: signature.parameters
                .map(s => convertSymbol(s, ctx))
                .filter(isNotNull),
            typeParameters: (_b = signature.typeParameters) === null || _b === void 0 ? void 0 : _b.map(s => convertType(s, ctx)),
            resolvedReturnType: convertType(ctx.checker.getReturnTypeOfSignature(signature), ctx),
            flags: signature.flags,
        });
    }, ctx);
}
function convertNode(node, ctx, childReverseMapping = undefined) {
    return findReferenceOrConvert(node, ideObjectId => {
        var _a;
        if (ctx.ts.isSourceFile(node)) {
            return {
                ideObjectId,
                ideObjectType: "SourceFileObject",
                fileName: (_a = childReverseMapping === null || childReverseMapping === void 0 ? void 0 : childReverseMapping.fileName) !== null && _a !== void 0 ? _a : node.fileName,
            };
        }
        else {
            const sourceFileParent = getSourceFileParent(node, ctx);
            if (!sourceFileParent || node.pos == -1 || node.end == -1) {
                return {
                    ideObjectId,
                    ideObjectType: "NodeObject",
                    parent: sourceFileParent ? convertNode(sourceFileParent, ctx) : undefined,
                };
            }
            const reverseMapping = runReverseMapper(sourceFileParent, node, ctx);
            return {
                ideObjectId,
                ideObjectType: "NodeObject",
                range: trimRange(ctx, sourceFileParent, node, reverseMapping),
                parent: convertNode(sourceFileParent, ctx, reverseMapping),
                computedProperty: (ctx.ts.isPropertyAssignment(node) || ctx.ts.isPropertySignature(node) || ctx.ts.isPropertyDeclaration(node) || ctx.ts.isMethodSignature(node) || ctx.ts.isMethodDeclaration(node))
                    && ctx.ts.isComputedPropertyName(node.name)
                    || undefined,
            };
        }
    }, ctx);
}
function trimRange(ctx, sourceFileParent, node, reverseMapping) {
    var _a, _b;
    let reverseStartPos = (_a = reverseMapping === null || reverseMapping === void 0 ? void 0 : reverseMapping.sourceRange) === null || _a === void 0 ? void 0 : _a.start;
    let reverseEndPos = (_b = reverseMapping === null || reverseMapping === void 0 ? void 0 : reverseMapping.sourceRange) === null || _b === void 0 ? void 0 : _b.end;
    let startPos;
    let endPos;
    if (reverseStartPos && reverseEndPos) {
        try {
            startPos = ctx.ts.getPositionOfLineAndCharacter(sourceFileParent, reverseStartPos.line, reverseStartPos.character);
            endPos = ctx.ts.getPositionOfLineAndCharacter(sourceFileParent, reverseEndPos.line, reverseEndPos.character);
        }
        catch (e) {
            // If any problems with the range, don't try to trim it and return as-is
            return {
                start: reverseStartPos,
                end: reverseEndPos
            };
        }
    }
    else {
        startPos = node.pos;
        endPos = node.end;
    }
    // Trim spaces around the node
    let text = sourceFileParent.text;
    while (startPos < endPos && isWhitespace(text.charAt(startPos))) {
        startPos++;
    }
    while (endPos > startPos && isWhitespace(text.charAt(endPos - 1))) {
        endPos--;
    }
    return {
        start: ctx.ts.getLineAndCharacterOfPosition(sourceFileParent, startPos),
        end: ctx.ts.getLineAndCharacterOfPosition(sourceFileParent, endPos)
    };
}
function isWhitespace(ch) {
    return ch == ' ' || ch == '\t' || ch == '\n' || ch == '\r';
}
function getSourceFileParent(node, ctx) {
    if (ctx.ts.isSourceFile(node))
        return undefined;
    let current = node.parent;
    while (current) {
        if (ctx.ts.isSourceFile(current))
            return current;
        current = current.parent;
    }
    return undefined;
}
function runReverseMapper(sourceFileParent, node, ctx) {
    if (!ctx.reverseMapper)
        return undefined;
    let startOffset = node.pos;
    let endOffset = node.end;
    // Trim spaces around the node
    const text = sourceFileParent.text;
    while (' \t\n\r\v'.indexOf(text.charAt(startOffset)) >= 0 && startOffset < endOffset) {
        startOffset++;
    }
    while (endOffset > 0 && ' \t\n\r\v'.indexOf(text.charAt(endOffset - 1)) >= 0 && startOffset < endOffset) {
        endOffset--;
    }
    return ctx.reverseMapper(sourceFileParent, {
        start: ctx.ts.getLineAndCharacterOfPosition(sourceFileParent, startOffset),
        end: ctx.ts.getLineAndCharacterOfPosition(sourceFileParent, endOffset),
    });
}
function convertPseudoBigInt(pseudoBigInt, ctx) {
    return findReferenceOrConvert(pseudoBigInt, ideObjectId => (Object.assign({ ideObjectId }, pseudoBigInt)), ctx);
}
function convertIndexInfo(indexInfo, ctx) {
    return findReferenceOrConvert(indexInfo, ideObjectId => {
        const result = {
            ideObjectId,
            ideObjectType: "IndexInfo",
            keyType: convertType(indexInfo.keyType, ctx),
            type: convertType(indexInfo.type, ctx),
            isReadonly: indexInfo.isReadonly,
        };
        if (indexInfo.declaration) {
            result.declaration = convertNode(indexInfo.declaration, ctx);
        }
        return result;
    }, ctx);
}
function findReferenceOrConvert(sourceObj, convertTarget, ctx) {
    let ideObjectId = ctx.getIdeObjectId(sourceObj);
    if (ideObjectId) {
        return { ideObjectIdRef: ideObjectId };
    }
    ideObjectId = ctx.registerIdeObject(sourceObj);
    const newObject = convertTarget(ideObjectId);
    return newObject;
}
function isNotNull(t) {
    return t != null;
}
function getSymbolType(ts, program, symbolId, cancellationToken, reverseMapper) {
    const typeChecker = program.getTypeChecker();
    const cacheInfo = typeChecker.webStormCacheInfo;
    if (!cacheInfo) {
        return undefined;
    }
    let symbol = cacheInfo.seenSymbolIds.get(symbolId);
    if (!symbol) {
        return undefined;
    }
    const ctx = new ConvertContext(ts, typeChecker, reverseMapper, cancellationToken);
    let prepared = {};
    if (ctx.checker.getTypeOfSymbol) {
        prepared = convertType(ctx.checker.getTypeOfSymbol(symbol), ctx);
    }
    else if (symbol.valueDeclaration) {
        prepared = convertType(ctx.checker.getTypeOfSymbolAtLocation(symbol, symbol.valueDeclaration), ctx);
    }
    prepared.ideTypeCheckerId = cacheInfo.ideTypeCheckerId;
    prepared.ideProjectId = cacheInfo.ideProjectId;
    return { responseRequired: true, response: prepared };
}
exports.getSymbolType = getSymbolType;
function getTypeProperties(ts, program, typeId, cancellationToken, reverseMapper) {
    const typeChecker = program.getTypeChecker();
    const cacheInfo = typeChecker.webStormCacheInfo;
    if (!cacheInfo) {
        return undefined;
    }
    let type = cacheInfo.seenTypeIds.get(typeId);
    if (!type) {
        return undefined;
    }
    const ctx = new ConvertContext(ts, typeChecker, reverseMapper, cancellationToken);
    let prepared = convertTypeProperties(type, ctx);
    prepared.ideTypeCheckerId = cacheInfo.ideTypeCheckerId;
    prepared.ideProjectId = cacheInfo.ideProjectId;
    return { responseRequired: true, response: prepared };
}
exports.getTypeProperties = getTypeProperties;
function convertTypeProperties(type, ctx) {
    return findReferenceOrConvert(type, ideObjectId => {
        let prepared = {
            ideObjectId,
            ideObjectType: "TypeObject",
            flags: type.flags,
            objectFlags: type.objectFlags
        };
        if (type.flags & ctx.ts.TypeFlags.Object) {
            assignObjectTypeProperties(type, ctx, prepared);
        }
        if (type.flags & ctx.ts.TypeFlags.UnionOrIntersection) {
            assignUnionOrIntersectionTypeProperties(type, ctx, prepared);
        }
        if (type.flags & ctx.ts.TypeFlags.Conditional) {
            assignConditionalTypeProperties(type, ctx, prepared);
        }
        return prepared;
    }, ctx);
}
function assignObjectTypeProperties(type, ctx, tscType) {
    tscType.constructSignatures = type.getConstructSignatures()
        .map((s) => convertSignature(s, ctx))
        .filter(isNotNull);
    tscType.callSignatures = type.getCallSignatures()
        .map((s) => convertSignature(s, ctx))
        .filter(isNotNull);
    tscType.properties = type.getProperties()
        .map((p) => convertSymbol(p, ctx))
        .filter(isNotNull);
    tscType.indexInfos = ctx.checker.getIndexInfosOfType &&
        ctx.checker.getIndexInfosOfType(type)
            .map((info) => convertIndexInfo(info, ctx))
            .filter(isNotNull);
}
function assignUnionOrIntersectionTypeProperties(type, ctx, tscType) {
    tscType.resolvedProperties = ctx.checker.getPropertiesOfType(type)
        .map((p) => convertSymbol(p, ctx))
        .filter(isNotNull);
    tscType.callSignatures = ctx.checker.getSignaturesOfType(type, ctx.ts.SignatureKind.Call)
        .map((s) => convertSignature(s, ctx))
        .filter(isNotNull);
    tscType.constructSignatures = ctx.checker.getSignaturesOfType(type, ctx.ts.SignatureKind.Construct)
        .map((s) => convertSignature(s, ctx))
        .filter(isNotNull);
}
function assignConditionalTypeProperties(type, ctx, tscType) {
    ctx.checker.getPropertiesOfType(type); // In TS 4 this triggers calculation of true and false types
    if (type.resolvedTrueType)
        tscType.resolvedTrueType = convertType(type.resolvedTrueType, ctx);
    if (type.resolvedFalseType)
        tscType.resolvedFalseType = convertType(type.resolvedFalseType, ctx);
}
function areTypesMutuallyAssignable(ts, program, type1Id, type2Id, cancellationToken) {
    var _a, _b, _c, _d, _e;
    const checker = program === null || program === void 0 ? void 0 : program.getTypeChecker();
    if (!checker)
        return undefined;
    const type1 = (_b = (_a = checker.webStormCacheInfo) === null || _a === void 0 ? void 0 : _a.seenTypeIds) === null || _b === void 0 ? void 0 : _b.get(type1Id);
    if (!type1)
        return undefined;
    const type2 = (_d = (_c = checker.webStormCacheInfo) === null || _c === void 0 ? void 0 : _c.seenTypeIds) === null || _d === void 0 ? void 0 : _d.get(type2Id);
    if (!type2)
        return undefined;
    const { isTypeAssignableTo } = checker;
    if (!isTypeAssignableTo)
        return undefined;
    const ctx = new ConvertContext(ts, checker, undefined, cancellationToken);
    const areMutuallyAssignable = isTypeAssignableTo(type1, type2) && ((_e = ctx.checkCancelled()) !== null && _e !== void 0 ? _e : isTypeAssignableTo(type2, type1));
    return {
        response: { areMutuallyAssignable },
        responseRequired: true,
    };
}
exports.areTypesMutuallyAssignable = areTypesMutuallyAssignable;
//# sourceMappingURL=ide-get-element-type.js.map