package log

import (
	"bytes"
	"errors"
	"io"
	"sync/atomic"
	"testing"

	"github.com/stretchr/testify/require"
)

// mockReadCloser implements io.ReadCloser for testing.
type mockReadCloser struct {
	reader      *bytes.Reader
	closeCalled bool
	closeErr    error
}

func (m *mockReadCloser) Read(p []byte) (int, error) {
	return m.reader.Read(p)
}

func (m *mockReadCloser) Close() error {
	m.closeCalled = true
	return m.closeErr
}

func newMockReadCloser(data []byte, closeErr error) *mockReadCloser {
	return &mockReadCloser{
		reader:   bytes.NewReader(data),
		closeErr: closeErr,
	}
}

func TestBodyByteCounter_Read(t *testing.T) {
	testCases := []struct {
		name        string
		inputData   []byte
		readSize    int
		expectedErr error
	}{
		{
			name:        "read all data at once",
			inputData:   []byte("hello world"),
			readSize:    20,
			expectedErr: io.EOF,
		},
		{
			name:        "read data in chunks",
			inputData:   []byte("hello world"),
			readSize:    5,
			expectedErr: nil,
		},
		{
			name:        "empty data",
			inputData:   []byte{},
			readSize:    5,
			expectedErr: io.EOF,
		},
	}

	for _, tc := range testCases {
		t.Run(tc.name, func(t *testing.T) {
			var count atomic.Int64

			mockBody := newMockReadCloser(tc.inputData, nil)
			counter := &bodyByteCounter{
				body:  mockBody,
				count: &count,
			}

			buf := make([]byte, tc.readSize)
			n, err := counter.Read(buf)

			require.LessOrEqual(t, n, len(tc.inputData), "Read returned more bytes than input data")

			if tc.expectedErr == io.EOF {
				if err == nil {
					// Deal with this Read() behaviour:
					//
					// It may return the (non-nil) error from the same call
					// or return the error (and n == 0) from a subsequent call.
					// An instance of this general case is that a Reader returning
					// a non-zero number of bytes at the end of the input stream may
					// return either err == EOF or err == nil. The next Read should
					// return 0, EOF.
					buf2 := make([]byte, 10)
					n2, err2 := counter.Read(buf2)

					require.Zero(t, n2, "expected subsequent read to return zero bytes")
					require.ErrorIs(t, err2, io.EOF, "Expected EOF error")
				} else {
					require.ErrorIs(t, err, io.EOF, "Expected EOF error")
				}
			}

			totalRead := n

			for err == nil {
				n, err = counter.Read(buf)
				if err != nil && err != io.EOF {
					require.NoError(t, err, "Unexpected error during read")
				}

				totalRead += n
			}

			require.Equal(t, len(tc.inputData), totalRead, "Total bytes read doesn't match input length")
			require.Equal(t, int64(len(tc.inputData)), count.Load(), "Byte counter doesn't match input length")
		})
	}
}

func TestBodyByteCounter_ReadPartial(t *testing.T) {
	var count atomic.Int64

	testData := []byte("hello world")
	mockBody := newMockReadCloser(testData, nil)
	counter := &bodyByteCounter{
		body:  mockBody,
		count: &count,
	}

	buf := make([]byte, 5)
	n, err := counter.Read(buf)
	require.Equal(t, 5, n, "First read returned unexpected number of bytes")
	require.NoError(t, err, "First read returned unexpected error")
	require.Equal(t, int64(5), count.Load(), "Counter after first read is incorrect")

	buf = make([]byte, 10)
	n, err = counter.Read(buf)
	require.Equal(t, 6, n, "Second read returned unexpected number of bytes")
	require.Equal(t, int64(11), count.Load(), "Counter after second read is incorrect")

	// Deal with this Read() behaviour:
	//
	// It may return the (non-nil) error from the same call
	// or return the error (and n == 0) from a subsequent call.
	// An instance of this general case is that a Reader returning
	// a non-zero number of bytes at the end of the input stream may
	// return either err == EOF or err == nil. The next Read should
	// return 0, EOF.
	if err != io.EOF {
		buf2 := make([]byte, 10)
		n2, err2 := counter.Read(buf2)

		require.Zero(t, n2, "expected subsequent read to return zero bytes")
		require.ErrorIs(t, err2, io.EOF, "Expected EOF error")
	}
}

func TestBodyByteCounter_Close(t *testing.T) {
	testCases := []struct {
		name     string
		closeErr error
	}{
		{
			name:     "successful close",
			closeErr: nil,
		},
		{
			name:     "error on close",
			closeErr: errors.New("close error"),
		},
	}

	for _, tc := range testCases {
		t.Run(tc.name, func(t *testing.T) {
			var count atomic.Int64

			mockBody := newMockReadCloser([]byte("test"), tc.closeErr)
			counter := &bodyByteCounter{
				body:  mockBody,
				count: &count,
			}

			err := counter.Close()
			if tc.closeErr != nil {
				require.EqualError(t, err, tc.closeErr.Error(), "Close returned unexpected error")
			} else {
				require.NoError(t, err, "Close should not return an error")
			}

			require.True(t, mockBody.closeCalled, "Close was not called on the underlying body")
		})
	}
}
