package expression

import (
	"fmt"
	"os"
	"regexp"
	"strings"

	"google.golang.org/protobuf/types/known/structpb"

	"gitlab.com/gitlab-org/step-runner/pkg/internal/expression"
)

type JobInput struct {
	Key   string
	Value *structpb.Value
}

// expressionRe matches full `${{ ... }}` expressions, capturing them as a group.
// It allows optional whitespace inside the braces, and matches non-nested content.
// Breakdown:
//
//	\${{         - matches the literal opening `${{`
//	\s*          - optional whitespace after `${{`
//	[^}]*?       - lazily matches any characters except `}`, i.e. the inner content
//	\s*          - optional whitespace before the closing braces
//	}}           - matches the literal closing `}}`
var expressionRe = regexp.MustCompile(`\${{\s*[^}]*?\s*}}`)

const (
	kindLiteral = iota
	kindExpr
)

type exprPart struct {
	kind int
	text string
}

// ExpandJobInputs is used by the Runner to expand job inputs when steps isn't involved
// Instead of sending the entire `expr` argument to `expression.ExpandString`, individual expressions are extracted
// from the `expr` string and sent one at a time. This allows us to constrain features to make it easy to replace the
// Steps expression language implementation. For example, we're restricting `ExpandJobInputs` to disallow expressions within expressions.
func ExpandJobInputs(jobInputs []JobInput, expr string) (*structpb.Value, error) {
	jobVars, err := buildJobVariables(jobInputs)
	if err != nil {
		return nil, fmt.Errorf("expanding expression: %w", err)
	}

	// Initially, we'll use an object that has sensible zero-values for contexts that we don't want to expose
	// Going forward, we should error if contexts are used that we don't support
	interpolationCtx := &expression.InterpolationContext{
		Env:         map[string]string{},
		ExportFile:  os.DevNull,
		Inputs:      map[string]*structpb.Value{},
		Job:         jobVars,
		OutputFile:  os.DevNull,
		StepDir:     os.DevNull,
		StepResults: map[string]*expression.StepResultView{},
		WorkDir:     os.DevNull,
	}

	parts := extractParts(expr)

	// string literal, contains no expressions
	if len(parts) == 1 && parts[0].kind == kindLiteral {
		return structpb.NewStringValue(parts[0].text), nil
	}

	// just an expression, e.g. "${{ job.inputs.foobar }}" (i.e. not a string template)
	if len(parts) == 1 {
		value, err := expression.Expand(interpolationCtx, structpb.NewStringValue(parts[0].text))
		if err != nil {
			return nil, fmt.Errorf("expanding expression: %w", err)
		}

		return value.Value, nil
	}

	value, err := expandStringTemplate(parts, interpolationCtx)
	if err != nil {
		return nil, fmt.Errorf("expanding expression: %w", err)
	}

	return value, nil
}

// expandStringTemplate expands string templates, e.g. "hello, ${{ job.inputs.username }}"
// Expressions in a string template must evaluate to a string
//   - This is because the new version of the steps expression language will require explicit casting, i.e. ${{ str(job.inputs.house_number) }}
//   - we add this constraint so folks using job.inputs don't rely on implicit casting that will soon be removed
func expandStringTemplate(parts []exprPart, interpolationCtx *expression.InterpolationContext) (*structpb.Value, error) {
	output := strings.Builder{}

	for _, part := range parts {
		if part.kind == kindLiteral {
			output.WriteString(part.text)
			continue
		}

		value, err := expression.Expand(interpolationCtx, structpb.NewStringValue(part.text))
		if err != nil {
			return nil, err
		}

		if _, ok := value.Value.Kind.(*structpb.Value_StringValue); !ok {
			exprContent := strings.TrimSpace(strings.NewReplacer("${{", "", "}}", "").Replace(part.text))
			return nil, fmt.Errorf("%s: must evaluate to a string when used in a string template", exprContent)
		}

		output.WriteString(value.Value.GetStringValue())
	}

	return structpb.NewStringValue(output.String()), nil
}

func extractParts(expr string) []exprPart {
	parts := make([]exprPart, 0)
	cursor := 0

	for _, match := range expressionRe.FindAllStringIndex(expr, -1) {
		start, end := match[0], match[1]
		if start > cursor {
			// Literal before expression
			parts = append(parts, exprPart{kind: kindLiteral, text: expr[cursor:start]})
		}

		// Expression
		parts = append(parts, exprPart{kind: kindExpr, text: expr[start:end]})
		cursor = end
	}

	if cursor < len(expr) {
		// Trailing literal
		parts = append(parts, exprPart{kind: kindLiteral, text: expr[cursor:]})
	}

	return parts
}

func buildJobVariables(jobInputs []JobInput) (map[string]*structpb.Value, error) {
	inputVars := map[string]*structpb.Value{}
	for _, input := range jobInputs {
		inputVars[input.Key] = input.Value
	}

	jobVars := map[string]*structpb.Value{}
	jobVars["inputs"] = structpb.NewStructValue(&structpb.Struct{Fields: inputVars})
	return jobVars, nil
}
