package runner

import (
	"fmt"
	"maps"
	"os"
	"strings"
	"sync"

	"golang.org/x/exp/slices"
)

// Environment represents environment variables.
// Environment variables are used in places such as the step export_file, step definitions with ENV, and OS environment.
// Environment does not merge maps, and so does not lose information. Merging of environments occurs when values are retrieved.
// An environment can be added as "lexical scope", these values have higher precedence when looking up a variable.
// Mutations to the environment take precedence over initialized variables, most recent mutations have the highest precedence
type Environment struct {
	vars        map[string]string // Variables of this lexical scoped environment
	parent      *Environment      // Variables of the parent lexical scoped environment
	mutationsMu sync.RWMutex      // Protects mutations only
	mutations   []*Environment    // Mutations to the environment
}

// NewEnvironmentFromOS returns the environment variables found in the OS runtime.
// Variables can be filtered by name, passing no names will return all variables.
func NewEnvironmentFromOS(rejectIf ...func(string) bool) (*Environment, error) {
	vars := map[string]string{}

	for _, nameValue := range os.Environ() {
		name, value, ok := strings.Cut(nameValue, "=")

		if !ok {
			return nil, fmt.Errorf("failed to parse environment variable: %s", nameValue)
		}

		if acceptEnvName(name, rejectIf) {
			vars[name] = value
		}
	}

	return NewEnvironment(vars), nil
}

func NewEnvironmentFromOSWithKnownVars() (*Environment, error) {
	knownVars := []string{
		"HTTPS_PROXY",
		"HTTP_PROXY",
		"LANG",
		"LC_ALL",
		"LC_CTYPE",
		"LOGNAME",
		"NO_PROXY",
		"PATH",
		"SHELL",
		"TERM",
		"TMPDIR",
		"TZ",
		"USER",
		"all_proxy",
		"http_proxy",
		"https_proxy",
		"no_proxy",
	}

	return NewEnvironmentFromOS(func(envName string) bool { return !slices.Contains(knownVars, envName) })
}

func NewEmptyEnvironment() *Environment {
	return NewEnvironment(map[string]string{})
}

func NewEnvironment(vars map[string]string) *Environment {
	return &Environment{vars: vars, parent: nil, mutations: nil}
}

// NewKVEnvironment creates an environment from a list of key values
func NewKVEnvironment(keyValues ...string) *Environment {
	if len(keyValues)%2 == 1 {
		panic("NewEnvironmentFromValues: odd argument count")
	}

	vars := make(map[string]string)

	for i := 0; i < len(keyValues); i += 2 {
		vars[keyValues[i]] = keyValues[i+1]
	}

	return NewEnvironment(vars)
}

func (e *Environment) AddLexicalScope(vars map[string]string) *Environment {
	if len(vars) == 0 {
		return e
	}

	return &Environment{vars: vars, parent: e, mutations: nil}
}

func (e *Environment) Len() int {
	return len(e.Values())
}

// Values returns a merged map of all environment variables.
// This method ensures a consistent snapshot by acquiring all locks in the chain.
func (e *Environment) Values() map[string]string {
	envs, releaseLocks := e.readLockEnvChain()
	defer releaseLocks()

	values := map[string]string{}

	for i := len(envs) - 1; i >= 0; i-- {
		maps.Copy(values, envs[i].vars)

		for _, mutation := range envs[i].mutations {
			maps.Copy(values, mutation.vars)
		}
	}

	return values
}

// ValueOf returns the value of a specific environment variable.
// This ensures consistency by acquiring locks in the same order as Values().
func (e *Environment) ValueOf(key string) string {
	// For single value lookups, we still need consistent snapshots
	// Optimize by checking each level and returning early
	envs, releaseLocks := e.readLockEnvChain()
	defer releaseLocks()

	// Check from shallowest to deepest (reverse order for precedence)
	for i := 0; i < len(envs); i++ {
		// Check mutations first (most recent has highest precedence)
		for j := len(envs[i].mutations) - 1; j >= 0; j-- {
			if value, exists := envs[i].mutations[j].vars[key]; exists {
				return value
			}
		}

		// Check base variables
		if value, exists := envs[i].vars[key]; exists {
			return value
		}
	}

	return ""
}

// Mutate adds a mutation to the environment.
func (e *Environment) Mutate(env *Environment) {
	if env == nil || len(env.vars) == 0 {
		return
	}

	e.mutationsMu.Lock()
	defer e.mutationsMu.Unlock()

	e.mutations = append(e.mutations, env)
}

func acceptEnvName(name string, rejectIf []func(string) bool) bool {
	for _, rejectIfFn := range rejectIf {
		if rejectIfFn(name) {
			return false
		}
	}

	return true
}

// readLockEnvChain returns collected environments from this one up to the root
// and applies a read lock to each mutation mutex
func (e *Environment) readLockEnvChain() ([]*Environment, func()) {
	var chain []*Environment
	current := e

	for current != nil {
		chain = append(chain, current)
		current = current.parent
	}

	// Acquire all read locks in consistent order (deepest to shallowest)
	// This prevents deadlocks and ensures consistent snapshots
	for i := len(chain) - 1; i >= 0; i-- {
		chain[i].mutationsMu.RLock()
	}

	releaseLocks := func() {
		for i := 0; i < len(chain); i++ {
			chain[i].mutationsMu.RUnlock()
		}
	}

	return chain, releaseLocks
}
