/*******************************************************************************
 * The information contained in this file is confidential and proprietary to
 * QLogic Corporation.  No part of this file may be reproduced or
 * distributed, in any form or by any means for any purpose, without the
 * express written permission of QLogic Corporation.
 *
 * (c) COPYRIGHT 2015 QLogic Corporation, ALL RIGHTS RESERVED.
 *******************************************************************************/
/* **********************************************************
 * Copyright 2015 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * qfle3_uplink.c --
 *
 *      Utility function implementation of native bxe driver.
 */
#include <vmkapi.h>
#include "qfle3.h"
#include "qfle3_sm.h"
#include "qfle3_mgmt.h"
#include "qfle3_ecore_link.h"
#include "qfle3_ecore_init.h"
/*
 * Local Functions
 */
static VMK_ReturnStatus qfle3_uplink_tx(vmk_AddrCookie driverData, vmk_PktList pktList);
static VMK_ReturnStatus qfle3_uplink_mtu_set(vmk_AddrCookie driverData, vmk_uint32 mtu);
static VMK_ReturnStatus qfle3_uplink_state_set(vmk_AddrCookie driverData,
					       vmk_UplinkState state);
VMK_ReturnStatus qfle3_uplink_stats_get(vmk_AddrCookie driverData,
					       vmk_UplinkStats * stats);
static VMK_ReturnStatus qfle3_uplink_associate(vmk_AddrCookie driverData,
					       vmk_Uplink uplink);
static VMK_ReturnStatus qfle3_uplink_disassociate(vmk_AddrCookie driverData);
static VMK_ReturnStatus qfle3_uplink_cap_enable(vmk_AddrCookie driverData,
						vmk_UplinkCap cap);
static VMK_ReturnStatus qfle3_uplink_cap_disable(vmk_AddrCookie driverData,
						 vmk_UplinkCap cap);
static VMK_ReturnStatus qfle3_uplink_start_io(vmk_AddrCookie driverData);
static VMK_ReturnStatus qfle3_uplink_quiesce_io(vmk_AddrCookie driverData);
static VMK_ReturnStatus qfle3_uplink_reset(vmk_AddrCookie driverData);
static VMK_ReturnStatus qfle3_remove_uplink_device(vmk_Device device);

void qfle3_deactive_dev(qfle3_adapter * adapter);
VMK_ReturnStatus qfle3_activate_dev(qfle3_adapter * adapter);

void qfle3_update_max_mf_config(qfle3_adapter *adapter, vmk_uint32 value);
static inline int qfle3_has_rx_work(struct qfle3_fastpath *fp);
static vmk_uint32
   qfle3_txeof(struct qfle3_adapter *adapter, struct qfle3_fp_txdata* txdata, vmk_uint32 budget, vmk_Bool isPanic);
extern vmk_uint32 enable_fcoe_queue;
extern vmk_uint32 netpoll_count;

/*
 * Global Data
 */
vmk_DeviceOps qfle3_uplink_device_ops = {
   .removeDevice = qfle3_remove_uplink_device,
};

static VMK_ReturnStatus
qfle3_remove_uplink_device(vmk_Device device)
{
   vmk_LogMessage("Removing uplink device\n");
   return vmk_DeviceUnregister(device);
}

vmk_UplinkOps qfle3_uplink_ops = {
   .uplinkTx = qfle3_uplink_tx,
   .uplinkMTUSet = qfle3_uplink_mtu_set,
   .uplinkStateSet = qfle3_uplink_state_set,
   .uplinkStatsGet = qfle3_uplink_stats_get,
   .uplinkAssociate = qfle3_uplink_associate,
   .uplinkDisassociate = qfle3_uplink_disassociate,
   .uplinkCapEnable = qfle3_uplink_cap_enable,
   .uplinkCapDisable = qfle3_uplink_cap_disable,
   .uplinkStartIO = qfle3_uplink_start_io,
   .uplinkQuiesceIO = qfle3_uplink_quiesce_io,
   .uplinkReset = qfle3_uplink_reset,
};
#if (ESX_DDK_VERSION >= 2017)
static vmk_UplinkQueueRSSGenericOps qfle3_rss_generic_ops = {
   .queueGetRSSEnginesInfo           =   qfle3_rss_get_engines_info,
   .queueGetRSSEngineSupportedConfig =   qfle3_rss_get_engines_supported_config,
   .queueGetRSSEngineCurConfig       =   qfle3_rss_get_engines_cur_config,
   .queueSetRSSEngineConfig          =   qfle3_rss_set_engine_config,
   .queueSetRSSEngineIndTable        =   qfle3_rss_set_engine_ind_table,
   .queueSetRSSEngineIndTableIdx     =   qfle3_rss_set_engine_ind_table_idx,
   .queueGetRSSEngineIndTable        =   qfle3_rss_get_engine_ind_table,
};
#else
static vmk_UplinkQueueRSSDynOps qfle3_rss_dynamic_ops = {
   .queueGetRSSParams             = qfle3_rss_get_params,
   .queueInitRSSState             = qfle3_rss_state_init,
   .queueUpdateRSSIndTable        = qfle3_rss_ind_tbl_update,
   .queueGetRSSIndTable           = qfle3_rss_get_ind_tbl,
};
#endif
VMK_ReturnStatus qfle3_priv_stats_get_len(vmk_AddrCookie driverData, //IN
					  vmk_ByteCount *length);
VMK_ReturnStatus
qfle3_priv_stats_get(vmk_AddrCookie driverData,  //IN
		     char *buffer,                 //IN/OUT
		     vmk_ByteCount length);     //IN

VMK_ReturnStatus
qfle3_geneve_port_update(vmk_AddrCookie cookie, vmk_uint16 port_num);

static vmk_UplinkPrivStatsOps qfle3_priv_stats_ops = {
   .privStatsLengthGet =  qfle3_priv_stats_get_len,
   .privStatsGet       =  qfle3_priv_stats_get,
};

#if (VMKAPI_REVISION >= VMK_REVISION_FROM_NUMBERS(2, 5, 0, 0))
static vmk_UplinkGeneveOffloadParams qfle3_geneve_offload_ops = {
         .portUpdate = qfle3_geneve_port_update,
	       /*
		*Driver: qfle3(bnx2x)
		*1. Egress
		*- IP checksum offload: HW is only capable of calculating one of the inner or outer IP checksum. For simplicity sake we can assume that outer IP checksum is not supported.
		*- Outer L4 checksum is not supported.
		*- Inner VLAN insertion is not supported.
		*- The same constraints regarding inner IPv6 packets with extension headers as qedentv above.
		*
		*2. Ingress
		*- Inner IP checksum is not supported.
		*- L4 checksum offload is not supported. Neither inner nor outer, so LRO is not supported.
		*- Inner VLAN removal is not supported.
		*- For IPv6 packets with extension headers, offloads (inner or outer) are not supported.
		*
		*3. Additional Constraints:
		*- Only one VXLAN UDP port allowed per physical port.
		*- Only one GENEVE UDP port allowed per physical port.
		*- GENEVE options length limited to 32 bytes.
		*/
         .maxHeaderOffset = 356,
         .flags = (VMK_UPLINK_GENEVE_FLAG_INNER_IPV4_CSO |
		   VMK_UPLINK_GENEVE_FLAG_INNER_IPV4_TSO |
		   VMK_UPLINK_GENEVE_FLAG_INNER_IPV6_CSO |
                   VMK_UPLINK_GENEVE_FLAG_INNER_IPV6_TSO |
		   VMK_UPLINK_GENEVE_FLAG_MAX_OPTIONS_LENGTH),
         .maxOptionsLength = 32,
};
#else
static vmk_UplinkGeneveOffloadParams qfle3_geneve_offload_ops = {
   .portUpdate = qfle3_geneve_port_update,
   .maxHeaderOffset = 356, /* Max inner L7 offset.(104+252) */
   .flags = 0,   /* No VMK_UPLINK_GENEVE_FLAG_OUTER_UDP_CSO*/
};
#endif

inline vmk_uint16
qfle3_tx_avail(struct qfle3_adapter *adapter, struct qfle3_fp_txdata* txdata)
{


   vmk_int16 used;
   vmk_uint16 prod;
   vmk_uint16 cons;

   prod = txdata->tx_bd_prod;
   cons = txdata->tx_bd_cons;

   used = SUB_S16(prod, cons);
   return (vmk_int16) (TX_BD_USABLE - used);
}

/*
 * Checks to ensure the 13 bd sliding window is >= MSS for TSO.
 * Check that (13 total bds - 3 bds) = 10 bd window >= MSS.
 * The window: 3 bds are = 1 for headers BD + 2 for parse BD and last BD
 * The headers comes in a seperate bd in FreeBSD so 13-3=10.
 * Returns: 0 if OK to send, 1 if packet needs further defragmentation
 */
static int
qfle3_parse_tsowind(qfle3_adapter *adapter, vmk_uint32 nsegs, vmk_PktHandle * pkt,
		    vmk_uint16 tso_mss, vmk_uint32 hlen, vmk_uint32 mustEncap)
{
   vmk_int32 num_wnds, wnd_size, wnd_sum;
   vmk_int32 frag_idx, wnd_idx, first_wnd_end;
   vmk_uint8 hdrs = 0;
   vmk_uint16 sum_frag = 0, payload_start = 0, payload_off = 0;
   const vmk_SgElem *sge;

   wnd_sum = 0;

   /* Find the TCP payload start segment index and the offset within this segment*/
   for(hdrs = 0; hdrs < nsegs; hdrs++) {
      sge = vmk_PktSgElemGet(pkt, hdrs);
      sum_frag += sge->length;
      if (hlen < sum_frag) {
         payload_off = sum_frag - hlen;
	 break;
      } else if (hlen == sum_frag) {
         payload_start++;
	 break;
      }
      payload_start++;
   }

   QFLE3_DBG(QFLE3_DBG_TX, "payload_start %d, payload_off %d", payload_start, payload_off);
   /* ETH + IP + TCP header bytes must reside in a whole number of BDs, which is
    * smaller than 11. When payload_off > 0, we need to split that segement, hence
    * one more segment.
    */ 
   if (payload_start == 0)
      hdrs = 1;
   else
      hdrs = (payload_off) ? (payload_start + 1) : payload_start;
   QFLE3_DBG(QFLE3_DBG_TX, "hdrs [%d]", hdrs);

#define QFLE3_TSO_MAX_HDRS ETH_TX_START_BD_HDR_NBDS

   if (hdrs > QFLE3_TSO_MAX_HDRS) {
      return (1);
   }
   /* Each (13 - number of header BDs - 2) consecutive TCP payload BDs, must hold
    * payload size which is greater than, or equal to, MSS size.
    */
   if (mustEncap)
      wnd_size = 13 - hdrs - 2 - 1;
   else
      wnd_size = 13 - hdrs - 2;

   /* Calculate first window's payload length */
   first_wnd_end = MIN((payload_start + wnd_size - 1), (nsegs - 1));
   for (frag_idx = payload_start; frag_idx <= first_wnd_end; frag_idx++) {
      sge = vmk_PktSgElemGet(pkt, frag_idx);
      if (frag_idx == payload_start && payload_off) {
         wnd_sum += payload_off;
      } else {
         wnd_sum += sge->length;
      }
   }

   QFLE3_DBG(QFLE3_DBG_TX,
      "wnd_size %d, first window len %d, tso_mss %d, last segment %d",
      wnd_size, wnd_sum, tso_mss, first_wnd_end);
   /*
    * check the first window
    */
   if (wnd_sum < tso_mss) {
      QFLE3_DBG(QFLE3_DBG_TX, "MSS verify failed, wnd_sum %d, tso_mss %d",
         wnd_sum, tso_mss);
      return (1);
   }

   num_wnds = nsegs - first_wnd_end;
   QFLE3_DBG(QFLE3_DBG_TX, "rest window num %d", num_wnds);
   /*
    * run through the windows, no need to check last window.
    */
   for (wnd_idx = 0; wnd_idx < (num_wnds - 1); wnd_idx++, frag_idx++) {
      /*
       * subtract the first mbuf->m_len of the last wndw(-header)
       */
      sge = vmk_PktSgElemGet(pkt, wnd_idx + payload_start);
      if (wnd_idx == 0 && payload_off) {
	 wnd_sum -= payload_off;
      } else {
	 wnd_sum -= sge->length;
      }
      /*
       * add the next mbuf len to the len of our new window
       */
      sge = vmk_PktSgElemGet(pkt, frag_idx);
      wnd_sum += sge->length;
      if (wnd_sum < tso_mss) {
	 QFLE3_DBG(QFLE3_DBG_TX, "MSS verify failed, wnd_idx %d, wnd_sum %d, tso_mss %d",
		   wnd_idx, wnd_sum, tso_mss);
	 return (1);
      }
   }
   return (0);
}

static inline vmk_Bool
qfle3_pktIsBcast(const vmk_uint8 * addr)
{
   return (addr[0] & addr[1] & addr[2] & addr[3] & addr[4] & addr[5]) == 0xff;
}

static inline vmk_Bool
qfle3_pktIsMcast(const vmk_uint8 * addr)
{
   return (0x01 & addr[0]);
}


static void
qfle3_set_pbd_lso_e2(vmk_uint16 mss, vmk_uint32 * parsing_data)
{
   *parsing_data |= ((mss <<
		      ETH_TX_PARSE_BD_E2_LSO_MSS_SHIFT) & ETH_TX_PARSE_BD_E2_LSO_MSS);

   /*
    * XXX test for IPv6 with extension header...
    */
#if 0
   struct ip6_hdr *ip6;
   if (ip6 && ip6->ip6_nxt == 'some ipv6 extension header')
      *parsing_data |= ETH_TX_PARSE_BD_E2_IPV6_WITH_EXT_HDR;
#endif
}

static void
qfle3_set_pbd_csum_e2(struct qfle3_fastpath *fp,
		      vmk_PktHandle * pkt, vmk_uint32 l4hoffset,
		      vmk_uint32 l4hsize,
		      vmk_uint16 l4type,
		      vmk_uint32 *parsing_data)
{
   /*
    * XXX assuming L4 header is contiguous to IPv4/IPv6 in the same mbuf
    */
   *parsing_data |=
      (((l4hoffset >> 1) << ETH_TX_PARSE_BD_E2_L4_HDR_START_OFFSET_W_SHIFT) &
       ETH_TX_PARSE_BD_E2_L4_HDR_START_OFFSET_W);

   if (l4type == VMK_PKT_HEADER_L4_TCP) {
      *parsing_data |= (((l4hsize >> 2) <<
			 ETH_TX_PARSE_BD_E2_TCP_HDR_LENGTH_DW_SHIFT) &
			ETH_TX_PARSE_BD_E2_TCP_HDR_LENGTH_DW);
   }
   return;
}


static vmk_Bool
qfle3_split_hdr(
   qfle3_adapter       *adapter,
   qfle3_txbuf_info     *tx_buf,
   vmk_uint16           hlen,
   vmk_uint8            *total_hdr_nbds)
{
   vmk_uint8 cnt = 0;
   vmk_uint16 sum_frag = 0;
   vmk_uint8  hdr_nbds = 0;
   vmk_Bool need_split = VMK_FALSE;

   for(cnt = 0; cnt < tx_buf->nsegs; cnt++)
   {
      hdr_nbds++;
      sum_frag += tx_buf->tx_segs[cnt].len;
      if (hlen <= sum_frag)
      {
	 if (hlen < sum_frag)
	 {
	    need_split = VMK_TRUE;
	 }
	 break;
      }
   }

   *total_hdr_nbds = hdr_nbds;
   return (need_split);
}


#define XMIT_ENC_OUTER_V4	(1 << 0)
#define XMIT_ENC_OUTER_V6	(1 << 1)
#define XMIT_ENC_INNER_V4	(1 << 2)
#define XMIT_ENC_INNER_V6	(1 << 3)
#define XMIT_ENC_INNER_TCP	(1 << 4)
#define XMIT_ENC_INNER_UDP  (1 << 5)
#define XMIT_ENC_OUTER_UDP  (1 << 6)
#define XMIT_ENC_BAD_FRAME  (0x7f)

static vmk_uint32 qfle3_lookup_encap(qfle3_adapter *adapter, vmk_PktHandle *pkt,
                         qfle3_pkt_tx_info_t *tx_info, struct qfle3_fastpath *fp)
{
   vmk_uint32 rc = 0;
   VMK_ReturnStatus status = VMK_OK;
   vmk_PktHeaderEntry *encap_hdr = NULL;

   if (QFLE3_IS_ENCAP_OFFLOAD_ENABLED(adapter) &&
#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
      vmk_PktIsInnerOffload(pkt)
#else
      vmk_PktIsEncapsulatedFrame(pkt)
#endif
      ) {

      status = vmk_PktHeaderEncapFind(pkt, &encap_hdr, NULL);
      if ((status == VMK_OK) &&
	  ((encap_hdr->type == VMK_PKT_HEADER_ENCAP_VXLAN) ||
           (encap_hdr->type == VMK_PKT_HEADER_ENCAP_GENEVE))) {

	 vmk_PktHeaderEntry *inner_l3header, *inner_l4header, *outer_l3header, *outer_l4header;
	 vmk_IPv4Hdr *ip4h;
	 vmk_IPv6Hdr *ip6h;
	 vmk_TCPHdr *tcphdr;
        
         if (encap_hdr->type == VMK_PKT_HEADER_ENCAP_VXLAN) {
            adapter->drv_stats.encap_stats.vxlan_tx_pkts++;
            fp->drv_stats.encap_stats.vxlan_tx_pkts++;
         }
         if (encap_hdr->type == VMK_PKT_HEADER_ENCAP_GENEVE) {
            adapter->drv_stats.encap_stats.geneve_tx_pkts++;
            fp->drv_stats.encap_stats.geneve_tx_pkts++;
         }
	 status = vmk_PktHeaderL3Find(pkt, &outer_l3header, NULL);
	 if (status == VMK_OK) {
	    if (outer_l3header->type == VMK_PKT_HEADER_L3_IPv6) {
	       rc |= XMIT_ENC_OUTER_V6;
	    } else  if (outer_l3header->type == VMK_PKT_HEADER_L3_IPv4) {
	       vmk_uint32 csum = 0;
	       rc |= XMIT_ENC_OUTER_V4;
	       status = vmk_PktHeaderDataGet(pkt, outer_l3header, (void **)&ip4h);
	       VMK_ASSERT(status == VMK_OK);

	       tx_info->encap_packet_outer_ip_length = ip4h->headerLength << 1;

	       csum = (vmk_uint32)(~ip4h->checksum) -
		  (vmk_uint32)ip4h->totalLength - (vmk_uint32)ip4h->fragmentInfo;

	       tx_info->encap_packet_outer_ip_csum_wo_len_flags_frag = vmk_CPUToBE16(vmk_NetCsumFold(csum));
	       vmk_PktHeaderDataRelease(pkt, outer_l3header, ip4h, VMK_FALSE);
	    }

	    tx_info->encap_packet_outer_ip_offset = outer_l3header->offset;
	 }

	 status = vmk_PktHeaderL4Find(pkt, &outer_l4header, NULL);
	 if (status == VMK_OK) {
	    if (outer_l4header->type == VMK_PKT_HEADER_L4_UDP){
	       rc |= XMIT_ENC_OUTER_UDP;
	       tx_info->encap_packet_outer_udp_hdr_offset = outer_l4header->offset;
	    }
	 } else {
	    QFLE3_DBG(QFLE3_DBG_TX, "Tunneled packet type 0x%x w/o outer L4 header", encap_hdr->type);
	    return 0;
	 }

	 status = vmk_PktHeaderEncapL4Find(pkt, &inner_l4header, NULL);
	 if (status == VMK_OK) {
	    tx_info->encap_packet_inner_L4_offset = inner_l4header->offset;
	    tx_info->encap_packet_inner_L4_length = vmk_PktHeaderLength(inner_l4header);
	    tx_info->encap_packet_inner_L4_protocol = inner_l4header->type;
	    tx_info->encap_packet_header_length = tx_info->encap_packet_inner_L4_offset +
	       tx_info->encap_packet_inner_L4_length;

	 } else {
	    QFLE3_DBG(QFLE3_DBG_TX, "Tunneled packet w/o 0x%x inner L4 header, no offload", encap_hdr->type);
	    return 0;
	 }

	 status = vmk_PktHeaderEncapL3Find(pkt, &inner_l3header, NULL);
	 if (status == VMK_OK) {
	    if (inner_l3header->type == VMK_PKT_HEADER_L3_IPv6) {
	       rc |= XMIT_ENC_INNER_V6;
	       status = vmk_PktHeaderDataGet(pkt, inner_l3header, (void **)&ip6h);
	       VMK_ASSERT(status == VMK_OK);
	       tx_info->encap_packet_inner_pseudo_csum =
		  vmk_CPUToBE16(~ vmk_NetCsumIPv6Pseudo(ip6h->saddr,
							ip6h->daddr,
							&ip6h->nextHeader,
							0, 0));
	       vmk_PktHeaderDataRelease(pkt, inner_l3header, ip6h, VMK_FALSE);

	    } else	if (inner_l3header->type == VMK_PKT_HEADER_L3_IPv4) {
	       rc |= XMIT_ENC_INNER_V4;

	       status = vmk_PktHeaderDataGet(pkt, inner_l3header, (void **)&ip4h);
	       VMK_ASSERT(status == VMK_OK);
	       tx_info->encap_packet_inner_ip_fragment_id = vmk_CPUToBE16(ip4h->identification);
	       tx_info->encap_packet_inner_pseudo_csum =
		  vmk_CPUToBE16(~ vmk_NetCsumIPv4Pseudo(&ip4h->saddr,
							&ip4h->daddr,
							&ip4h->protocol,
							0, 0));
	       vmk_PktHeaderDataRelease(pkt, inner_l3header, ip4h, VMK_FALSE);
	    }

	    if (inner_l4header->type == VMK_PKT_HEADER_L4_TCP) {
	       rc |= XMIT_ENC_INNER_TCP;
	       status = vmk_PktHeaderDataGet(pkt, inner_l4header, (void **)&tcphdr);
	       VMK_ASSERT(status == VMK_OK);

	       tx_info->encap_packet_tcp_send_seq = vmk_CPUToBE32(tcphdr->seq);
	       tx_info->encap_packet_tcp_flag = tcphdr->flags;
	       tx_info->encap_packet_inner_L4_protocol = VMK_PKT_HEADER_L4_TCP;

	       vmk_PktHeaderDataRelease(pkt, inner_l4header, tcphdr, VMK_FALSE);

	       tx_info->encap_packet_outer_ip_offset = outer_l3header->offset;
	    } else if(inner_l4header->type == VMK_PKT_HEADER_L4_UDP)
	       rc |= XMIT_ENC_INNER_UDP;
	    else {
	        QFLE3_DBG(QFLE3_DBG_TX, "Tunneled pkt type 0x%x inner L4 type 0x%x, no offload",
                                           encap_hdr->type, inner_l4header->type);
                return 0;
	    }

	    tx_info->encap_packet_inner_ip_offset = inner_l3header->offset;
	 } else {
            QFLE3_DBG(QFLE3_DBG_TX, "Tunneled pkt 0x%x w/o inner L3 header, no offload",
                                                        encap_hdr->type);
            return 0;
         }

	 if (rc & XMIT_ENC_OUTER_V4)
	    tx_info->encap_packet_outer_ip_hdr_to_payload = inner_l4header->offset +
	       vmk_PktHeaderLength(inner_l4header) - outer_l3header->offset;
	 else if (rc & XMIT_ENC_OUTER_V6)
	    tx_info->encap_packet_outer_ip_hdr_to_payload = inner_l4header->offset +
	       vmk_PktHeaderLength(inner_l4header) - outer_l3header->offset - sizeof(vmk_IPv6Hdr);
      } else {
         QFLE3_INFO("bad tunneled packet type 0x%x, no outer L2 header", encap_hdr->type);
         return XMIT_ENC_BAD_FRAME;
      }
   }

   return (rc);
}

static void qfle3_update_pbds_tso_enc(vmk_PktHandle *pkt,
				      struct eth_tx_parse_bd_e2 *pbd_1st_e2,
				      struct eth_tx_parse_2nd_bd *pbd_2nd_e2,
				      vmk_uint32 xmit_type, qfle3_pkt_tx_info_t *tx_infop)
{
   vmk_uint16 outerip_start_w = 0, outerip_len = 0;

   /* from outer IP to payload */
   pbd_2nd_e2->fw_ip_hdr_to_payload_w = tx_infop->encap_packet_outer_ip_hdr_to_payload >> 1;

   /* outer IP header info */
   if (xmit_type & XMIT_ENC_OUTER_V4) {
      outerip_len = tx_infop->encap_packet_outer_ip_length;
      pbd_2nd_e2->fw_ip_csum_wo_len_flags_frag = tx_infop->encap_packet_outer_ip_csum_wo_len_flags_frag;
   } else if (xmit_type & XMIT_ENC_OUTER_V6){
      pbd_2nd_e2->fw_ip_hdr_to_payload_w -= ((sizeof(vmk_IPv6Hdr)) >> 1);
      SET_FLAG(pbd_1st_e2->data.tunnel_data.flags, ETH_TUNNEL_DATA_IPV6_OUTER, 1);
   }

   pbd_2nd_e2->tcp_send_seq = tx_infop->encap_packet_tcp_send_seq;
   pbd_2nd_e2->tcp_flags = tx_infop->encap_packet_tcp_flag;

   outerip_start_w = tx_infop->encap_packet_outer_ip_offset >> 1;

   pbd_2nd_e2->global_data |=
      outerip_start_w | (outerip_len <<
			 ETH_TX_PARSE_2ND_BD_IP_HDR_LEN_OUTER_W_SHIFT);

   if (xmit_type & XMIT_ENC_OUTER_UDP) {
      SET_FLAG(pbd_2nd_e2->global_data, ETH_TX_PARSE_2ND_BD_TUNNEL_UDP_EXIST, 1);
      pbd_2nd_e2->tunnel_udp_hdr_start_w = tx_infop->encap_packet_outer_udp_hdr_offset >> 1;
   }

   if (xmit_type & XMIT_ENC_INNER_V4) {
      pbd_2nd_e2->hw_ip_id = tx_infop->encap_packet_inner_ip_fragment_id;
   }
   pbd_1st_e2->data.tunnel_data.pseudo_csum = tx_infop->encap_packet_inner_pseudo_csum;
}
typedef struct FcoeFCHdr{
    vmk_uint8 reserved[13];
    vmk_uint8 sof;
    vmk_uint8 Rctl;
    vmk_uint8 reserved1[11];
    vmk_uint8 seq_id;
    vmk_uint8 df_ctl;
    vmk_uint16 seq_cnt;
    vmk_uint16 ox_id;
    vmk_uint16 rx_id;
} VMK_ATTRIBUTE_PACKED FcoeFCHdr;

typedef struct EthHdrVLAN {
   /** Destination MAC address. */
  vmk_EthHdr ethhdr;
   /** Ethernet type of the payload. */
   vmk_VLANHdr  vlanhdr;
   FcoeFCHdr fcoehdr;
} VMK_ATTRIBUTE_PACKED EthHdrVLAN;

typedef struct EthHdr {
   /** Destination MAC address. */
  vmk_EthHdr ethhdr;
   FcoeFCHdr fcoehdr;
} VMK_ATTRIBUTE_PACKED EthHdr;

vmk_Bool getEthType(qfle3_adapter * adapter,vmk_PktHandle *pkt, vmk_uint16 *ethType, vmk_uint16 *vlanId)
{

    vmk_EthHdr *ethHdr = NULL;
    vmk_PktHeaderEntry *l2Header;
    VMK_ReturnStatus status;
    EthHdr *fullHdr;
    status = vmk_PktHeaderL2Find(pkt, &l2Header, NULL);
    if (status != VMK_OK)
       return VMK_FALSE;
    status = vmk_PktHeaderDataGet(pkt, l2Header, (void **) &ethHdr);
    if (status != VMK_OK)
       return VMK_FALSE;

    *ethType = ethHdr->type;
    if (*ethType == VMK_ETH_TYPE_VLAN_NBO){
         EthHdrVLAN * vlanHdr;
         vlanHdr = (EthHdrVLAN * )ethHdr;
         *ethType = vlanHdr->vlanhdr.type;
         *vlanId = vlanHdr->vlanhdr.vlanIDHigh << 8 | vlanHdr->vlanhdr.vlanIDLow;
         if (vlanHdr->ethhdr.type == 0x0689 && vlanHdr->fcoehdr.Rctl == 0x5) {
            vmk_uint16 oxid;
            vmk_uint16 rxid;
            oxid = vmk_BE16ToCPU(vlanHdr->fcoehdr.ox_id);
            rxid = vmk_BE16ToCPU(vlanHdr->fcoehdr.rx_id);
            QFLE3_DBG(QFLE3_DBG_DCB, "FC4XRdy vlan OX_Id 0x%x, RX_Id 0x%x\n", oxid, rxid);
         }
         vmk_PktHeaderDataRelease(pkt, l2Header, ethHdr, VMK_FALSE);
         return VMK_TRUE;
    }

    if (*ethType == 0x0689) // && fullHdr->fcoehdr.Rctl == 0x5) // FC4XTdy
    {
        unsigned int len;
        vmk_uint8 *frameVA;
        vmk_uint16 oxid;
        vmk_uint16 rxid;
        vmk_uint16 seq_cnt;
        vmk_uint8 seq_id;
        len = MIN(vmk_PktFrameMappedLenGet(pkt), vmk_PktFrameLenGet(pkt));
        frameVA = (vmk_uint8 *) vmk_PktFrameMappedPointerGet(pkt);

        fullHdr = (EthHdr *) frameVA;
//        if (frameVA[28] == 0x1) {
           oxid = fullHdr->fcoehdr.ox_id;
           rxid = fullHdr->fcoehdr.rx_id;
           seq_id = fullHdr->fcoehdr.seq_id;
           seq_cnt = fullHdr->fcoehdr.seq_cnt;
           QFLE3_DBG(QFLE3_DBG_DCB, "FC4XRdy  OX_Id 0x%x, RX_Id 0x%x, seq_id 0x%x, seq_cnt 0x%x: \n", oxid, rxid, seq_id, seq_cnt);
//           frameVA+=28;
//           QFLE3_DBG(QFLE3_DBG_DCB,"%02x %02x %02x %02x %02x %02x %02x %02x\n",
//            frameVA[0],frameVA[1],frameVA[2],frameVA[3],frameVA[4],frameVA[5],frameVA[6],frameVA[7]);
//           QFLE3_DBG(QFLE3_DBG_DCB,"%02x %02x %02x %02x %02x %02x %02x %02x\n",
//           frameVA[8],frameVA[9],frameVA[10], frameVA[11],frameVA[12],frameVA[13],frameVA[14],frameVA[15]);
//
//           QFLE3_DBG(QFLE3_DBG_DCB,"%02x %02x %02x %02x %02x %02x %02x %02x\n",
//            frameVA[16],frameVA[17],frameVA[18],frameVA[19],frameVA[20],frameVA[21],frameVA[22],frameVA[23]);
//        }
    }

    vmk_PktHeaderDataRelease(pkt, l2Header, ethHdr, VMK_FALSE);
    return VMK_FALSE;
}
void count_protocol_pkt(qfle3_adapter * adapter, vmk_PktHandle *pkt, vmk_Bool tx)
{
    vmk_uint16 rawType = 0xffff;
    vmk_uint32 ethType = 0xffff;
    vmk_Bool tagged;
    vmk_uint16 vlanid = 0;
    int i;
    struct protocol_count * pc;
    if (tx)
        pc = &adapter->drv_stats_priv.tx_p;
    else
        pc = &adapter->drv_stats_priv.rx_p;

    tagged = getEthType(adapter, pkt, &rawType, &vlanid);
    ethType = rawType;
    if (tagged){
         ethType |= 0x80000000;
         ethType |= vlanid << 16;
    }else{ // check metadata
       vlanid = vmk_PktVlanIDGet(pkt);
       ethType |= vlanid << 16;
    }

    for (i =0; i< pc->protocol_count; i++) {
        if (pc->protocols[i] == ethType){
            pc->protocol_pkt_count[i] ++;
            return;
        }
    }
    if (pc->protocol_count < 64) {
        pc->protocol_count++;
        pc->protocols[pc->protocol_count-1] = ethType;
        pc->protocol_pkt_count[pc->protocol_count-1] = 1;
    } else {
//        QFLE3_ERR("protocol count out of range\n");
    }
}

int
qfle3_xmit_pkt(qfle3_adapter * adapter, vmk_uint32 qid, vmk_PktHandle * pkt)
{
   struct qfle3_fastpath *fp = QFLE3_GET_FP_FROM_TQID(adapter,qid);
   vmk_uint8 cos = FIRST_TX_COS_INDEX;
   struct qfle3_fp_txdata *txdata = fp->txdata_ptr[cos];
   /* Tx buffer hostside descriptor */
   qfle3_txbuf_info *tx_buf;
   /* Tx buffer h/w descriptor */
   struct eth_tx_parse_bd_e2 *pbd_e2 = NULL;
   struct eth_tx_parse_2nd_bd *pbd_2nd_e2 = NULL;
   struct eth_tx_bd *tx_data_bd, *tx_hdr_bd;
   struct eth_tx_bd *tx_total_pkt_size_bd;
   struct eth_tx_start_bd *tx_start_bd;
   /* Frame information */
   vmk_uint8 mac_type;
   int nsegs, nbds, total_seg_len;
   vmk_VlanID vlanID;
   vmk_EthHdr *ethHdr = NULL;
   vmk_Bool isMcast, isBcast;
   vmk_PktHeaderEntry *l2Header, *l3Header, *l4Header;
   vmk_Bool mustTSO, mustCsum, mustVlan;
   vmk_uint16 tso_mss = 0, l4hoffset, l4hsize, hlen = 0, last_segment_pad_size = 0;
   const vmk_SgElem *sgex;
   vmk_PktHandle *flatPkt = NULL, *copypkt = NULL;  /* defragment */
   vmk_uint32 mustEncap = 0;
   qfle3_pkt_tx_info_t tx_info = {0};

   /* Ring information */
   vmk_uint16 tx_bd_avail;
   vmk_uint16 bd_prod, pkt_prod, total_pkt_size;
   vmk_uint32 pbd_e2_parsing_data = 0;
   /* loop and return */
   VMK_ReturnStatus status = VMK_OK;
   int i;
   vmk_Bool do_split = VMK_FALSE;
   vmk_uint8 need_hdrs = 1;
   vmk_uint16 ethType=0xffff;

   total_seg_len = nbds = total_pkt_size = 0;
   tx_start_bd = NULL;
   tx_data_bd = NULL;
   tx_hdr_bd = NULL;
   tx_total_pkt_size_bd = NULL;

   if (!(fp->fp_state & QFLE3_SM_Q_STARTED)){

      QFLE3_DBG(QFLE3_DBG_TX, " fp[%d] NOT ready FP_state[%x] \n", fp->qid, fp->fp_state);
      return (VMK_BUSY);
   }

   adapter->drv_stats_priv.tx_pkt_rcvd++;
//   count_protocol_pkt(adapter, pkt, VMK_TRUE);
   vmk_SpinlockLock(fp->fp_lock);
   status = vmk_PktHeaderL2Find(pkt, &l2Header, NULL);
   if (status != VMK_OK) {
      QFLE3_ERR("pkt header L2 find failed, drop\n");
      goto drop_pkt;
   }
   status = vmk_PktHeaderDataGet(pkt, l2Header, (void **) &ethHdr);
   if (status != VMK_OK) {
      QFLE3_ERR("pkt header data get failed, drop\n");
      goto drop_pkt;
   }

   QFLE3_DBG(QFLE3_DBG_TX, "Eth header found @ [%p] on fp[%d] ret %d\n", ethHdr, fp->qid, status);
   isMcast = qfle3_pktIsMcast(ethHdr->daddr);
   isBcast = qfle3_pktIsBcast(ethHdr->daddr);
   if (isBcast) {
      mac_type = BROADCAST_ADDRESS;
   } else if (isMcast) {
      mac_type = MULTICAST_ADDRESS;
   } else
      mac_type = UNICAST_ADDRESS;

   ethType = ethHdr->type;
   if (ethType == VMK_ETH_TYPE_VLAN_NBO){
      EthHdrVLAN * vlanHdr;
      vlanHdr = (EthHdrVLAN * )ethHdr;
      ethType = vlanHdr->vlanhdr.type;
   }
   vmk_PktHeaderDataRelease(pkt, l2Header, ethHdr, VMK_FALSE);
   /* TODO: fix this */
   /* Redirect the FCoE packets to dedicated FCoE queue. */
   if (!IS_FCOE_FP(fp) && adapter->sw_fcoe && IS_FCOE_PKT(ethType)) {
      cos = FIRST_TX_ONLY_COS_INDEX;
      txdata = fp->txdata_ptr[cos];
      adapter->drv_stats_priv.tx_fcoe_redirected++;

      QFLE3_DBG(QFLE3_DBG_DCB, "Redirecting fp[%d] cos %d, from cid %d to cid %d, pkt count %jd\n",
         fp->qid, cos, fp->txdata_ptr[0]->cid, fp->txdata_ptr[cos]->cid, fp->drv_stats.tx_pkts);
   }

   tx_bd_avail = qfle3_tx_avail(adapter, txdata);
   total_seg_len = vmk_PktSgArrayTotalLenGet(pkt);
   total_pkt_size = vmk_PktFrameLenGet(pkt);
   nsegs = vmk_PktSgArrayGet(pkt)->numElems;
   sgex = vmk_PktSgElemGet(pkt, nsegs - 1);
   if (VMK_UNLIKELY((total_seg_len - total_pkt_size) >= sgex->length)) {
      /* Delete padding buffers. */
      status = vmk_PktDup(pkt, &copypkt);
      if (status != VMK_OK) {
         QFLE3_ERR("PktDup failed, drop");
         goto drop_pkt;
      }
   }
   /*
    * make sure there is enough room in the send queue
    */
   if(VMK_LIKELY(copypkt == NULL))
      nsegs = vmk_PktSgArrayGet(pkt)->numElems;
   else 
      nsegs = vmk_PktSgArrayGet(copypkt)->numElems;
   if (VMK_UNLIKELY(tx_bd_avail < (nsegs + 2))) {
      QFLE3_DBG(QFLE3_DBG_TX, "Stop: tx queue fp[%d] busy\n", adapter->num_rxqs_drv + qid);
      QFLE3_DBG(QFLE3_DBG_TX, "Stop: tx_bd_avail = %d\n", tx_bd_avail);
      QFLE3_DBG(QFLE3_DBG_TX, "Stop: prod = %d\n", txdata->tx_bd_prod);
      QFLE3_DBG(QFLE3_DBG_TX, "Stop: cons = %d\n", txdata->tx_bd_cons);
      QFLE3_DBG(QFLE3_DBG_TX, "Stop: pkt tx cnt %jd\n", fp->drv_stats.tx_pkts);
      //if (!IS_FCOE_FP(fp) && !cos){
      if (!IS_FCOE_FP(fp)){
         qfle3_tq_pause (adapter, qid);
      }
      if (copypkt)
         vmk_PktRelease(copypkt);
      vmk_SpinlockUnlock(fp->fp_lock);
      return (VMK_BUSY);
     } else if(copypkt) {
       vmk_PktRelease(pkt);
       pkt = copypkt;
       fp->drv_stats_priv.tx_pkt_padded++;
     }

   /*
    * set flag according to packet type (UNICAST_ADDRESS is default)
    */
   status = vmk_PktHeaderL2Find(pkt, &l2Header, NULL);
   if (status != VMK_OK)
      goto drop_pkt;

   status = vmk_PktHeaderDataGet(pkt, l2Header, (void **) &ethHdr);
   if (status != VMK_OK)
      goto drop_pkt;
   QFLE3_DBG(QFLE3_DBG_TX, "Eth header found @ [%p] ret %d\n", ethHdr, status);
   isMcast = qfle3_pktIsMcast(ethHdr->daddr);
   isBcast = qfle3_pktIsBcast(ethHdr->daddr);
   if (isBcast) {
      mac_type = BROADCAST_ADDRESS;
   } else if (isMcast) {
      mac_type = MULTICAST_ADDRESS;
   } else
      mac_type = UNICAST_ADDRESS;
   vmk_PktHeaderDataRelease(pkt, l2Header, ethHdr, VMK_FALSE);

   mustTSO = vmk_PktIsLargeTcpPacket(pkt);
   mustCsum = vmk_PktIsMustCsum(pkt);
   mustVlan = vmk_PktMustVlanTag(pkt);
   if (mustCsum || mustTSO)
   mustEncap = qfle3_lookup_encap(adapter,pkt, &tx_info, fp);

   total_pkt_size = vmk_PktFrameLenGet(pkt);
   if (mustEncap == XMIT_ENC_BAD_FRAME) {
      QFLE3_ERR("bad encap frame, drop\n");
      goto drop_pkt;
   }

   QFLE3_DBG(QFLE3_DBG_DEAD |QFLE3_DBG_TX | QFLE3_DBG_TX_SG, "XXX Tx Entr: tso[%d], csum[%d], vlan[%d], pkttype %d, "
	     "Bcast %d, Mcast %d, total_len %d, sgarry_len %d, seg_num %d, vxlan flag %d\n",
	     mustTSO, mustCsum, mustVlan, mac_type, isBcast, isMcast, total_pkt_size,
	     vmk_PktSgArrayTotalLenGet(pkt), nsegs, mustEncap);

   /* SV: Debug: FCoE LL2 tx*/
   if (fp->qid == FCOE_IDX(adapter)) {
      QFLE3_DBG(QFLE3_DBG_TX, "FCoE Tx: src-mac=0x%x %x %x %x %x %x\n",
         ethHdr->saddr[0], ethHdr->saddr[1], ethHdr->saddr[2],
         ethHdr->saddr[3], ethHdr->saddr[4], ethHdr->saddr[5]);
   }
   
   if (mustTSO) {
      if (total_pkt_size > QFLE3_TSO_MAX_SIZE) {
         QFLE3_ERR("bad tso size, drop\n");
	 goto drop_pkt;
      }
      tso_mss = vmk_PktGetLargeTcpPacketMss(pkt);
   }

   if (mustTSO && (nsegs < QFLE3_TSO_MAX_SEGMENTS)) {
      if (mustEncap == 0) {
         status = vmk_PktHeaderL4Find(pkt, &l4Header, NULL);
         if (VMK_UNLIKELY(status != VMK_OK))
             goto drop_pkt;
         status = vmk_PktHeaderL3Find(pkt, &l3Header, NULL);
         if (VMK_UNLIKELY(status != VMK_OK))
            goto drop_pkt;
         l4hoffset = l4Header->offset;
         l4hsize = vmk_PktHeaderLength(l4Header);
         hlen = l4hoffset + l4hsize;
      } else {
         hlen = tx_info.encap_packet_header_length;
      }
      if (VMK_LIKELY(qfle3_parse_tsowind(adapter, nsegs, pkt, tso_mss,
            hlen, mustEncap) == 0)) {
         goto tx_encap_continue; /* OK to send */
      }
   } else if ((mustEncap && (nsegs <= QFLE3_MAX_SEGMENTS - 1))
         || ((mustEncap == 0) && (nsegs <= QFLE3_MAX_SEGMENTS))) {
      goto tx_encap_continue; /* OK to send */
   }

   QFLE3_DBG(QFLE3_DBG_TX, "To defrag pkt %d\n", nsegs);
 
   /*
    * lets try to defragment this mbuf and remap it
    */
   status = vmk_PktDup(pkt, &flatPkt);
   if (VMK_UNLIKELY(status != VMK_OK)) {
      QFLE3_ERR("vmk_PktDup failed when defragmenting, drop\n");
      goto drop_pkt;
   }
      vmk_PktRelease(pkt);
      pkt = flatPkt;
      fp->drv_stats_priv.tx_pkt_padded++;
      nsegs = vmk_PktSgArrayGet(pkt)->numElems;

tx_encap_continue:

   total_seg_len = vmk_PktSgArrayTotalLenGet(pkt);
   if (VMK_UNLIKELY(total_pkt_size != total_seg_len))
      last_segment_pad_size = total_seg_len - total_pkt_size;
   /*
    * get the H/W pointer for packets and BDs
    */

   pkt_prod = txdata->tx_pkt_prod;
   bd_prod = txdata->tx_bd_prod;

   tx_buf = &txdata->txbuf_chain[TX_BD(pkt_prod)];
   VMK_ASSERT(tx_buf->mapType == QFLE3_MAP_INVALID);

   for (i = 0; i < nsegs; i++) {
      sgex = vmk_PktSgElemGet(pkt, i);
      status = qfle3_dma_map_ma(adapter, sgex->addr, sgex->length,
				VMK_DMA_DIRECTION_FROM_MEMORY, adapter->dmaEngine,
				&tx_buf->tx_segs[i].ioa);
      if (status != VMK_OK)
	 goto unmap_drop_pkt;
      tx_buf->tx_segs[i].len = sgex->length;
   }
   tx_buf->nsegs = nsegs;
   tx_buf->mapType = QFLE3_MAP_ELEM;

   /*
    * store the mbuf into the mbuf ring
    */
   tx_buf->pkt = pkt;
   tx_buf->first_bd = txdata->tx_bd_prod;
   tx_buf->flags = 0;

   /*
    * prepare the first transmit (start) BD for the mbuf
    */

   tx_start_bd = &txdata->tx_chain[TX_BD(bd_prod)].start_bd;
//   vmk_LogMessage("start bd vir %p, phy %p\n",tx_start_bd, (void *)(txdata->tx_chain_ioa + (tx_start_bd - &txdata->tx_chain[0].start_bd)));
//   vmk_LogMessage("hw bd cons %d, sw bd prod %d, sw bd cons %d\n",*txdata->tx_cons_sb, txdata->tx_bd_prod, txdata->tx_bd_cons);

   QFLE3_DBG(QFLE3_DBG_TX,
	     "sending pkt_prod=%u tx_buf=%p next_idx=%u bd=%u tx_start_bd=%p\n",
	     pkt_prod, tx_buf, txdata->tx_pkt_prod, bd_prod, tx_start_bd);

   tx_start_bd->addr_lo = htole32(U64_LO(tx_buf->tx_segs[0].ioa));
   tx_start_bd->addr_hi = htole32(U64_HI(tx_buf->tx_segs[0].ioa));
   tx_start_bd->nbytes = htole16(tx_buf->tx_segs[0].len);
   QFLE3_DBG(QFLE3_DBG_TX,
             "seg0 len %d, start_bd.nbytes %d\n", tx_buf->tx_segs[0].len, tx_start_bd->nbytes);
   tx_start_bd->bd_flags.as_bitfield = ETH_TX_BD_FLAGS_START_BD;

   tx_start_bd->general_data = (1 << ETH_TX_START_BD_HDR_NBDS_SHIFT);
   nbds = 1;

   vmk_VlanPriority priority;
   
   if (mustVlan) {
      vlanID = vmk_PktVlanIDGet(pkt);
      priority = vmk_PktPriorityGet(pkt);
      vlanID = (vlanID | ((vmk_uint16) priority) << 13);
      
      tx_start_bd->vlan_or_ethertype = htole16(vlanID);
      tx_start_bd->bd_flags.as_bitfield |=
	 (X_ETH_OUTBAND_VLAN << ETH_TX_BD_FLAGS_VLAN_MODE_SHIFT);
   } else {
      /*
       * used by FW for packet accounting
       */
      tx_start_bd->vlan_or_ethertype = htole16(txdata->tx_pkt_prod);
      QFLE3_DBG(QFLE3_DBG_TX, "vlan_or_ethertype = %d\n", htole16(txdata->tx_pkt_prod));
   }

   if (mustCsum || mustTSO) {
      tx_start_bd->bd_flags.as_bitfield |= ETH_TX_BD_FLAGS_L4_CSUM;

      if (mustEncap == 0) {
	 status = vmk_PktHeaderL4Find(pkt, &l4Header, NULL);
	 if (status != VMK_OK)
	    goto unmap_drop_pkt;
	 status = vmk_PktHeaderL3Find(pkt, &l3Header, NULL);
	 if (status != VMK_OK)
	    goto unmap_drop_pkt;
	 l4hoffset = l4Header->offset;
	 l4hsize = vmk_PktHeaderLength(l4Header);
	 qfle3_set_pbd_csum_e2(fp, pkt, l4hoffset, l4hsize, l4Header->type, &pbd_e2_parsing_data);
	 if (l3Header->type == VMK_PKT_HEADER_L3_IPv4) {
	    tx_start_bd->bd_flags.as_bitfield |= ETH_TX_BD_FLAGS_IP_CSUM;
	 }

	 if (l3Header->type == VMK_PKT_HEADER_L3_IPv6)  {
	    tx_start_bd->bd_flags.as_bitfield |= ETH_TX_BD_FLAGS_IPV6;
	 }

	 if (l4Header->type == VMK_PKT_HEADER_L4_UDP) {
	    tx_start_bd->bd_flags.as_bitfield |= ETH_TX_BD_FLAGS_IS_UDP;
	 }
      } else {
	 l4hoffset = tx_info.encap_packet_inner_L4_offset;
	 l4hsize = tx_info.encap_packet_inner_L4_length;
	 qfle3_set_pbd_csum_e2(fp, pkt, l4hoffset, l4hsize,
			       tx_info.encap_packet_inner_L4_protocol, &pbd_e2_parsing_data);

	 if (mustEncap & XMIT_ENC_INNER_V6)	{
	    tx_start_bd->bd_flags.as_bitfield |= ETH_TX_BD_FLAGS_IPV6;
	 }

	 if (mustEncap & XMIT_ENC_INNER_UDP) {
	    tx_start_bd->bd_flags.as_bitfield |= ETH_TX_BD_FLAGS_IS_UDP;
	 }
      }
      hlen = l4hoffset + l4hsize;
   }

   SET_FLAG(pbd_e2_parsing_data, ETH_TX_PARSE_BD_E2_ETH_ADDR_TYPE, mac_type);

   nbds++;
   bd_prod = TX_BD_NEXT(bd_prod);
   pbd_e2 = &txdata->tx_chain[TX_BD(bd_prod)].parse_bd_e2;
   vmk_Memset(pbd_e2, 0, sizeof(struct eth_tx_parse_bd_e2));

   if (adapter->flags & TX_SWITCHING)
       ecore_set_fw_mac_addr(&pbd_e2->data.mac_addr.dst_hi,
                             &pbd_e2->data.mac_addr.dst_mid,
                             &pbd_e2->data.mac_addr.dst_lo,
                             ethHdr->daddr);

   if (mustEncap && (mustCsum || mustTSO)) {
      /* add addition parse BD indication to start BD */
      SET_FLAG(tx_start_bd->general_data, ETH_TX_START_BD_PARSE_NBDS, 1);
      /* set encapsulation flag in start BD */
      SET_FLAG(tx_start_bd->general_data, ETH_TX_START_BD_TUNNEL_EXIST, 1);

      nbds++;
      bd_prod = TX_BD_NEXT(bd_prod);
      pbd_2nd_e2 = &txdata->tx_chain[TX_BD(bd_prod)].parse_2nd_bd;
      vmk_Memset(pbd_2nd_e2, 0, sizeof(struct eth_tx_parse_bd_e2));

      pbd_e2->data.tunnel_data.ip_hdr_start_inner_w = (tx_info.encap_packet_inner_ip_offset >> 1);

      if (mustTSO)
	 qfle3_update_pbds_tso_enc(pkt, pbd_e2, pbd_2nd_e2, mustEncap, &tx_info);
   }

   /*
    * setup the parsing BD with TSO specific info
    */
   if (adapter->trigger_error == QFLE3_TRIGGER_ASSERT){
      mustTSO = VMK_TRUE;
      
      QFLE3_ERR("Trigger ASSERT with mustTSO");
      tso_mss= 0x266b;
      adapter->trigger_error = 0;
   }
   if (mustTSO) {
      int curhdr = 0;
      vmk_uint16 rest_hlen = hlen;

      QFLE3_DBG(QFLE3_DBG_TX,"TSO start");
      qfle3_set_pbd_lso_e2(tso_mss, &pbd_e2_parsing_data);
      tx_start_bd->bd_flags.as_bitfield |= ETH_TX_BD_FLAGS_SW_LSO;
      do_split = qfle3_split_hdr(adapter, tx_buf, hlen, &need_hdrs);
      QFLE3_DBG(QFLE3_DBG_TX, "TSO chkhdr result: do_split = %d, hdrs %d",
                do_split, need_hdrs);
      /*fix HDR NBDS*/
      SET_FLAG(tx_start_bd->general_data, ETH_TX_START_BD_HDR_NBDS, need_hdrs);

      /*
       * split the first BD into header/data making the fw job easy
       */

      QFLE3_DBG(QFLE3_DBG_TX,"TSO split begin, hlen %d", rest_hlen);
      while (curhdr < need_hdrs) {
	 if (curhdr == 0) {
	    if (rest_hlen > tx_buf->tx_segs[curhdr].len) {
	       tx_start_bd->nbytes = htole16(tx_buf->tx_segs[curhdr].len);
	       rest_hlen -= tx_buf->tx_segs[curhdr].len;
	    } else {
	       tx_start_bd->nbytes = htole16(rest_hlen);
	    }
	 } else {
	    tx_hdr_bd = &txdata->tx_chain[TX_BD(bd_prod)].reg_bd;
	    tx_hdr_bd->addr_hi = htole32(U64_HI(tx_buf->tx_segs[curhdr].ioa));
	    tx_hdr_bd->addr_lo = htole32(U64_LO(tx_buf->tx_segs[curhdr].ioa));

	    if (rest_hlen > tx_buf->tx_segs[curhdr].len) {
	       tx_hdr_bd->nbytes = htole16(tx_buf->tx_segs[curhdr].len);
	       rest_hlen -= tx_buf->tx_segs[curhdr].len;
	    } else {
	       tx_hdr_bd->nbytes = htole16(rest_hlen);
	    }

	 }

	 curhdr++;
	 if (curhdr < need_hdrs) {
	    nbds++;
	    bd_prod = TX_BD_NEXT(bd_prod);
	    QFLE3_DBG(QFLE3_DBG_TX,"TSO split continue search hdrs, hlen %d", rest_hlen);
	    continue;
	 }

	 if (do_split == VMK_FALSE) {
	    QFLE3_DBG(QFLE3_DBG_TX,"hdrs %d, no need split", need_hdrs);
	    break;
	 }

	 nbds++;
	 bd_prod = TX_BD_NEXT(bd_prod);
	 /*
	  * new transmit BD after the tx_parse_bd or rest hdr bd.
	  */
         tx_data_bd = &txdata->tx_chain[TX_BD(bd_prod)].reg_bd;
	 tx_data_bd->addr_hi = htole32(U64_HI(tx_buf->tx_segs[curhdr - 1].ioa + rest_hlen));
	 tx_data_bd->addr_lo = htole32(U64_LO(tx_buf->tx_segs[curhdr - 1].ioa + rest_hlen));
	 tx_data_bd->nbytes = htole16(tx_buf->tx_segs[curhdr - 1].len - rest_hlen);
	 if (tx_total_pkt_size_bd == NULL) {
	    tx_total_pkt_size_bd = tx_data_bd;
	 }

	 QFLE3_DBG(QFLE3_DBG_TX,
		   "TSO split header size is %d (%x:%x) nbds %d\n",
		   le16toh(tx_start_bd->nbytes),
		   le32toh(tx_start_bd->addr_hi), le32toh(tx_start_bd->addr_lo), nbds);
      }
      mustEncap ? fp->drv_stats.encap_stats.encap_expected_tso++ : fp->drv_stats.expected_tso++;
      mustEncap ? adapter->drv_stats.encap_stats.encap_expected_tso++ : adapter->drv_stats.expected_tso++;
   }

   if (mustCsum) {
      mustEncap ? fp->drv_stats.encap_stats.encap_expected_cso++ : fp->drv_stats.expected_cso++;
      mustEncap ? adapter->drv_stats.encap_stats.encap_expected_cso++ : adapter->drv_stats.expected_cso++;
   }

   if (pbd_e2_parsing_data) {
      QFLE3_DBG(QFLE3_DBG_TX, "fill parsing bd data %x\n", htole32(pbd_e2_parsing_data));
      pbd_e2->parsing_data = htole32(pbd_e2_parsing_data);
   }

   /*
    * prepare remaining BDs, start tx bd contains first seg/frag
    */

   for (i = need_hdrs; i < nsegs; i++) {
      nbds++;
      bd_prod = TX_BD_NEXT(bd_prod);
      tx_data_bd = &txdata->tx_chain[TX_BD(bd_prod)].reg_bd;
      tx_data_bd->addr_lo = htole32(U64_LO(tx_buf->tx_segs[i].ioa));
      tx_data_bd->addr_hi = htole32(U64_HI(tx_buf->tx_segs[i].ioa));
      if (i == (nsegs - 1))
	 tx_data_bd->nbytes = htole16(tx_buf->tx_segs[i].len - last_segment_pad_size);
      else
	 tx_data_bd->nbytes = htole16(tx_buf->tx_segs[i].len);
      tx_data_bd->total_pkt_bytes = 0;
      if (tx_total_pkt_size_bd == NULL) {
	 tx_total_pkt_size_bd = tx_data_bd;
      }
   }

   tx_start_bd->nbd = htole16(nbds);
   QFLE3_DBG(QFLE3_DBG_TX, "tx_total_ndbs %d\n", nbds);

   if (tx_total_pkt_size_bd != NULL) {
      tx_total_pkt_size_bd->total_pkt_bytes = htole16(total_pkt_size);
   } else {
      tx_start_bd->nbytes = htole16(tx_buf->tx_segs[0].len - last_segment_pad_size);
   }
   /*BD Debug code */
   {
      vmk_uint8 n_hdrs = tx_start_bd->general_data & ETH_TX_START_BD_HDR_NBDS;
      vmk_uint16 rest_bds = nbds;
      vmk_uint16 tmp_bd = tx_buf->first_bd;

      /*Start BD*/
      QFLE3_DBG(QFLE3_DBG_TX | QFLE3_DBG_TX_SG,
		"XXX Tx Strt: %p bd#=%d nbd=%d vlan=%d "
		"ipcsum %x, L4csum %x, vlan_mode %x, startbd %x, isudp %x, sw_lso %x, ipv6 %x, hdr_nbyts=%d, low %x\n",
		tx_start_bd,
		tmp_bd,
		le16toh(tx_start_bd->nbd),
		le16toh(tx_start_bd->vlan_or_ethertype),
		tx_start_bd->bd_flags.as_bitfield & ETH_TX_BD_FLAGS_IP_CSUM,
		(tx_start_bd->bd_flags.as_bitfield & ETH_TX_BD_FLAGS_L4_CSUM) >> 1,
		(tx_start_bd->bd_flags.as_bitfield & ETH_TX_BD_FLAGS_VLAN_MODE) >> 2,
		(tx_start_bd->bd_flags.as_bitfield & ETH_TX_BD_FLAGS_START_BD) >> 4,
		(tx_start_bd->bd_flags.as_bitfield & ETH_TX_BD_FLAGS_IS_UDP) >> 5,
		(tx_start_bd->bd_flags.as_bitfield & ETH_TX_BD_FLAGS_SW_LSO) >> 6,
		(tx_start_bd->bd_flags.as_bitfield & ETH_TX_BD_FLAGS_IPV6) >> 7,
		le16toh(tx_start_bd->nbytes),
		tx_start_bd->addr_lo
	 );
      QFLE3_DBG(QFLE3_DBG_TX | QFLE3_DBG_TX_SG,
		"XXX        : general_data: hdr_nbds %d, no_added_tags %x, force_vlan %x, parse_nbds %d, tunnel_exist %x\n",
		(tx_start_bd->general_data & ETH_TX_START_BD_HDR_NBDS),
		(tx_start_bd->general_data & ETH_TX_START_BD_NO_ADDED_TAGS) >> 3,
		(tx_start_bd->general_data & ETH_TX_START_BD_FORCE_VLAN_MODE) >> 4,
		(tx_start_bd->general_data & ETH_TX_START_BD_PARSE_NBDS) >> 5,
		(tx_start_bd->general_data & ETH_TX_START_BD_TUNNEL_EXIST) >> 7
	 );
      rest_bds--;

      /* pbd_e2 */
      tmp_bd = TX_BD_NEXT(tmp_bd);
      QFLE3_DBG(QFLE3_DBG_TX | QFLE3_DBG_TX_SG,
		"XXX Tx Pars: %p bd#=%d, ip_hdr_start_inner_w %d, flags %d, L4_hdr_offsetW %d, tcp_hdr_lenW %d, ipv6_exthdr %d, lso_mss %d, eth_addr_type 0x%x\n",
		pbd_e2,
		tmp_bd,
		pbd_e2->data.tunnel_data.ip_hdr_start_inner_w,
		pbd_e2->data.tunnel_data.flags,
		pbd_e2->parsing_data & ETH_TX_PARSE_BD_E2_L4_HDR_START_OFFSET_W,
		(pbd_e2->parsing_data & ETH_TX_PARSE_BD_E2_TCP_HDR_LENGTH_DW) >> 11,
		(pbd_e2->parsing_data & ETH_TX_PARSE_BD_E2_IPV6_WITH_EXT_HDR) >> 15,
		(pbd_e2->parsing_data & ETH_TX_PARSE_BD_E2_LSO_MSS) >> 16,
		(pbd_e2->parsing_data & ETH_TX_PARSE_BD_E2_ETH_ADDR_TYPE) >> 30
	 );
      rest_bds--;

      /* pbd_2nd_e2 */
      if(mustEncap) {
	 tmp_bd = TX_BD_NEXT(tmp_bd);
	 QFLE3_DBG(QFLE3_DBG_TX | QFLE3_DBG_TX_SG,
		   "XXX Tx 2Par: %p bd#=%d, outer_ip_hdr_start W 0x%x, tunnel udp 0x%x, outer_ip_hdr_len %x, bdtype %d, tcpflags 0x%x\n",
		   pbd_2nd_e2,
		   tmp_bd,
		   pbd_2nd_e2->global_data & ETH_TX_PARSE_2ND_BD_IP_HDR_START_OUTER_W,
		   (pbd_2nd_e2->global_data & ETH_TX_PARSE_2ND_BD_TUNNEL_UDP_EXIST) >> 7,
		   (pbd_2nd_e2->global_data & ETH_TX_PARSE_2ND_BD_IP_HDR_LEN_OUTER_W) >> 8, pbd_2nd_e2->bd_type, pbd_2nd_e2->tcp_flags);
	 QFLE3_DBG(QFLE3_DBG_TX | QFLE3_DBG_TX_SG,
		   "XXX        : tunnel_udp_hdr_start w %d, fw_ip_hdr_to_payload w 0x%x, fw_ip_csum_wo_len_flags_frag 0x%x, hw_ip_id %d, tcp_send_seq 0x%x\n",
		   pbd_2nd_e2->tunnel_udp_hdr_start_w,
		   pbd_2nd_e2->fw_ip_hdr_to_payload_w,
		   pbd_2nd_e2->fw_ip_csum_wo_len_flags_frag,
		   pbd_2nd_e2->hw_ip_id,
		   pbd_2nd_e2->tcp_send_seq);
	 rest_bds--;
      }

      for (i = 0; i < n_hdrs - 1; i++) {
	 tmp_bd = TX_BD_NEXT(tmp_bd);
	 tx_hdr_bd = &txdata->tx_chain[TX_BD(tmp_bd)].reg_bd;
	 QFLE3_DBG(QFLE3_DBG_TX | QFLE3_DBG_TX_SG,
		   "XXX Tx Hdrs: %p bd#=%d, nbytes=%d hi=0x%x lo: 0x%x\n",
		   tx_hdr_bd,
		   tmp_bd,
		   le16toh(tx_hdr_bd->nbytes),
		   le32toh(tx_hdr_bd->addr_hi), le32toh(tx_hdr_bd->addr_lo));
	 rest_bds--;
      }
      tmp_bd = TX_BD_NEXT(tmp_bd);
      while (rest_bds--) {
	 tx_data_bd = &txdata->tx_chain[TX_BD(tmp_bd)].reg_bd;
	 QFLE3_DBG(QFLE3_DBG_TX | QFLE3_DBG_TX_SG,
		   "XXX Tx Frag: %p bd#=%d, totalbytes=%d nbytes=%d hi=0x%x lo: 0x%x\n",
		   tx_data_bd,
		   tmp_bd,
		   le16toh(tx_data_bd->total_pkt_bytes),
		   le16toh(tx_data_bd->nbytes),
		   le32toh(tx_data_bd->addr_hi), le32toh(tx_data_bd->addr_lo));
	 tmp_bd = TX_BD_NEXT(tmp_bd);
      }
   }
   QFLE3_DBG(QFLE3_DBG_TX | QFLE3_DBG_TX_SG, "XXX\n");
   QFLE3_DBG(QFLE3_DBG_TX, "doorbell: cid=%d nbds=%d bd=%u\n", txdata->cid, nbds, bd_prod);
   /*
    * update TX BD producer index value for next TX
    */
   bd_prod = TX_BD_NEXT(bd_prod);

   /*
    * If the chain of tx_bd's describing this frame is adjacent to or spans
    * an eth_tx_next_bd element then we need to increment the nbds value.
    */
   if (TX_BD_IDX(bd_prod) < nbds) {
      nbds++;
   }

   txdata->tx_db.data.prod += nbds;

   /*
    * producer points to the next free tx_bd at this point
    */
   txdata->tx_pkt_prod++;
   vmk_CPUMemFenceReadWrite();
   DOORBELL(adapter, txdata->cid, txdata->tx_db.raw);
   vmk_CPUMemFenceReadWrite();
   QFLE3_DBG(QFLE3_DBG_TX,"Writing tx prod %d for CID %d\n", txdata->tx_db.data.prod, txdata->cid);
   txdata->tx_bd_prod += nbds;
   vmk_CPUMemFenceReadWrite();
   adapter->drv_stats.tx_pkts++;
   fp->drv_stats.tx_pkts++;
   fp->tx_stats.tx_pkts++;
   adapter->drv_stats.tx_bytes += total_pkt_size;
   fp->drv_stats.tx_bytes += total_pkt_size;
   vmk_SpinlockUnlock(fp->fp_lock);
   adapter->drv_stats_priv.tx_pkt_accepted++;
   return (VMK_OK);
  unmap_drop_pkt:
   qfle3_unmap_txbuf(adapter, tx_buf, TX_BD(pkt_prod), qid);
  drop_pkt:
   vmk_PktRelease(pkt);
   fp->drv_stats.tx_Drops++;
   adapter->drv_stats.tx_Drops++;
   vmk_SpinlockUnlock(fp->fp_lock);
   return (VMK_OK);
}

static VMK_ReturnStatus
qfle3_xmit_pktlist(qfle3_adapter * adapter, vmk_PktList pktList,
		   vmk_UplinkQueueID vmkqid)
{
   vmk_uint32 qid;
   VMK_ReturnStatus status = VMK_OK;
   vmk_PktHandle *pkt;

   qid = vmk_UplinkQueueIDVal(vmkqid);
   if (qid >= QFLE3_NUM_TX_ETH_QUEUES(adapter)) {
      vmk_PktListReleaseAllPkts(pktList);
      goto out;
   }

   while (!vmk_PktListIsEmpty(pktList)) {
      if (VMK_UNLIKELY(qfle3_tq_stopped(adapter, qid))) {

//         QFLE3_ERR("Return BUSY Q stopped fp[%d]\n", qid);
	 status = VMK_BUSY;
	 goto out;
      }
      pkt = vmk_PktListPopFirstPkt(pktList);
      status = qfle3_xmit_pkt(adapter, qid, pkt);

      if (VMK_UNLIKELY(status != VMK_OK)) {
         /*
          * Place the pkt back to the pktList if not successfully
          * handled by driver, so that net-scheduler will later
          * re-submit the pkt for transmission.
          */
         vmk_PktListPrependPkt(pktList, pkt);
         
         QFLE3_DBG(QFLE3_DBG_UPLINK, "Return BUSY qfle3_xmit_pkt didn't complete fp[%d]", qid);
	 goto out;
      }
   }
out:
   return status;
}


static VMK_ReturnStatus

qfle3_uplink_tx(vmk_AddrCookie driverData, vmk_PktList pktList)
{
   vmk_PktHandle *pkt;
   vmk_UplinkQueueID vmkqid;
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING)){
//      QFLE3_DBG(QFLE3_DBG_UPLINK, "Uplink is Resetting, Rejecting Pkt Tx");
      return VMK_BUSY;
   }

   pkt = vmk_PktListGetFirstPkt(pktList);
   vmkqid = vmk_PktQueueIDGet(pkt);

   return qfle3_xmit_pktlist(adapter, pktList, vmkqid);
}

static VMK_ReturnStatus
qfle3_uplink_mtu_set(vmk_AddrCookie driverData, vmk_uint32 mtu)
{
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   vmk_uint16 sm_state;
//	   vmk_UplinkSharedData *sd = &adapter->uplinkSharedData;
   QFLE3_SMCMD_STATUS cmd_status;
   vmk_uint32 rc;
	
   QFLE3_DBG(QFLE3_DBG_KERNEL, "Uplink.mtu_set old %u new %u Adapter state is %x", adapter->mtu, mtu, adapter->state->vector[0]);
	
   if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING)){
      QFLE3_DBG(QFLE3_DBG_UPLINK, "Uplink is Resetting, Rejecting MTU change");
      return VMK_FAILURE;
   }
   sm_state = qfle3_sm_getstate(adapter);
   QFLE3_DBG(QFLE3_DBG_SM,  "Uplink.uplinkMTUSet state is %s", qfle3_sm_get_string(sm_state));
	
   if (mtu < QFLE3_MIN_MTU || mtu > QFLE3_MAX_MTU)
      return VMK_FAILURE;
	
   if (adapter->mtu == mtu)
      return VMK_OK;
	
	
   // Check to make sure we are in IOSTARTED STATE AND that the adapter is not in
   // the process of resetting, otherwise, cache the value and return to process it
   // at a later time;  for now we'll only check the adapter state and deal with
   // reset later
   adapter->uplinkCachedData.mtu = mtu;
   adapter->uplinkCachedNewData |= QFLE3_UPLINK_MTU_CHANGED;
   QFLE3_INFO("MTU set, caching mtu change \n");
   QFLE3_DBG(QFLE3_DBG_SM, "uplinkCachedNewdata 0x%x\n", adapter->uplinkCachedNewData);
	
   if (vmk_BitVectorAtomicTestAndSet(adapter->state, QFLE3_STATE_BIT_RESETTING)) {
      
      QFLE3_ERR("Uplink.mtu_set adapter resetting");
      return VMK_OK;
   }

   qfle3_link_down(adapter);
   cmd_status = qfle3_sm_cmd(adapter, QFLE3_SMCMD_UPLINKRESET, 0);

   if (cmd_status == QFLE3_SMCMD_STATUS_COMPLETED){
      if (SHMEM2_HAS(adapter,curr_cfg)) {
         if (IS_MF_BD(adapter)) {
            rc = qfle3_mfw_command(adapter, DRV_MSG_CODE_CONFIG_CHANGE,
               DRV_MSG_CODE_CONFIG_CHANGE_MTU_SIZE);
            if (rc == FW_MSG_CODE_CONFIG_CHANGE_DONE)
               SHMEM2_WR(adapter, curr_cfg, CURR_CFG_MET_OS);
         }
      }
      
      QFLE3_INFO("Uplink.mtu_set complete");
      return VMK_OK;
   }else if (cmd_status == QFLE3_SMCMD_STATUS_WRONGSTATE){
   	 QFLE3_ERR("Setting MTU in the wrong state %s", qfle3_sm_get_string(sm_state));
   	 return VMK_FAILURE;
   }else {
   	 QFLE3_ERR("ERROR Setting MTU");
   	 return VMK_FAILURE;
   }
}

static VMK_ReturnStatus
qfle3_uplink_state_set(vmk_AddrCookie driverData, vmk_UplinkState state)
{
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   vmk_UplinkSharedData *sd = &adapter->uplinkSharedData;
   vmk_uint16 sm_state;

   QFLE3_DBG(QFLE3_DBG_KERNEL, "Uplink.state_set %s, %s,%s,%s",
	     (state & VMK_UPLINK_STATE_ENABLED)? "On":"Off",
	     (state & VMK_UPLINK_STATE_PROMISC)? "Promisc":"",
	     (state & VMK_UPLINK_STATE_BROADCAST_OK)? "rx Brodcast":"",
	     (state & VMK_UPLINK_STATE_MULTICAST_OK)? "rx Multicast":"");

   if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING)){
      QFLE3_ERR("Uplink is Resetting, Rejecting State set");
      return VMK_FAILURE;
   }
   sm_state = qfle3_sm_getstate(adapter);
   QFLE3_DBG(QFLE3_DBG_SM, "Uplink.uplinkStateSet state is %s\n", qfle3_sm_get_string(sm_state));

   QFLE3_SHARED_DATA_WRITE_BEGIN(adapter);
   sd->state = state;
   QFLE3_SHARED_DATA_WRITE_END(adapter);

   if (state == (VMK_UPLINK_STATE_PROMISC |
		 VMK_UPLINK_STATE_BROADCAST_OK |
		 VMK_UPLINK_STATE_ENABLED |
		 VMK_UPLINK_STATE_MULTICAST_OK)) {
      adapter->rx_mode = QFLE3_RX_MODE_ALLMULTI;
   } else if (state == (VMK_UPLINK_STATE_BROADCAST_OK |
			VMK_UPLINK_STATE_ENABLED |
			VMK_UPLINK_STATE_MULTICAST_OK)) {
      adapter->rx_mode = QFLE3_RX_MODE_NORMAL;
   }
   qfle3_set_rx_mode(adapter);

   QFLE3_DBG(QFLE3_DBG_KERNEL, "Uplink.state_set Done");
   return VMK_OK;

}

VMK_ReturnStatus
qfle3_uplink_stats_get(vmk_AddrCookie driverData, vmk_UplinkStats * stats)
{
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_QUIESCED)){
      QFLE3_ERR("Unable to retrieve uplink stats");
      return VMK_OK;
   }
    //qfle3_stats_handle(adapter, STATS_EVENT_UPDATE);
    QFLE3_DBG(QFLE3_DBG_KERNEL, "Uplink.stats_get");
   vmk_Memset(stats, 0, sizeof(vmk_UplinkStats));
   stats->rxPkts = adapter->drv_stats.rx_pkts;
   stats->txPkts = adapter->drv_stats.tx_pkts;
   stats->rxBytes = adapter->drv_stats.rx_bytes;
   stats->txBytes = adapter->drv_stats.tx_bytes;
   stats->rxErrors = adapter->drv_stats.rx_Errors;
   stats->txErrors = adapter->drv_stats.tx_Errors;
   stats->rxDrops = adapter->drv_stats.rx_Drops;
   stats->txDrops = adapter->drv_stats.tx_Drops;
   stats->rxMulticastPkts = adapter->drv_stats.rx_MulticastPkts;
   stats->txMulticastPkts = adapter->drv_stats.tx_MulticastPkts;
   stats->rxBroadcastPkts = adapter->drv_stats.rx_BroadcastPkts;
   stats->txBroadcastPkts = adapter->drv_stats.tx_BroadcastPkts;
   stats->collisions = adapter->drv_stats.collisions;
   stats->rxLengthErrors = adapter->drv_stats.rx_LengthErrors;
   stats->rxOverflowErrors = adapter->drv_stats.rx_OverflowErrors;
   stats->rxCRCErrors = adapter->drv_stats.rx_CRCErrors;
   stats->rxFrameAlignErrors = adapter->drv_stats.rx_FrameAlignErrors;
   stats->rxFifoErrors = adapter->drv_stats.rx_FifoErrors;
   stats->rxMissErrors = adapter->drv_stats.rx_MissErrors;
   stats->txAbortedErrors = adapter->drv_stats.tx_AbortedErrors;
   stats->txCarrierErrors = adapter->drv_stats.tx_CarrierErrors;
   stats->txFifoErrors = adapter->drv_stats.tx_FifoErrors;
   stats->txHeartbeatErrors = adapter->drv_stats.tx_HeartbeatErrors;
   stats->txWindowErrors = adapter->drv_stats.tx_WindowErrors;
   QFLE3_DBG(QFLE3_DBG_KERNEL, "Uplink.stats_get, Done");
   return VMK_OK;
}


static VMK_ReturnStatus
qfle3_message_level_get(vmk_AddrCookie driverData, vmk_uint32 * level)
{
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   //QFLE3_DBG(QFLE3_DBG_UPLINK, "Uplink.message_level_get");
   *level = adapter->debug_mask;
   return VMK_OK;
}

static VMK_ReturnStatus
qfle3_message_level_set(vmk_AddrCookie driverData, vmk_uint32 level)
{
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   //QFLE3_ERR("Uplink.message_level_set %u", level);
   adapter->debug_mask = level;
   return VMK_OK;
}


static vmk_UplinkMessageLevelOps qfle3_message_level_ops = {
   .getMessageLevel = qfle3_message_level_get,
   .setMessageLevel = qfle3_message_level_set,
};



VMK_ReturnStatus qfle3_pauseparamsget(vmk_AddrCookie driverData,vmk_UplinkPauseParams *params)
{
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   struct elink_params *link_params = &adapter->link_params;
   int cfx_idx = qfle3_get_link_cfg_idx(adapter);
   vmk_uint16 cfg_reg;
	
   vmk_Memset(params, 0, sizeof(vmk_UplinkPauseParams));
   QFLE3_DBG(QFLE3_DBG_UPLINK, "qfle3_pauseparamsget, cfg_idx %d\n", cfx_idx);
	
   params->autoNegotiate = (link_params->req_flow_ctrl[cfx_idx] == ELINK_FLOW_CTRL_AUTO);
	
   if (!params->autoNegotiate)
      cfg_reg = link_params->req_flow_ctrl[cfx_idx];
   else
      cfg_reg = link_params->req_fc_auto_adv;
	
   params->rxPauseEnabled = ((cfg_reg & ELINK_FLOW_CTRL_RX) == ELINK_FLOW_CTRL_RX);
   params->txPauseEnabled = ((cfg_reg & ELINK_FLOW_CTRL_TX) == ELINK_FLOW_CTRL_TX);
	
   QFLE3_DBG(QFLE3_DBG_UPLINK, "qfle3_pauseparamsget:  autoneg %d rx_pause %d tx_pause %d\n",
	     params->autoNegotiate, params->rxPauseEnabled, params->txPauseEnabled);

   return VMK_OK;
}
void qfle3_acquire_phy_lock(struct qfle3_adapter *adapter);
void qfle3_release_phy_lock(struct qfle3_adapter *adapter);

VMK_ReturnStatus qfle3_pauseparamsset(vmk_AddrCookie driverData, vmk_UplinkPauseParams params)
{
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   int cfx_idx = qfle3_get_link_cfg_idx(adapter);

   QFLE3_DBG(QFLE3_DBG_UPLINK, "qfle3_pauseparamsget:  autoneg %d rx_pause %d tx_pause %d, cfg_idx %d\n",
      params.autoNegotiate, params.rxPauseEnabled, params.txPauseEnabled, cfx_idx);

   adapter->link_params.req_flow_ctrl[cfx_idx] = ELINK_FLOW_CTRL_AUTO;

   if (params.rxPauseEnabled)
      adapter->link_params.req_flow_ctrl[cfx_idx] |= ELINK_FLOW_CTRL_RX;

   if (params.txPauseEnabled)
      adapter->link_params.req_flow_ctrl[cfx_idx] |= ELINK_FLOW_CTRL_TX;

   if (adapter->link_params.req_flow_ctrl[cfx_idx] == ELINK_FLOW_CTRL_AUTO)
      adapter->link_params.req_flow_ctrl[cfx_idx] = ELINK_FLOW_CTRL_NONE;

   if (params.autoNegotiate) {
      if (!(adapter->port.supported[cfx_idx] & ELINK_SUPPORTED_Autoneg)) {
         QFLE3_DBG(QFLE3_DBG_UPLINK, "autoneg not supported\n");
         return VMK_FAILURE;
      }

      if (adapter->link_params.req_line_speed[cfx_idx] == ELINK_SPEED_AUTO_NEG) {
         adapter->link_params.req_flow_ctrl[cfx_idx] = ELINK_FLOW_CTRL_AUTO;
      }
      adapter->link_params.req_fc_auto_adv = 0;

      if (params.rxPauseEnabled)
         adapter->link_params.req_fc_auto_adv |= ELINK_FLOW_CTRL_RX;

      if (params.txPauseEnabled)
         adapter->link_params.req_fc_auto_adv |= ELINK_FLOW_CTRL_TX;

      if (!adapter->link_params.req_fc_auto_adv)
         adapter->link_params.req_fc_auto_adv = ELINK_FLOW_CTRL_NONE;

   }

   if (qfle3_sm_getstate(adapter) >= QFLE3_SM_ACTIVATED){
      QFLE3_DBG(QFLE3_DBG_STATS, "Executing qfle3_stats_handle for STATS_EVENT_STOP\n");
      qfle3_stats_handle(adapter, STATS_EVENT_STOP);
      qfle3_force_link_reset(adapter);
      qfle3_link_set(adapter);
//		qfle3_acquire_phy_lock(adapter);
//		qfle3_report_link(adapter);
//		qfle3_release_phy_lock(adapter);
   }

   return VMK_OK;
}

static vmk_UplinkPauseParamsOps qfle3_pauseparam = {
   .pauseParamsGet = qfle3_pauseparamsget,
   .pauseParamsSet = qfle3_pauseparamsset,
};



VMK_ReturnStatus qfle3_getwolstate(vmk_AddrCookie driverData,vmk_UplinkWolState *params)
{
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;

   vmk_Memset(params, 0, sizeof(vmk_UplinkWolState));

   QFLE3_DBG(QFLE3_DBG_UPLINK, "qfle3_getwolstate\n");
   if (adapter->flags & NO_WOL_FLAG) {
      params->supported = 0;
      params->enabled = 0;
   } else {
      params->supported = VMK_UPLINK_WAKE_ON_MAGIC;
      if (adapter->wol)
	 params->enabled = VMK_UPLINK_WAKE_ON_MAGIC;
      else
	 params->enabled = 0;
   }
   vmk_Memset(params->secureONPassword, 0, VMK_UPLINK_WOL_STRING_MAX);
   return VMK_OK;
}
VMK_ReturnStatus qfle3_setwolstate(vmk_AddrCookie driverData, vmk_UplinkWolState *params)
{
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   vmk_uint32 rc;
   QFLE3_DBG(QFLE3_DBG_UPLINK, "qfle3_setwolstate\n");

   if (params->enabled & ~VMK_UPLINK_WAKE_ON_MAGIC) {
      QFLE3_DBG(QFLE3_DBG_UPLINK, "WOL not supported\n");
      return VMK_FAILURE;
   }
   if (params->enabled & VMK_UPLINK_WAKE_ON_MAGIC) {
      if (adapter->flags & NO_WOL_FLAG) {
	 QFLE3_DBG(QFLE3_DBG_UPLINK, "WOL not supported\n");
	 return VMK_FAILURE;
      }
      adapter->wol = 1;
   } else
      adapter->wol = 0;

   if (SHMEM2_HAS(adapter, curr_cfg)) {
      if (IS_MF_BD(adapter)) {
         rc = qfle3_mfw_command(adapter, DRV_MSG_CODE_CONFIG_CHANGE, 
            DRV_MSG_CODE_CONFIG_CHANGE_WOL_ENA);

         if (rc == FW_MSG_CODE_CONFIG_CHANGE_DONE)
            SHMEM2_WR(adapter, curr_cfg, CURR_CFG_MET_OS);
      }
   }

   return VMK_OK;
}


static vmk_UplinkWOLOps qfle3_wol = {
   .getWOLState = qfle3_getwolstate,
   .setWOLState = qfle3_setwolstate,
};

VMK_ReturnStatus qfle3_getcoalesceparam(vmk_AddrCookie driverData, vmk_UplinkCoalesceParams *params)
{
   qfle3_adapter *adapter = (qfle3_adapter *)driverData.ptr;
   vmk_Memset(params, 0, sizeof(vmk_UplinkCoalesceParams));
   params->rxUsecs = adapter->rx_ticks;
   params->txUsecs = adapter->tx_ticks;

   params->rxMaxFrames = 0;     /* FIXME */
   params->txMaxFrames = 0;

   return VMK_OK;
}
VMK_ReturnStatus qfle3_setcoalesceparam(vmk_AddrCookie driverData, vmk_UplinkCoalesceParams *params)
{
   qfle3_adapter *adapter = (qfle3_adapter *)driverData.ptr;

   if (params->useAdaptiveRx || params->useAdaptiveTx || params->rateSampleInterval) {
      QFLE3_INFO("Adaptive RX or TX coalescing algorithm not supported\n");
      return VMK_NOT_SUPPORTED;
   }
   if (params->rxMaxFrames || params->txMaxFrames){
      QFLE3_INFO("Packet Count Based TX and RX Coalescing not supported\n");
      return VMK_NOT_SUPPORTED;
   }

   if (params->rxUsecs > QFLE3_MAX_COALESCE_TOUT || params->txUsecs > QFLE3_MAX_COALESCE_TOUT) {
      QFLE3_INFO("Can't support requested coalesce value, MAX supported 0x%x\n", QFLE3_MAX_COALESCE_TOUT);
      return VMK_NOT_SUPPORTED;
   }

   if ((qfle3_sm_getstate(adapter) >= QFLE3_SM_ACTIVATED) &&
      ((adapter->rx_ticks != (vmk_uint16)params->rxUsecs) ||
      (adapter->tx_ticks != (vmk_uint16)params->txUsecs))) {
         adapter->rx_ticks = (vmk_uint16)params->rxUsecs;
         adapter->tx_ticks = (vmk_uint16)params->txUsecs;
         qfle3_update_coalesce(adapter);
   }

   return VMK_OK;
}

static vmk_UplinkCoalesceParamsOps qfle3_coalesce = {
   .getParams = qfle3_getcoalesceparam,
   .setParams = qfle3_setcoalesceparam,
};

VMK_ReturnStatus qfle3_ringparamsget(vmk_AddrCookie driverData, vmk_UplinkRingParams *params)
{
   qfle3_adapter *adapter = (qfle3_adapter *)driverData.ptr;
   vmk_Memset(params, 0, sizeof(vmk_UplinkRingParams));
   params->rxMaxPending =  (RX_BD_USABLE_PER_PAGE*RX_BD_MAX_PAGES);

   if(adapter->rx_ring_size)
      params->rxPending = adapter->rx_ring_size;
   else
      params->rxPending = RX_BD_USABLE;


   params->txMaxPending = (TX_BD_USABLE_PER_PAGE*TX_BD_MAX_PAGES);
   if(adapter->tx_ring_size)
      params->txPending = adapter->tx_ring_size;
   else
      params->txPending = TX_BD_USABLE;

   return VMK_OK;

}

static VMK_ReturnStatus qfle3_ringparamsset(vmk_AddrCookie driverData, 
   vmk_UplinkRingParams *params)
{
   qfle3_adapter *adapter = (qfle3_adapter *)driverData.ptr;
   vmk_uint16 sm_state;
   QFLE3_SMCMD_STATUS cmd_status;
   int i;

   QFLE3_DBG(QFLE3_DBG_UPLINK, "set ring paramsters rx_pending %d, tx_pending %d", 
      params->rxPending, params->txPending);

   sm_state = qfle3_sm_getstate(adapter);

   if ((params->rxPending > (RX_BD_TOTAL_PER_PAGE * RX_BD_MAX_PAGES)) || 
      (params->rxPending < (RX_BD_TOTAL_PER_PAGE * RX_BD_MIN_PAGES)) ||
       (params->txPending > (TX_BD_TOTAL_PER_PAGE * TX_BD_MAX_PAGES)) || 
       (params->txPending < (TX_BD_TOTAL_PER_PAGE * TX_BD_MIN_PAGES))) {
      QFLE3_DBG(QFLE3_DBG_UPLINK, "Command parameters not supported rxPending %d, txPending %d\n", 
         params->rxPending, params->txPending);
      return VMK_FAILURE;
   }
   if ((adapter->rx_ring_size == ((params->rxPending / RX_BD_TOTAL_PER_PAGE) * RX_BD_TOTAL_PER_PAGE)) &&
       (adapter->tx_ring_size == ((params->txPending / TX_BD_TOTAL_PER_PAGE) * TX_BD_TOTAL_PER_PAGE)))
      return VMK_OK;

   adapter->uplinkCachedData.rx_ring_page_num = params->rxPending / RX_BD_TOTAL_PER_PAGE;
   for (i = RX_BD_MAX_PAGES; i >= RX_BD_MIN_PAGES ; i = i/2){
      if (adapter->uplinkCachedData.rx_ring_page_num >= i) {
         adapter->uplinkCachedData.rx_ring_page_num = i;
         break;
      }	
   }
   adapter->uplinkCachedData.tx_ring_page_num = params->txPending / TX_BD_TOTAL_PER_PAGE;
   for (i = TX_BD_MAX_PAGES; i >= TX_BD_MIN_PAGES ; i = i/2){
      if (adapter->uplinkCachedData.tx_ring_page_num >= i) {
         adapter->uplinkCachedData.tx_ring_page_num = i;
         break;
      }
   }
//	   adapter->uplinkCachedData.rx_ring_page_num = 16;
//	   adapter->uplinkCachedData.tx_ring_page_num = 32;
   adapter->uplinkCachedNewData |= QFLE3_UPLINK_RINGSIZE_CHANGED;
   QFLE3_DBG(QFLE3_DBG_UPLINK, "Setting Rx Ring Size to %d pages, TX Ring Size to %d pages.\n", 
      adapter->uplinkCachedData.rx_ring_page_num, adapter->uplinkCachedData.tx_ring_page_num);
   
   if (sm_state == QFLE3_SM_IOSTARTED)
   {
      if (vmk_BitVectorAtomicTestAndSet(adapter->state, QFLE3_STATE_BIT_RESETTING)) {
         return VMK_OK;
      }
      qfle3_link_down(adapter);
      cmd_status = qfle3_sm_cmd(adapter, QFLE3_SMCMD_UPLINKRESET, 0);
      if (cmd_status == QFLE3_SMCMD_STATUS_COMPLETED) {
         return VMK_OK;
      } else {
         return VMK_FAILURE;
      }
   }
   return VMK_OK;
}


static vmk_UplinkRingParamsOps qfle3_ring = {
   .ringParamsGet = qfle3_ringparamsget,
   .ringParamsSet = qfle3_ringparamsset,
};

VMK_ReturnStatus
qfel3_get_regdump_len(vmk_AddrCookie drv_data, vmk_uint32 *len)
{
   qfle3_adapter *adapter = (qfle3_adapter *)drv_data.ptr;
   *len = DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0;
   QFLE3_DBG(QFLE3_DBG_UPLINK, "Len = %u bytes.\n", *len);
   return VMK_OK;
}

VMK_ReturnStatus
qfel3_dump_regs(vmk_AddrCookie drv_data, vmk_AddrCookie buf)
{
   qfle3_adapter *adapter = (qfle3_adapter *)drv_data.ptr;
   void *fwdmp_buf = (void *)buf.ptr;
   struct chip_core_dmp *dmp;
   VMK_ReturnStatus status = VMK_OK;
   extern void disable_pause(struct qfle3_adapter *adapter);

   if (fwdmp_buf == NULL){
      return VMK_FAILURE;
   }
   if (adapter->fwdmp_buf) {
      QFLE3_INFO("Retriving previously saved GRC dump");
      
      if (adapter->fwdmp_flags & GRC_DMP_ANY_ENABLED) {
       if (!(adapter->fwdmp_flags & GRC_DMP_COLLECT_VALID)) {
          /* Dump is not saved. */
            
            QFLE3_INFO("Attempt to dump saved GRC, none saved");
          status = VMK_NOT_FOUND;
       }
       dmp = (struct chip_core_dmp *)adapter->fwdmp_buf;
       vmk_Memcpy(fwdmp_buf, dmp, dmp->fw_hdr.dmp_size);

       /* Free the GRC dump buffer. */
       vmk_HeapFree(qfle3_mod_info.heapID,adapter->fwdmp_buf);
       adapter->fwdmp_flags &= ~GRC_DMP_COLLECT_VALID;
       QFLE3_INFO("GRC-dump is retrieved & freed at 0x%p.\n",
              adapter->fwdmp_buf);
       adapter->fwdmp_buf = NULL;
       return status;
      }
      return VMK_NOT_FOUND;
   }
      
   QFLE3_INFO("Collecting LIVE GRC dump, 0x%x bytes", DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0);
   vmk_Memset(fwdmp_buf, 0, DRV_DUMP_CRASH_DMP_BUF_SIZE_E3B0);
   dmp = (struct chip_core_dmp *)fwdmp_buf;
   qfle3_get_grcdump(adapter, fwdmp_buf, VMK_TRUE);
   /* Re-enable parity attentions */
   ecore_clear_blocks_parity(adapter);
   ecore_enable_blocks_parity(adapter);
   disable_pause(adapter);

   QFLE3_INFO("Writing %u bytes for regdump.\n", dmp->fw_hdr.dmp_size);
      
   return VMK_OK;
}

static vmk_UplinkRegDumpOps qfel3_regdump_ops = {
   .regDumpLenGet  = qfel3_get_regdump_len,
   .regDump        = qfel3_dump_regs,
};

static VMK_ReturnStatus
qfle3_linkstatus_setcb(vmk_AddrCookie driverData,
		       vmk_LinkStatus *linkInfo)
{
   vmk_LinkSpeed speed;
   struct elink_phy *phy;
   //VMK_ReturnStatus status;
   vmk_UplinkSharedData *sd;
   vmk_uint32 advertising, cfg_idx, phy_idx;
   qfle3_adapter *adapter = (qfle3_adapter *)driverData.ptr;

   QFLE3_INFO("Uplink.linkstatus_setcb %s speed %d, duplex %s", 
      (linkInfo->state == VMK_LINK_STATE_DOWN) ? "down": "up",
      linkInfo->speed,
      (linkInfo->duplex == VMK_LINK_DUPLEX_AUTO) ? "auto": ((linkInfo->duplex == VMK_LINK_DUPLEX_HALF) ? "half": "full"));

   if (linkInfo->state == VMK_LINK_STATE_DOWN) {
      qfle3_link_down(adapter);
      QFLE3_DBG(QFLE3_DBG_UPLINK, "Taking down link");
      
      if (adapter->flags & MCP_SUPPORTS_S_CHANNEL_DOWN){
         
         QFLE3_DBG(QFLE3_DBG_UPLINK, "Setting S Channel Down");
         qfle3_mfw_command(adapter, DRV_MSG_CODE_UPDATE_DRIVER_STATE, DRV_MSG_CODE_DRIVER_STATE_DISABLED);
      }
      if (QFLE3_SMCMD_STATUS_COMPLETED != qfle3_sm_cmd(adapter, QFLE3_SMCMD_PAUSEIO, 0)){
         QFLE3_ERR("Unable to bring down link");
         return VMK_FAILURE;
      }
//      if (adapter->sm_state == QFLE3_SM_ACTIVATED){
//         qfle3_sm_cmd(adapter, QFLE3_SMCMD_STOP, 0);
//      }
      QFLE3_DBG(QFLE3_DBG_KERNEL, "Uplink.linkstatus_setcb %s Done", (linkInfo->state == VMK_LINK_STATE_DOWN) ? "down": "up");

      return VMK_OK;
   }

   QFLE3_DBG(QFLE3_DBG_UPLINK, "Bringing up link");
//      qfle3_sm_cmd(adapter, QFLE3_SMCMD_START, 0);
   if (QFLE3_SMCMD_STATUS_COMPLETED != qfle3_sm_cmd(adapter, QFLE3_SMCMD_RESUMEIO, 0)) {
      QFLE3_ERR("Failed to bring up link");
      return VMK_FAILURE;
   }
   if (adapter->flags & MCP_SUPPORTS_S_CHANNEL_DOWN){
      QFLE3_DBG(QFLE3_DBG_UPLINK, "Setting S Channel Up");
      qfle3_mfw_command(adapter, DRV_MSG_CODE_UPDATE_DRIVER_STATE, DRV_MSG_CODE_DRIVER_STATE_ACTIVE);
   }
   speed = linkInfo->speed;

   if (IS_MF_SD(adapter))
      return 0;

#ifdef QFLE3_NPAR
   if (IS_MF_SI(adapter)) {
      vmk_uint32 part;
      vmk_uint32 line_speed = adapter->link_vars.line_speed;

      /* use 10G if no link detected */
      if (!line_speed)
	 line_speed = 10000;

#if 0
      if (adapter->common.bc_ver < REQ_BC_VER_4_SET_MF_BW) {
	 QFLE3_DBG(QFLE3_DBG_LINK,
		   "To set speed BC %X or higher is required, please upgrade BC\n",
		   REQ_BC_VER_4_SET_MF_BW);
	 return VMK_NOT_SUPPORTED;
      }
#endif
    if(adapter->pending_max) {
    	part = adapter->pending_max;
       	adapter->pending_max = 0;
    }else
      part = (speed * 100) / line_speed;

      if (line_speed < speed || !part) {
	 QFLE3_DBG(QFLE3_DBG_LINK,
		   "Speed setting should be in a range from 1%% to 100%% of actual line speed\n");
	 return VMK_NOT_SUPPORTED;
      }

	 qfle3_update_max_mf_config(adapter, part);

      return 0;
   }
#endif

   cfg_idx = qfle3_get_link_cfg_idx(adapter);
   phy_idx = qfle3_get_cur_phy_idx(adapter);
   phy = &adapter->link_params.phy[phy_idx];

   QFLE3_DBG(QFLE3_DBG_LINK, "cfg_idx = %x\n", cfg_idx);

   if (linkInfo->duplex == VMK_LINK_DUPLEX_AUTO &&
       linkInfo->speed == VMK_LINK_SPEED_AUTO) {

      vmk_uint32 an_supported_speed = adapter->port.supported[cfg_idx];
      if (adapter->link_params.phy[ELINK_EXT_PHY1].type ==
	  PORT_HW_CFG_XGXS_EXT_PHY_TYPE_BCM84833)
	 an_supported_speed |= (ELINK_SUPPORTED_100baseT_Half |
				ELINK_SUPPORTED_100baseT_Full);

      if (!(adapter->port.supported[cfg_idx] & ELINK_SUPPORTED_Autoneg)) {
	 QFLE3_DBG(QFLE3_DBG_LINK, "Autoneg not supported\n");
	 return VMK_NOT_SUPPORTED;
      }

      adapter->link_params.req_line_speed[cfg_idx] = ELINK_SPEED_AUTO_NEG;

      adapter->link_params.req_duplex[cfg_idx] = DUPLEX_FULL;

      adapter->port.advertising[cfg_idx] |= (ELINK_ADVERTISED_Autoneg);

   } else { /* forced speed */
      /* advertise the requested speed and duplex if supported */
      switch (speed) {
      case VMK_LINK_SPEED_10_MBPS:
	 if (linkInfo->duplex == VMK_LINK_DUPLEX_FULL) {
	    if (!(adapter->port.supported[cfg_idx] &
		  ELINK_SUPPORTED_10baseT_Full)) {
	       QFLE3_WARN("10M full not supported\n");
	       return VMK_NOT_SUPPORTED;
	    }

	    advertising = (ELINK_ADVERTISED_10baseT_Full |
			   ELINK_ADVERTISED_TP);
	 } else {
	    if (!(adapter->port.supported[cfg_idx] &
		  ELINK_SUPPORTED_10baseT_Half)) {
	       QFLE3_DBG(QFLE3_DBG_LINK,
			 "10M half not supported\n");
	       return VMK_NOT_SUPPORTED;
	    }

	    advertising = (ELINK_ADVERTISED_10baseT_Half |
			   ELINK_ADVERTISED_TP);
	 }
	 break;

      case VMK_LINK_SPEED_100_MBPS:
	 if (linkInfo->duplex == VMK_LINK_DUPLEX_FULL) {
	    if (!(adapter->port.supported[cfg_idx] &
		  ELINK_SUPPORTED_100baseT_Full)) {
	       QFLE3_DBG(QFLE3_DBG_LINK,
			 "100M full not supported\n");
	       return VMK_NOT_SUPPORTED;
	    }

	    advertising = (ELINK_ADVERTISED_100baseT_Full |
			   ELINK_ADVERTISED_TP);
	 } else {
	    if (!(adapter->port.supported[cfg_idx] &
		  ELINK_SUPPORTED_100baseT_Half)) {
	       QFLE3_DBG(QFLE3_DBG_LINK,
			 "100M half not supported\n");
	       return VMK_NOT_SUPPORTED;
	    }

	    advertising = (ELINK_ADVERTISED_100baseT_Half |
			   ELINK_ADVERTISED_TP);
	 }
	 break;

      case VMK_LINK_SPEED_1000_MBPS:
	 if (linkInfo->duplex != VMK_LINK_DUPLEX_FULL) {
	    QFLE3_DBG(QFLE3_DBG_LINK,
		      "1G half duplex not supported\n");
	    return VMK_NOT_SUPPORTED;
	 }

	 if ((adapter->port.supported[cfg_idx] &
	     ELINK_SUPPORTED_1000baseT_Full) &&
		((phy->media_type == ELINK_ETH_PHY_BASE_T) ||
				(phy->media_type == ELINK_ETH_PHY_KR) ||
				(phy->media_type == ELINK_ETH_PHY_SFP_1G_FIBER))) {
			advertising = (ELINK_ADVERTISED_1000baseT_Full |
					ELINK_ADVERTISED_TP);
	 } else if (adapter->port.supported[cfg_idx] &
		    ELINK_SUPPORTED_1000baseKX_Full) {
	    advertising = ELINK_ADVERTISED_1000baseKX_Full;
	 } else {
	    QFLE3_DBG(QFLE3_DBG_LINK,
		      "1G speed not supported\n");
	    return VMK_NOT_SUPPORTED;
	 }

	 break;

      case 2500:
	 if (linkInfo->duplex != VMK_LINK_DUPLEX_FULL) {
	    QFLE3_DBG(QFLE3_DBG_LINK,
		      "2.5G half not supported\n");
	    return VMK_NOT_SUPPORTED;
	 }

	 if (!(adapter->port.supported[cfg_idx]
	       & ELINK_SUPPORTED_2500baseX_Full)) {
	    QFLE3_DBG(QFLE3_DBG_LINK,
		      "2.5G full not supported\n");
	    return VMK_NOT_SUPPORTED;
	 }

	 advertising = (ELINK_ADVERTISED_2500baseX_Full |
			ELINK_ADVERTISED_TP);
	 break;

      case VMK_LINK_SPEED_10000_MBPS:
	 if (linkInfo->duplex != VMK_LINK_DUPLEX_FULL) {
	    QFLE3_DBG(QFLE3_DBG_LINK,
		      "10G half not supported\n");
	    return VMK_NOT_SUPPORTED;
	 }

	 if ((adapter->port.supported[cfg_idx] &
	      ELINK_SUPPORTED_10000baseT_Full) &&
	     (adapter->link_params.phy[phy_idx].media_type !=
	      ELINK_ETH_PHY_SFP_1G_FIBER)) {
	    advertising = (ELINK_ADVERTISED_10000baseT_Full |
			   ELINK_ADVERTISED_FIBRE);
	 } else if (adapter->port.supported[cfg_idx] &
		    ELINK_SUPPORTED_10000baseKR_Full) {
	    advertising = (ELINK_ADVERTISED_10000baseKR_Full |
			   ELINK_ADVERTISED_FIBRE);
	 } else {
	    QFLE3_DBG(QFLE3_DBG_LINK,
		      "10G full not supported\n");
	    return VMK_NOT_SUPPORTED;
	 }

	 break;

      default:
	 QFLE3_WARN("Unsupported speed %u\n", speed);
	 return VMK_NOT_SUPPORTED;
      }

      adapter->link_params.req_line_speed[cfg_idx] = speed;
      adapter->link_params.req_duplex[cfg_idx] =
             (linkInfo->duplex == VMK_LINK_DUPLEX_FULL ?
              DUPLEX_FULL : DUPLEX_HALF);
      adapter->port.advertising[cfg_idx] = advertising;
   }

   QFLE3_DBG(QFLE3_DBG_LINK, "req_line_speed %d\n"
	     "  req_duplex %d  advertising 0x%x\n",
	     adapter->link_params.req_line_speed[cfg_idx],
	     adapter->link_params.req_duplex[cfg_idx],
	     adapter->port.advertising[cfg_idx]);

   QFLE3_DBG(QFLE3_DBG_STATS, "Executing qfle3_stats_handle for STATS_EVENT_STOP\n");
   qfle3_stats_handle(adapter, STATS_EVENT_STOP);
   qfle3_force_link_reset(adapter);
   qfle3_link_set(adapter);

   sd = &adapter->uplinkSharedData;
   QFLE3_SHARED_DATA_WRITE_BEGIN(adapter);
   vmk_Memcpy(&sd->link, linkInfo, sizeof(*linkInfo));
   QFLE3_SHARED_DATA_WRITE_END(adapter);
   return VMK_OK;
}

static VMK_ReturnStatus
qfle3_vxlanPortUpdate(vmk_AddrCookie cookie,
		      vmk_uint16 portNBO)
{
   int rc;
   int status = VMK_OK;
   qfle3_adapter *adapter = (qfle3_adapter *)cookie.ptr;
   struct ecore_func_state_params func_params = {NULL};
   struct ecore_func_switch_update_params *switch_update_params =
           &func_params.params.switch_update;

   QFLE3_DBG(QFLE3_DBG_UPLINK, "Update vxlan port to %u", vmk_BE16ToCPU(portNBO));

   SET_BIT(RAMROD_COMP_WAIT, &func_params.ramrod_flags);
   SET_BIT(RAMROD_RETRY, &func_params.ramrod_flags);

   func_params.f_obj = &adapter->func_obj;
   func_params.cmd = ECORE_F_CMD_SWITCH_UPDATE;

   /* Function parameters */
   SET_BIT(ECORE_F_UPDATE_TUNNEL_CFG_CHNG,
           &switch_update_params->changes);

   switch_update_params->vxlan_dst_port = vmk_BE16ToCPU(portNBO);

   if (adapter->vxlan_filters_en) {
      SET_BIT(ECORE_F_UPDATE_TUNNEL_INNER_CLSS_VXLAN,
              &switch_update_params->changes);
      SET_BIT(ECORE_F_UPDATE_TUNNEL_INNER_CLSS_VXLAN_INNER_VNI,
	      &switch_update_params->changes);
   }

   rc = ecore_func_state_change(adapter, &func_params);
   if (rc) {
      QFLE3_ERR("failed to change vxlan dst port to %d (rc = 0x%x)\n",
                portNBO, rc);
      status = VMK_FAILURE;	
   }

   return status;
}

#if (VMKAPI_REVISION >= VMK_REVISION_FROM_NUMBERS(2, 5, 0, 0))
static vmk_UplinkVXLANOffloadParams qfle3EncapOffloadOps = {
   .vxlanPortUpdate = qfle3_vxlanPortUpdate,
   .flags           = (VMK_UPLINK_VXLAN_FLAG_INNER_IPV4_CSO |
                       VMK_UPLINK_VXLAN_FLAG_INNER_IPV4_TSO |
                       VMK_UPLINK_VXLAN_FLAG_INNER_IPV6_CSO |
                       VMK_UPLINK_VXLAN_FLAG_INNER_IPV6_TSO),
};
#else
static vmk_UplinkEncapOffloadOps qfle3EncapOffloadOps = {
   .vxlanPortUpdate = qfle3_vxlanPortUpdate,

};
#endif

/*
 * callbacks for NetDump and firmware dump
 */

/*
***********************************************************************
* qfle3_panicTx --

*
* \brief Handler used by vmkernel to send packets
*
* \note  This handler is called when vmkernel is in panic state. For
*        TX completion, driver must call asynchronous vmkapi
*        vmk_PktListRelease() or vmk_PktRelease inside
*        vmk_UplinkNetDumpPanicPollCB().  It should not call
*        vmk_NetPollQueueCompPkt() or request other asynchronous task
*        to perform the completion.
*
* \param[in]   driverData   Points to the internal module structure for
*                           the device associated to the uplink. Before
*                           calling vmk_DeviceRegister, device driver
*                           needs to assign this pointer to member
*                           driverData in structure vmk_UplinkRegData.
* \param[in]   pktList      List of packets to transmit
*
* \retval      VMK_OK       If transmit succeed
* \retval      Other        If transmit failed
***********************************************************************
*/
VMK_ReturnStatus
qfle3_panicTx(vmk_AddrCookie driverData, vmk_PktList pktList)
{
   qfle3_adapter *adapter = (qfle3_adapter *)driverData.ptr;
   vmk_UplinkQueueID vmkqid;
   vmk_PktHandle *pkt;

   if (QFLE3_PANIC_FWDMP_ENABLED(adapter))
      qfle3_disable_fwdmp(adapter);
 
   QFLE3_SHARED_DATA_READ_BEGIN(adapter);
   vmkqid = adapter->uplinkSharedData.queueInfo->defaultTxQueueID;
   QFLE3_SHARED_DATA_READ_END(adapter);

   VMK_PKTLIST_ITER_STACK_DEF(iter);
   /* Netdump pkt doesn't have queue ID set */
   for(vmk_PktListIterStart(iter, pktList);
       vmk_PktListIterIsAtEnd(iter) != VMK_TRUE; vmk_PktListIterMove(iter)){
      pkt = vmk_PktListIterGetPkt(iter);
      vmk_PktQueueIDSet(pkt, vmkqid);
      //QFLE3_ERR("qfle3_panicTx set pkt qid %d\n", vmk_UplinkQueueIDVal(vmkqid));
   }

   return qfle3_uplink_tx(driverData, pktList);
}

/*
***********************************************************************
* qfle3_panicPoll --

*
* \brief Handler used by vmkernel to poll for packets received by
*        the device associated to an uplink. Might be ignored.
*
* \note  This handler is called when vmkernel is in panic state. Driver
*        should not call vmk_NetPollRxPktQueue to queue any RX packets.
*        Instead, it must insert RX packets into pktList parameter
*        and return to vmkernel.
*
* \note  Driver must perform all TX completion in this callback by
*        calling vmk_PktListRelease or vmk_PktRelease, since this
*        callback is guaranteed to be called more frequently than
*        vmk_UplinkNetDumpPanicTxCB.
*
* \param[in]   clientData   Points to the internal module structure
*                           returned by callback function
*                           vmk_UplinkNetDumpPanicInfoGetCB(). It can be
*                           different from the driver data specified in
*                           vmk_UplinkRegData during device registration.
* \param[out]  pktList      List of packets received
*
* \retval      VMK_OK       Always
***********************************************************************
*/

VMK_ReturnStatus
qfle3_panicPoll(vmk_AddrCookie driverData, vmk_PktList pktList)
{
   qfle3_adapter *adapter = (qfle3_adapter *)driverData.ptr;
   int i, qid;
   /* panic tx only use the default tx queue, so only poll the default tx queue. */
   struct qfle3_fastpath *txq = QFLE3_GET_FP_FROM_QID(adapter,QFLE3_DEFAULT_TX_QID(adapter));

   struct qfle3_fastpath *rxq;
   vmk_uint32 rxDone = 0;
   vmk_uint32 txDone = 0;
   u8 cos;

   if (QFLE3_PANIC_FWDMP_ENABLED(adapter))
      qfle3_disable_fwdmp(adapter);

   /* Only Poll the default TX queue */
   if (qfle3_has_tx_work(txq)){
      /* Netqueue has been enabled so delete this assert */
      /* VMK_ASSERT(txq->qid == 1); */
      for_each_cos_in_tx_queue(txq, cos) {
         txDone += qfle3_txeof(adapter, txq->txdata_ptr[cos], 1024, VMK_TRUE);
      }
      //QFLE3_ERR("qfle3_panicPoll calls qfle3_txeof, txDone %d\n", txDone);
   }
   FOREACH_RX_ETH_QUEUE(i,qid,rxq) 
      // Todo, when netqueue related state flag is ready it will poll all rx queue
      if (i == 0){
   	 if(qfle3_has_rx_work(rxq)){
   	    rxDone = qfle3_rxeof(adapter, rxq, 1024, NULL, VMK_TRUE, pktList);
   	    //QFLE3_ERR("qfle3_panicPoll calls qfle3_rxeof, rxDone %d\n", rxDone);
   	 }
      }
   }

   return VMK_OK;
}


VMK_ReturnStatus qfle3_panicInfoGet(vmk_AddrCookie driverData,
				    vmk_UplinkPanicInfo *panicInfo)
{
   qfle3_adapter *adapter = (qfle3_adapter *)driverData.ptr;

   panicInfo->clientData = driverData;
   QFLE3_INFO("qfle3_panicInfoGet called\n");

   return VMK_OK;
}


static vmk_UplinkNetDumpOps qfle3_netdump_ops = {
   .panicTx = qfle3_panicTx,
   .panicPoll = qfle3_panicPoll,
   .panicInfoGet = qfle3_panicInfoGet,
};


/*
*****************************************************************************
*
* qfle3_priv_stats_get_len
*
*     Get length of Private stats from the driver
*
*  Parameters:
*     vmk_AddrCookie: private driver data pointing to the adapter
*     length: length of stats buffer
*
*  Results:
*     VMK_ReturnStatus
*
*  Side effects:
*     None
*
*****************************************************************************
*/
VMK_ReturnStatus
qfle3_priv_stats_get_len(vmk_AddrCookie driverData, //IN
			 vmk_ByteCount *length)     //IN/OUT
{
   VMK_ASSERT(length != NULL);

   /* This value is presently not considered due to
    * a bug in vmware.
    */
   *length = QFLE3_STATS_SIZE;
   return VMK_OK;
}



/*
*****************************************************************************
*
* qfle3_priv_stats_get
*
*     Get Private stats from the driver
*
*  Parameters:
*     driverData: private driver data pointing to the adapter
*     statsBuf: statistic buffer to copy stats
*     length: length of buffer
*
*  Results:
*     VMK_ReturnStatus
*
*  Side effects:
*     None
*
*****************************************************************************
*/

VMK_ReturnStatus
qfle3_priv_stats_get(vmk_AddrCookie driverData,  //IN
		     char *buffer,                 //IN/OUT
		     vmk_ByteCount length)       //IN
{
   qfle3_adapter *adapter = (qfle3_adapter *)driverData.ptr;
   struct qfle3_fastpath *fp;
   qfle3_drv_stats_priv_t *st;
   vmk_uint32 idx = 0;
   vmk_uint32 i = 0, qid, relative_txq_idx = 0;
   vmk_uint32 leadrss = 0;

   char *buf = NULL;

   buf = qfle3_heap_alloc(QFLE3_STATS_SIZE);
   if (buf == NULL) {
      return VMK_FAILURE;
   }
   if (length > QFLE3_STATS_SIZE) {
      length = QFLE3_STATS_SIZE;
   }

   /* Print Driver level statistics. */
   idx += vmk_Sprintf(buf+ idx, "\n         ** qfle3 Driver Private Stats **\n");
   /* Leading rss queue */
   FOREACH_RX_ETH_QUEUE(i,qid,fp)
      if( fp->is_leading_rss == 1) {
	 st = &fp->drv_stats_priv;
	 leadrss = 1;
	 idx += vmk_Sprintf(buf+ idx, "leading rss queue rssq[%d] rx_bytes: %lu\n", i, st->rssq_rx_bytes);
      }
   }
   if (leadrss == 0){
      idx += vmk_Sprintf(buf+ idx, "NO leading rss queue rssq[0] rx_bytes: 0\n");
   }

   /* Hidden RSS queue starting from fp[num_rxqs_vmk] to fp[num_rxqs_drv] */
   FOREACH_RSS_QUEUE(i,qid,fp)
      st = &fp->drv_stats_priv;
      idx += vmk_Sprintf(buf+ idx, "hidden  rss queue rssq[%d] rx_bytes: %lu\n", i, st->rssq_rx_bytes);
   }

   /*rx queues*/
   FOREACH_RX_ETH_QUEUE(i,qid,fp)
      idx += vmk_Sprintf(buf+ idx, "rxq[%d] rx_pkts: %lu\n", i, fp->drv_stats.rx_pkts);
      idx += vmk_Sprintf(buf+ idx, "rxq[%d] rx_bytes: %lu\n", i, fp->drv_stats.rx_bytes);
      idx += vmk_Sprintf(buf+ idx, "rxq[%d] rx_Errors: %lu\n", i, fp->drv_stats.rx_Errors);
      idx += vmk_Sprintf(buf+ idx, "rxq[%d] rx_Drops: %lu\n", i, fp->drv_stats.rx_Drops);
      idx += vmk_Sprintf(buf+ idx, "rxq[%d] tpa_rx_bytes: %lu\n", i, fp->drv_stats_priv.tpa_rx_bytes);
      idx += vmk_Sprintf(buf+ idx, "rxq[%d] tpa_rx_pkts: %lu\n", i, fp->drv_stats_priv.tpa_rx_pkts);
   }

   /*tx queueus*/
   
   idx += vmk_Sprintf(buf+ idx, "tso segments over limit: %lu\n", 
                                        adapter->drv_stats_priv.tso_seg_overlimit);
   FOREACH_TX_ETH_QUEUE(i,qid,fp)
      //relative_txq_idx = i - adapter->num_rxqs_drv;
      st = &fp->drv_stats_priv;
      relative_txq_idx = i;
      idx += vmk_Sprintf(buf+ idx, "txq[%d] tx_pkts: %lu\n", relative_txq_idx, fp->drv_stats.tx_pkts);
      idx += vmk_Sprintf(buf+ idx, "txq[%d] tx_bytes: %lu\n", relative_txq_idx, fp->drv_stats.tx_bytes);
      idx += vmk_Sprintf(buf+ idx, "txq[%d] tx_Errors: %lu\n", relative_txq_idx, fp->drv_stats.tx_Errors);
      idx += vmk_Sprintf(buf+ idx, "txq[%d] tx_Drops: %lu\n", relative_txq_idx, fp->drv_stats.tx_Drops);
      idx += vmk_Sprintf(buf+ idx, "txq[%d] tx_padded: %lu\n", relative_txq_idx, st->tx_pkt_padded);
      idx += vmk_Sprintf(buf+ idx, "txq[%d] expected_tso: %lu\n", relative_txq_idx,
                                           fp->drv_stats.expected_tso);
      idx += vmk_Sprintf(buf+ idx, "txq[%d] expected_cso: %lu\n", relative_txq_idx,
                                           fp->drv_stats.expected_cso);
      idx += vmk_Sprintf(buf+ idx, "txq[%d] vxlan_tx_pkts: %lu\n", relative_txq_idx, 
                                           fp->drv_stats.encap_stats.vxlan_tx_pkts);
      idx += vmk_Sprintf(buf+ idx, "txq[%d] geneve_tx_pkts: %lu\n", relative_txq_idx, 
                                           fp->drv_stats.encap_stats.geneve_tx_pkts);
      idx += vmk_Sprintf(buf+ idx, "txq[%d] encap_expected_tso: %lu\n", relative_txq_idx,
                                           fp->drv_stats.encap_stats.encap_expected_tso);
      idx += vmk_Sprintf(buf+ idx, "txq[%d] encap_expected_cso: %lu\n", relative_txq_idx,
                                           fp->drv_stats.encap_stats.encap_expected_cso);
      
      idx += vmk_Sprintf(buf+ idx, "txq[%d] tso segments over limit: %lu\n", relative_txq_idx, 
                                           st->tso_seg_overlimit);
   }
   /* copy to vmkernel owned memory */
   vmk_Memcpy(buffer, buf, length);

   /* free local heap memory for stats */
   qfle3_heap_free(buf);

   return VMK_OK;
}

#define ETHERNET_PACKET_HEADER_SIZE (14)
#define ETHERNET_VLAN_ENCAP_LEN (4)
#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
vmk_UplinkTSOConstraints tso_cons = {
   /* tx_bd_nbytes is 16-bit */
   .maxL2PayloadSize = QFLE3_TSO_MAX_SIZE - (ETHERNET_PACKET_HEADER_SIZE +
					     ETHERNET_VLAN_ENCAP_LEN),
   .maxSegmentCount  = QFLE3_TSO_MAX_SEGMENTS - 1,
   .maxSegmentSize   = 0,
};

static VMK_ReturnStatus
qfle3_adv_modes_get(vmk_AddrCookie driverData,
		vmk_UplinkAdvertisedMode *modes,
		vmk_uint32 *numModes)
{
	qfle3_adapter *adapter = (qfle3_adapter *)driverData.ptr;
	vmk_uint32 i = 0, num;
	int cfg_idx = qfle3_get_link_cfg_idx(adapter);
	int phy_idx = qfle3_get_cur_phy_idx(adapter);
	struct elink_phy *phy;
	vmk_LinkMediaType media;

	phy = &adapter->link_params.phy[phy_idx];

	switch (phy->media_type) {
		case ELINK_ETH_PHY_SFPP_10G_FIBER:
			media = VMK_LINK_MEDIA_BASE_SFI;
			break;
		case ELINK_ETH_PHY_KR:
			if (phy->supported & (ELINK_SUPPORTED_20000baseKR2_Full) )
				media = VMK_LINK_MEDIA_BASE_KR2;
			else if (phy->supported & (ELINK_SUPPORTED_20000baseMLD2_Full) )
				media = VMK_LINK_MEDIA_BASE_MLD2;
			else 
				media = VMK_LINK_MEDIA_BASE_KR;
			break;
		case ELINK_ETH_PHY_DA_TWINAX:
			media = VMK_LINK_MEDIA_BASE_TWINAX;
			break;
		case ELINK_ETH_PHY_BASE_T:
			media = VMK_LINK_MEDIA_BASE_T;
			break;
		case ELINK_ETH_PHY_NOT_PRESENT:
			media = VMK_LINK_MEDIA_NONE;
			break;
		case ELINK_ETH_PHY_CX4:
		case ELINK_ETH_PHY_SFP_1G_FIBER:
		case ELINK_ETH_PHY_XFP_FIBER:
		case ELINK_ETH_PHY_UNSPECIFIED:
		default:
			media = VMK_LINK_MEDIA_UNKNOWN;
			break;
	}

	if (adapter->link_params.req_line_speed[cfg_idx] == ELINK_SPEED_AUTO_NEG) {
		adapter->advModes[i].speed = VMK_LINK_SPEED_AUTO;
		adapter->advModes[i].duplex = VMK_LINK_DUPLEX_AUTO;
		adapter->advModes[i].media = media;
		i++;
	}

	if (adapter->port.supported[cfg_idx] & ELINK_SUPPORTED_10baseT_Half) {
		adapter->advModes[i].speed = VMK_LINK_SPEED_10_MBPS;
		adapter->advModes[i].duplex = VMK_LINK_DUPLEX_HALF;
		adapter->advModes[i].media = media;
		i++;
	}

	if (adapter->port.supported[cfg_idx] & ELINK_SUPPORTED_10baseT_Full) {
		adapter->advModes[i].speed = VMK_LINK_SPEED_10_MBPS;
		adapter->advModes[i].duplex = VMK_LINK_DUPLEX_FULL;
		adapter->advModes[i].media = media;
		i++;
	}

	if (adapter->port.supported[cfg_idx] & ELINK_SUPPORTED_100baseT_Half) {
		adapter->advModes[i].speed = VMK_LINK_SPEED_100_MBPS;
		adapter->advModes[i].duplex = VMK_LINK_DUPLEX_HALF;
		adapter->advModes[i].media = media;
		i++;
	}

	if (adapter->port.supported[cfg_idx] & ELINK_SUPPORTED_100baseT_Full) {
		adapter->advModes[i].speed = VMK_LINK_SPEED_100_MBPS;
		adapter->advModes[i].duplex = VMK_LINK_DUPLEX_FULL;
		adapter->advModes[i].media = media;
		i++;
	}

#if 0
	//TODO find out if we support Half duplex on 1000MBPS.
	if (adapter->port.supported[cfg_idx] & ELINK_SUPPORTED_1000baseT_Half) {
		adapter->supportedModes[i].speed = VMK_LINK_SPEED_1000_MBPS;
		adapter->supportedModes[i].duplex = VMK_LINK_DUPLEX_HALF;
		adapter->advModes[i].media = media;
		i++;
	}
#endif

	if (adapter->port.supported[cfg_idx] & ELINK_SUPPORTED_1000baseT_Full) {
		if ((phy->media_type == ELINK_ETH_PHY_BASE_T) ||
				(phy->media_type == ELINK_ETH_PHY_KR) ||
				(phy->media_type == ELINK_ETH_PHY_SFP_1G_FIBER)) {
			adapter->advModes[i].speed = VMK_LINK_SPEED_1000_MBPS;
			adapter->advModes[i].duplex = VMK_LINK_DUPLEX_FULL;
			adapter->advModes[i].media = media;
			i++;
		}	
	}

	if (adapter->port.supported[cfg_idx] & ELINK_SUPPORTED_2500baseX_Full) {
		adapter->advModes[i].speed = VMK_LINK_SPEED_2500_MBPS;
		adapter->advModes[i].duplex = VMK_LINK_DUPLEX_FULL;
		adapter->advModes[i].media = media;
		i++;
	}

	if (adapter->port.supported[cfg_idx] & ELINK_SUPPORTED_10000baseT_Full) {
		adapter->advModes[i].speed = VMK_LINK_SPEED_10000_MBPS;
		adapter->advModes[i].duplex = VMK_LINK_DUPLEX_FULL;
		adapter->advModes[i].media = media;
		i++;
	}

	if (adapter->port.supported[cfg_idx] & ELINK_SUPPORTED_20000baseKR2_Full) {
		adapter->advModes[i].speed = VMK_LINK_SPEED_20000_MBPS;
		adapter->advModes[i].duplex = VMK_LINK_DUPLEX_FULL;
		adapter->advModes[i].media = media;
		i++;
	}

	num = MIN(*numModes, i);
	VMK_ASSERT(num > 0);
	vmk_Memcpy(modes, adapter->advModes,
			num * sizeof(vmk_UplinkAdvertisedMode));
	*numModes = num;

	return VMK_OK;
}

static VMK_ReturnStatus
qfle3_adv_modes_set(vmk_AddrCookie driverData,
                    vmk_UplinkAdvertisedMode *modes,
                    vmk_uint32 numModes)
{
   return VMK_NOT_SUPPORTED;
}

static vmk_UplinkAdvertisedModesOps qfle3_adv_modes_ops = {
   .getAdvertisedModes  = qfle3_adv_modes_get,
   .setAdvertisedModes  = qfle3_adv_modes_set,
};
#endif
VMK_ReturnStatus
qfle3_get_eeprom_len(vmk_AddrCookie drv_data, vmk_int32 *len)
{
	qfle3_adapter *adapter = (qfle3_adapter *)drv_data.ptr;

	*len = adapter->hw_info.flash_size;

	QFLE3_INFO("NVM Len = 0x%x\n", *len);

	return VMK_OK;
}

vmk_Bool
qfle3_is_nvm_accessible(struct qfle3_adapter *adapter)
{
	VMK_ReturnStatus status = VMK_FAILURE;
	vmk_uint32 pmc = 0;

	if (adapter->pm_cap)
		status = vmk_PCIReadConfig(vmk_ModuleCurrentID, adapter->pdev,
				VMK_PCI_CONFIG_ACCESS_16, adapter->pm_cap + PCI_PM_CTRL , &pmc);

	if ((status == VMK_OK && ((pmc & PCI_PM_CTRL_STATE_MASK) != 0)))
		return VMK_FALSE;

	return VMK_TRUE;
}

/* Per pf misc lock must be acquired before the per port mcp lock. Otherwise,
 * HAD we done things the other way around, if two pfs from the same port would
 * attempt to access nvram at the same time, we could run into a scenario such
 * as:
 * pf A takes the port lock.
 * pf B succeeds in taking the same lock since they are from the same port.
 * pf A takes the per pf misc lock. Performs eeprom access.
 * pf A finishes. Unlocks the per pf misc lock.
 * Pf B takes the lock and proceeds to perform it's own access.
 * pf A unlocks the per port lock, while pf B is still working (!).
 * mcp takes the per port lock and corrupts pf B's access (and/or has it's own
 * access corrupted by pf B)
 */
static VMK_ReturnStatus
qfle3_acquire_nvram_lock(struct qfle3_adapter *adapter)
{
	int port = QFLE3_PORT(adapter);
	int count, i;
	vmk_uint32 val = 0;

	/* acquire HW lock: protect against other PFs in PF Direct Assignment */
	qfle3_acquire_hw_lock(adapter, HW_LOCK_RESOURCE_NVRAM);

	/* adjust timeout for emulation/FPGA */
	count = QFLE3_NVRAM_TIMEOUT_COUNT;

	/* request access to nvram interface */
	REG_WR(adapter, MCP_REG_MCPR_NVM_SW_ARB,
			(MCPR_NVM_SW_ARB_ARB_REQ_SET1 << port));

	for (i = 0; i < count*10; i++) {
		val = REG_RD(adapter, MCP_REG_MCPR_NVM_SW_ARB);
		if (val & (MCPR_NVM_SW_ARB_ARB_ARB1 << port))
			break;

		vmk_DelayUsecs(5);
	}

	if (!(val & (MCPR_NVM_SW_ARB_ARB_ARB1 << port))) {
		QFLE3_ERR("cannot get access to nvram interface\n");
		qfle3_release_hw_lock(adapter, HW_LOCK_RESOURCE_NVRAM);
		return VMK_BUSY;
	}

	return VMK_OK;
}

static VMK_ReturnStatus
qfle3_release_nvram_lock(struct qfle3_adapter *adapter)
{
	int port = QFLE3_PORT(adapter);
	int count, i;
	vmk_uint32 val = 0;

	/* adjust timeout for emulation/FPGA */
	count = QFLE3_NVRAM_TIMEOUT_COUNT;

	/* relinquish nvram interface */
	REG_WR(adapter, MCP_REG_MCPR_NVM_SW_ARB,
			(MCPR_NVM_SW_ARB_ARB_REQ_CLR1 << port));

	for (i = 0; i < count*10; i++) {
		val = REG_RD(adapter, MCP_REG_MCPR_NVM_SW_ARB);
		if (!(val & (MCPR_NVM_SW_ARB_ARB_ARB1 << port)))
			break;

		vmk_DelayUsecs(5);
	}

	if (val & (MCPR_NVM_SW_ARB_ARB_ARB1 << port)) {
                QFLE3_ERR(
				"cannot free access to nvram interface\n");
		return VMK_BUSY;
	}

	/* release HW lock: protect against other PFs in PF Direct Assignment */
	qfle3_release_hw_lock(adapter, HW_LOCK_RESOURCE_NVRAM);
	return VMK_OK;
}

static void
qfle3_enable_nvram_access(struct qfle3_adapter *adapter)
{
	vmk_uint32 val;

	val = REG_RD(adapter, MCP_REG_MCPR_NVM_ACCESS_ENABLE);

	/* enable both bits, even on read */
	REG_WR(adapter, MCP_REG_MCPR_NVM_ACCESS_ENABLE,
			(val | MCPR_NVM_ACCESS_ENABLE_EN |
			 MCPR_NVM_ACCESS_ENABLE_WR_EN));
}

static void
qfle3_disable_nvram_access(struct qfle3_adapter *adapter)
{
	vmk_uint32 val;

	val = REG_RD(adapter, MCP_REG_MCPR_NVM_ACCESS_ENABLE);

	/* disable both bits, even after read */
	REG_WR(adapter, MCP_REG_MCPR_NVM_ACCESS_ENABLE,
			(val & ~(MCPR_NVM_ACCESS_ENABLE_EN |
					 MCPR_NVM_ACCESS_ENABLE_WR_EN)));
}

static VMK_ReturnStatus
qfle3_nvram_read_dword(struct qfle3_adapter *adapter, vmk_uint32 offset,
		__be32 *ret_val, vmk_uint32 cmd_flags)
{
	int count, i;
	vmk_uint32 val;
	VMK_ReturnStatus status;

	/* build the command word */
	cmd_flags |= MCPR_NVM_COMMAND_DOIT;

	/* need to clear DONE bit separately */
	REG_WR(adapter, MCP_REG_MCPR_NVM_COMMAND, MCPR_NVM_COMMAND_DONE);

	/* address of the NVRAM to read from */
	REG_WR(adapter, MCP_REG_MCPR_NVM_ADDR,
			(offset & MCPR_NVM_ADDR_NVM_ADDR_VALUE));

	/* issue a read command */
	REG_WR(adapter, MCP_REG_MCPR_NVM_COMMAND, cmd_flags);

	/* adjust timeout for emulation/FPGA */
	count = QFLE3_NVRAM_TIMEOUT_COUNT;

	/* wait for completion */
	*ret_val = 0;
	status = VMK_BUSY;

	for (i = 0; i < count; i++) {
		vmk_DelayUsecs(5);
		val = REG_RD(adapter, MCP_REG_MCPR_NVM_COMMAND);

		if (val & MCPR_NVM_COMMAND_DONE) {
			val = REG_RD(adapter, MCP_REG_MCPR_NVM_READ);
			*ret_val = vmk_CPUToBE32(val);
			status = VMK_OK;
			break;
		}
	}
	if (status  != VMK_OK)
		QFLE3_ERR("nvram read timeout expired\n");
	return status;
}

VMK_ReturnStatus
qfle3_nvram_read(struct qfle3_adapter *adapter, vmk_uint32 offset,
		char *ret_buf, vmk_uint32 buf_size, vmk_uint32 *out_len)
{
	vmk_uint32 cmd_flags;
	__be32 val;
	VMK_ReturnStatus status;

	if ((offset & 0x03) || (buf_size & 0x03) || (buf_size == 0)) {
		QFLE3_ERR(
				"Invalid parameter: offset 0x%x  buf_size 0x%x\n",
				offset, buf_size);
		return VMK_BAD_PARAM;
	}

	if (offset + buf_size > adapter->hw_info.flash_size) {
		QFLE3_ERR(
				"Invalid parameter: offset (0x%x) + buf_size (0x%x) > flash_size (0x%x)\n",
				offset, buf_size, adapter->hw_info.flash_size);
		return VMK_BAD_PARAM;
	}

	/* request access to nvram interface */
	status = qfle3_acquire_nvram_lock(adapter);
	if (status != VMK_OK)
		return status;

	/* enable access to nvram interface */
	qfle3_enable_nvram_access(adapter);

	/* read the first word(s) */
	cmd_flags = MCPR_NVM_COMMAND_FIRST;
	while ((buf_size > sizeof(vmk_uint32)) && (status == VMK_OK)) {
		status = qfle3_nvram_read_dword(adapter, offset, &val, cmd_flags);
		vmk_Memcpy(ret_buf, &val, 4);

		/* advance to the next dword */
		offset += sizeof(vmk_uint32);
		ret_buf += sizeof(vmk_uint32);
		buf_size -= sizeof(vmk_uint32);
		*out_len += sizeof(vmk_uint32);
		cmd_flags = 0;
	}

	if (status  == VMK_OK) {
		cmd_flags |= MCPR_NVM_COMMAND_LAST;
		status = qfle3_nvram_read_dword(adapter, offset, &val, cmd_flags);
		if (status == VMK_OK) {
			vmk_Memcpy(ret_buf, &val, 4);
			*out_len += sizeof(vmk_uint32);
		}
	}

	/* disable access to nvram interface */
	qfle3_disable_nvram_access(adapter);
	qfle3_release_nvram_lock(adapter);

	return status;
}
static VMK_ReturnStatus qfle3_nvram_read32(struct qfle3_adapter *adapter, u32 offset, u32 *buf,
			      int buf_size)
{
	VMK_ReturnStatus rc;
   vmk_uint32 out_len;

	rc = qfle3_nvram_read(adapter, offset, (u8 *)buf, buf_size, &out_len);

	if (rc == VMK_OK) {
		__be32 *be = (__be32 *)buf;

		while ((buf_size -= 4) >= 0)
			*buf++ = vmk_BE32ToCPU(*be++);
	}

	return rc;
}
#define CRC32_RESIDUAL			0xdebb20e3
#define CRC_BUFF_SIZE			256
#define min_t(type, x, y) ({			\
	type __min1 = (x);			\
	type __min2 = (y);			\
	__min1 < __min2 ? __min1: __min2; })

static VMK_ReturnStatus qfle3_nvram_crc(struct qfle3_adapter *adapter,
			   int offset,
			   int size,
			   u8 *buff)
{
	u32 crc = ~0;
	VMK_ReturnStatus rc = VMK_OK;
   int done = 0;
   vmk_uint32 out_len = 0;
   extern 
      unsigned int mm_crc32(unsigned char *address, unsigned int size, unsigned int crc);
   
	QFLE3_DBG(QFLE3_DBG_NVM,
	   "NVRAM CRC from 0x%08x to 0x%08x\n", offset, offset + size);

	while (done < size) {
		int count = min_t(int, size - done, CRC_BUFF_SIZE);

		rc = qfle3_nvram_read(adapter, offset + done, buff, count, &out_len);

		if (rc)
			return rc;

//		crc = crc32_le(crc, buff, count);
      crc = mm_crc32(buff, count, crc);
		done += count;
	}

   
	if (crc != CRC32_RESIDUAL)
		rc = VMK_FAILURE;

	return rc;
}


struct crc_pair {
	int offset;
	int size;
};

static VMK_ReturnStatus qfle3_test_nvram_tbl(struct qfle3_adapter *adapter,
				const struct crc_pair *nvram_tbl, u8 *buf)
{
	int i;

	for (i = 0; nvram_tbl[i].size; i++) {
		int rc = qfle3_nvram_crc(adapter, nvram_tbl[i].offset,
					 nvram_tbl[i].size, buf);
		if (rc) {
			QFLE3_DBG(QFLE3_DBG_NVM,
			   "nvram_tbl[%d] has failed crc test (rc %d)\n",
			   i, rc);
			return rc;
		}
	}

	return 0;
}
#define GRC_SAVED_OFFSET 1
#define GRC_FRESH_OFFSET 2
VMK_ReturnStatus
qfle3_dump_eeprom(vmk_AddrCookie drv_data, vmk_AddrCookie buf,
      vmk_uint32 buf_len, vmk_uint32 offset, vmk_uint32 *out_len)
{
   VMK_ReturnStatus status;
   qfle3_adapter *adapter = (qfle3_adapter *)drv_data.ptr;
   char *ret_buf= (char *)buf.ptr;
//	struct chip_core_dmp *dmp;

   *out_len = 0;
//   QFLE3_INFO("dump eeprom len %d offset 0x%x \n", buf_len, offset);
   if (!qfle3_is_nvm_accessible(adapter)) {
      QFLE3_ERR("Cannot access eeprom. Try again later.\n");
      return VMK_FAILURE;
   }

   QFLE3_DBG(QFLE3_DBG_NVM, "eeprom: offset 0x%x len 0x%x\n",
      offset, buf_len);

   status = qfle3_nvram_read(adapter, offset, ret_buf, buf_len, out_len);
   return status;
}


static VMK_ReturnStatus 
qfle3_nvram_write_dword(struct qfle3_adapter *adapter, u32 offset, __be32 val,
				   vmk_uint32 cmd_flags)
{
	int count, i;
   VMK_ReturnStatus status;

	/* build the command word */
	cmd_flags |= MCPR_NVM_COMMAND_DOIT | MCPR_NVM_COMMAND_WR;

	/* need to clear DONE bit separately */
	REG_WR(adapter, MCP_REG_MCPR_NVM_COMMAND, MCPR_NVM_COMMAND_DONE);

	/* write the data */
	REG_WR(adapter, MCP_REG_MCPR_NVM_WRITE, val);

	/* address of the NVRAM to write to */
	REG_WR(adapter, MCP_REG_MCPR_NVM_ADDR,
	       (offset & MCPR_NVM_ADDR_NVM_ADDR_VALUE));

	/* issue the write command */
	REG_WR(adapter, MCP_REG_MCPR_NVM_COMMAND, cmd_flags);

	/* adjust timeout for emulation/FPGA */
	count = QFLE3_NVRAM_TIMEOUT_COUNT;
	if (CHIP_REV_IS_SLOW(adapter))
		count *= 100;

	/* wait for completion */
	status = VMK_BUSY;
	for (i = 0; i < count; i++) {
		vmk_DelayUsecs(5);
		val = REG_RD(adapter, MCP_REG_MCPR_NVM_COMMAND);
		if (val & MCPR_NVM_COMMAND_DONE) {
         status = VMK_OK;
			break;
		}
	}

	if (status != VMK_OK)		
      QFLE3_ERR("nvram write timeout expired\n");
	return status;
}
VMK_ReturnStatus qfle3_nvram_write(struct qfle3_adapter *adapter, vmk_uint32 offset, char *data_buf,
			     int buf_size)
{
	VMK_ReturnStatus rc = VMK_OK;
	vmk_uint32 cmd_flags;
	vmk_uint32 val;
	vmk_uint32 written_so_far;

//	if (buf_size == 1)	/* ethtool */
//		return qfle3_nvram_write1(bp, offset, data_buf, buf_size);

	if ((offset & 0x03) || (buf_size & 0x03) || (buf_size == 0)) {
		QFLE3_ERR("Invalid parameter: offset 0x%x  buf_size 0x%x\n",
		   offset, buf_size);
		return VMK_FAILURE;
	}

	if (offset + buf_size > adapter->hw_info.flash_size) {
		QFLE3_ERR("Invalid parameter: offset (0x%x) + buf_size (0x%x) > flash_size (0x%x)\n",
		   offset, buf_size, adapter->hw_info.flash_size);
		return VMK_FAILURE;
	}

	/* request access to nvram interface */
	rc = qfle3_acquire_nvram_lock(adapter);
	if (rc != VMK_OK)
		return rc;

	/* enable access to nvram interface */
	qfle3_enable_nvram_access(adapter);

	written_so_far = 0;
	cmd_flags = MCPR_NVM_COMMAND_FIRST;
	while ((written_so_far < buf_size) && (rc == 0)) {
		if (written_so_far == (buf_size - sizeof(u32)))
			cmd_flags |= MCPR_NVM_COMMAND_LAST;
		else if (((offset + 4) % QFLE3_NVRAM_PAGE_SIZE) == 0)
			cmd_flags |= MCPR_NVM_COMMAND_LAST;
		else if ((offset % QFLE3_NVRAM_PAGE_SIZE) == 0)
			cmd_flags |= MCPR_NVM_COMMAND_FIRST;

		vmk_Memcpy(&val, data_buf, 4);

		/* Notice unlike qfle3_nvram_read_dword() this will will not
		 * change val using be32_to_cpu(), which causes data to flip
		 * if the eeprom is read and then written back. This is due
		 * to tools utilizing this functionality that would break
		 * if this would be resolved.
		 */
		rc = qfle3_nvram_write_dword(adapter, offset, val, cmd_flags);

		/* advance to the next dword */
		offset += sizeof(u32);
		data_buf += sizeof(u32);
		written_so_far += sizeof(u32);

		/* At end of each 4Kb page, release nvram lock to allow MFW
		 * chance to take it for its own use.
		 */
		if ((cmd_flags & MCPR_NVM_COMMAND_LAST) &&
		    (written_so_far < buf_size)) {
			QFLE3_ERR("Releasing NVM lock after offset 0x%x\n",
			   (u32)(offset - sizeof(u32)));
			qfle3_release_nvram_lock(adapter);
			vmk_DelayUsecs(1000);
			rc = qfle3_acquire_nvram_lock(adapter);
			if (rc != VMK_OK)
				return rc;
		}

		cmd_flags = 0;
	}

	/* disable access to nvram interface */
	qfle3_disable_nvram_access(adapter);
	qfle3_release_nvram_lock(adapter);

	return rc;
}

VMK_ReturnStatus
qfle3_write_eeprom(vmk_AddrCookie drv_data, vmk_uint32 magic,
		vmk_AddrCookie buf, vmk_uint32 bufLen, vmk_uint32 offset)
{
	qfle3_adapter *adapter = (qfle3_adapter *)drv_data.ptr;
#define MC_ASSERT_MAGIC 1234
#define MCP_ASSERT_MAGIC 1235
#define BUS_MASTER_MAGIC 1236
#define FATAL_HW_ERR_MAGIC 1237

   if (magic == MCP_ASSERT_MAGIC) {
      vmk_uint32 return_code;
      QFLE3_INFO("Triggered MCP Assert\n");
      return_code = qfle3_mfw_command(adapter, 0x10f00000, 0);
      return VMK_OK;
   }
   
   if (magic == MC_ASSERT_MAGIC) {
      
      QFLE3_INFO("Triggered MC Assert\n");
      adapter->trigger_error = QFLE3_TRIGGER_ASSERT;
      return VMK_OK;
   }
   
   if (magic == BUS_MASTER_MAGIC) {
      vmk_uint32 val;
      
      QFLE3_INFO("Triggered PCI bus timeout\n");
      val = REG_RD(adapter, 0x2004);
      QFLE3_INFO("register 0x2004 is 0x%x", val);
      REG_WR(adapter,0x2004,0x100002);
      val = REG_RD(adapter, 0x2004);
      QFLE3_INFO("register 0x2004 is written as 0x%x", val);
      return VMK_OK;
   }
   
   if (magic == FATAL_HW_ERR_MAGIC) {
      adapter->trigger_error = QFLE3_TRIGGER_FATAL_HW_ERROR;
      QFLE3_INFO("Triggered FATAL HW ERROR\n");
      return VMK_OK;
   }
	/* Not supported. */
	return VMK_NOT_SUPPORTED;

}

static vmk_UplinkEEPROMOps qfle3_eeprom_ops = {
	.eepromLenGet   = qfle3_get_eeprom_len,
	.eepromDump     = qfle3_dump_eeprom,
	.eepromSet      = qfle3_write_eeprom,
};

VMK_ReturnStatus
qfle3_get_xcvr(vmk_AddrCookie drv_data, vmk_UplinkTransceiverType *xcvrType)
{
   vmk_uint32 phy_idx;
   struct elink_phy *phy;
   qfle3_adapter *adapter = (qfle3_adapter *)drv_data.ptr;

   phy_idx = qfle3_get_cur_phy_idx(adapter);
   phy = &adapter->link_params.phy[phy_idx];

   if (phy->media_type == ELINK_ETH_PHY_BASE_T) {
	   *xcvrType = VMK_UPLINK_TRANSCEIVER_TYPE_INTERNAL;
   } else {
	   *xcvrType = VMK_UPLINK_TRANSCEIVER_TYPE_EXTERNAL;
   }
   return VMK_OK;
}

VMK_ReturnStatus
qfle3_set_xcvr(vmk_AddrCookie drv_data, vmk_UplinkTransceiverType xcvrType)
{
//	qfle3_adapter *adapter = (qfle3_adapter *)drv_data.ptr;

	return VMK_NOT_SUPPORTED;
}


static vmk_UplinkTransceiverTypeOps  qfle3_xcvr_ops = {
   .getTransceiverType = qfle3_get_xcvr,
   .setTransceiverType = qfle3_set_xcvr,
};

VMK_ReturnStatus
qfle3_get_cable_type(vmk_AddrCookie driverData,
		vmk_UplinkCableType *cableType)
{

   vmk_uint32 phy_idx;
   struct elink_phy *phy;
   qfle3_adapter *adapter = (qfle3_adapter *)driverData.ptr;

   phy_idx = qfle3_get_cur_phy_idx(adapter);
   phy = &adapter->link_params.phy[phy_idx];

   switch (phy->media_type) {
	   case ELINK_ETH_PHY_SFPP_10G_FIBER:
	   case ELINK_ETH_PHY_SFP_1G_FIBER:
	   case ELINK_ETH_PHY_XFP_FIBER: 
	   case ELINK_ETH_PHY_KR:
		   *cableType = VMK_UPLINK_CABLE_TYPE_FIBRE;
			break;
	   case ELINK_ETH_PHY_DA_TWINAX:
		   *cableType = VMK_UPLINK_CABLE_TYPE_DA;
			break;
	   case ELINK_ETH_PHY_BASE_T:
		   *cableType = VMK_UPLINK_CABLE_TYPE_TP;
		   break;
	   default:
		   *cableType = VMK_UPLINK_CABLE_TYPE_OTHER;
   }

	return VMK_OK;
}

VMK_ReturnStatus
qfle3_set_cable_types(vmk_AddrCookie driverData,
		vmk_UplinkCableType cableType)
{
	return VMK_NOT_SUPPORTED;
}

VMK_ReturnStatus
qfle3_get_suported_ctypes(vmk_AddrCookie driverData,
		vmk_UplinkCableType *cableType)

{
   struct elink_phy *phy;
   vmk_uint32 phy_idx;
   qfle3_adapter *adapter = (qfle3_adapter *)driverData.ptr;

   phy_idx = qfle3_get_cur_phy_idx(adapter);
   phy = &adapter->link_params.phy[phy_idx];

   switch (phy->media_type) {
	   case ELINK_ETH_PHY_SFPP_10G_FIBER:
	   case ELINK_ETH_PHY_SFP_1G_FIBER:
	   case ELINK_ETH_PHY_XFP_FIBER: 
	   case ELINK_ETH_PHY_KR:
	   case ELINK_ETH_PHY_DA_TWINAX:
		   *cableType = VMK_UPLINK_CABLE_TYPE_FIBRE;
			break;
	   case ELINK_ETH_PHY_BASE_T:
		   *cableType = VMK_UPLINK_CABLE_TYPE_TP;
		   break;
	   default:
		   *cableType = VMK_UPLINK_CABLE_TYPE_OTHER;
   }

	return VMK_OK;
}

static vmk_UplinkCableTypeOps qfle3_cable_type_ops = {
	.getCableType = qfle3_get_cable_type,
	.getSupportedCableTypes = qfle3_get_suported_ctypes,
	.setCableType = qfle3_set_cable_types,
};
//typedef enum  {
//   QFLE3_SELFTEST_MEMORY,
//   QFLE3_SELFTEST_INTERRUPT,
//   QFLE3_SELFTEST_REGISTER,
//   QFLE3_SELFTEST_CLOCK,
//   QFLE3_SELFTEST_NVRAM,
//   QFLE3_SELFTEST_MAX
//} qfle3_selftest_cmd_t;
/* Display strings for qedrntv_selftest_cmd_t */
static
const char selftest_strings[QFLE3_SELFTEST_MAX][32] = {
        "Memory\t\t",
        "Interrupt\t",
        "Register\t",
        "Clock\t\t",
        "NVRAM\t\t",
        "LOOPBACK\t\t",
};
enum {
   QFLE3_CHIP_E1_OFST = 0,
   QFLE3_CHIP_E1H_OFST,
   QFLE3_CHIP_E2_OFST,
   QFLE3_CHIP_E3_OFST,
   QFLE3_CHIP_E3B0_OFST,
   QFLE3_CHIP_MAX_OFST
};

VMK_ReturnStatus qfle3_selftest_memory(qfle3_adapter *adapter)
{
   
	int i, j;
   VMK_ReturnStatus rc = VMK_FAILURE;
	u32 val, index;
	static const struct {
		u32 offset;
		int size;
	} mem_tbl[] = {
		{ CCM_REG_XX_DESCR_TABLE,   CCM_REG_XX_DESCR_TABLE_SIZE },
		{ CFC_REG_ACTIVITY_COUNTER, CFC_REG_ACTIVITY_COUNTER_SIZE },
		{ CFC_REG_LINK_LIST,        CFC_REG_LINK_LIST_SIZE },
		{ DMAE_REG_CMD_MEM,         DMAE_REG_CMD_MEM_SIZE },
		{ TCM_REG_XX_DESCR_TABLE,   TCM_REG_XX_DESCR_TABLE_SIZE },
		{ UCM_REG_XX_DESCR_TABLE,   UCM_REG_XX_DESCR_TABLE_SIZE },
		{ XCM_REG_XX_DESCR_TABLE,   XCM_REG_XX_DESCR_TABLE_SIZE },

		{ 0xffffffff, 0 }
	};

	static const struct {
		char *name;
		u32 offset;
		u32 hw_mask[QFLE3_CHIP_MAX_OFST];
	} prty_tbl[] = {
		{ "CCM_PRTY_STS",  CCM_REG_CCM_PRTY_STS,
			{0x3ffc0, 0,   0, 0} },
		{ "CFC_PRTY_STS",  CFC_REG_CFC_PRTY_STS,
			{0x2,     0x2, 0, 0} },
		{ "DMAE_PRTY_STS", DMAE_REG_DMAE_PRTY_STS,
			{0,       0,   0, 0} },
		{ "TCM_PRTY_STS",  TCM_REG_TCM_PRTY_STS,
			{0x3ffc0, 0,   0, 0} },
		{ "UCM_PRTY_STS",  UCM_REG_UCM_PRTY_STS,
			{0x3ffc0, 0,   0, 0} },
		{ "XCM_PRTY_STS",  XCM_REG_XCM_PRTY_STS,
			{0x3ffc1, 0,   0, 0} },

		{ NULL, 0xffffffff, {0, 0, 0, 0} }
	};

	if (!qfle3_is_nvm_accessible(adapter)) {
                QFLE3_INFO("cannot access eeprom when the interface is down\n");
		return rc;
	}

	if (CHIP_IS_E1(adapter))
		index = QFLE3_CHIP_E1_OFST;
	else if (CHIP_IS_E1H(adapter))
		index = QFLE3_CHIP_E1H_OFST;
	else if (CHIP_IS_E2(adapter))
		index = QFLE3_CHIP_E2_OFST;
	else /* e3 */
		index = QFLE3_CHIP_E3_OFST;

	/* pre-Check the parity status */
	for (i = 0; prty_tbl[i].offset != 0xffffffff; i++) {
		val = REG_RD(adapter, prty_tbl[i].offset);
		if (val & ~(prty_tbl[i].hw_mask[index])) {
         QFLE3_INFO(
			   "%s is 0x%x\n", prty_tbl[i].name, val);
			goto test_mem_exit;
		}
	}

	/* Go through all the memories */
	for (i = 0; mem_tbl[i].offset != 0xffffffff; i++)
		for (j = 0; j < mem_tbl[i].size; j++)
			val = REG_RD(adapter, mem_tbl[i].offset + j*4);

	/* Check the parity status */
	for (i = 0; prty_tbl[i].offset != 0xffffffff; i++) {
		val = REG_RD(adapter, prty_tbl[i].offset);
		if (val & ~(prty_tbl[i].hw_mask[index])) {
         QFLE3_INFO(
			   "%s is 0x%x\n", prty_tbl[i].name, val);
			goto test_mem_exit;
		}
	}

	rc = VMK_OK;
   QFLE3_INFO(
      "Memory Test successful\n");
test_mem_exit:
	return rc;
}
VMK_ReturnStatus qfle3_selftest_interrupt(qfle3_adapter *adapter)
{
   int rc;
	struct ecore_queue_state_params params = {NULL};

//	if (!netif_running(bp->dev)) {
//      QFLE3_DBG(QFLE3_DBG_UPLINK,
//		   "cannot access eeprom when the interface is down\n");
//		return -ENODEV;
//	}

	params.q_obj = &adapter->sp_objs->q_obj;
	params.cmd = ECORE_Q_CMD_EMPTY;

	ECORE_SET_BIT(RAMROD_COMP_WAIT, &params.ramrod_flags);

	rc = ecore_queue_state_change(adapter, &params);

   if (rc == ECORE_SUCCESS)
      return VMK_OK;
   else
      return VMK_FAILURE;
}
#define QFLE3_CHIP_MASK_E1	(1 << QFLE3_CHIP_E1_OFST)
#define QFLE3_CHIP_MASK_E1H	(1 << QFLE3_CHIP_E1H_OFST)
#define QFLE3_CHIP_MASK_E2	(1 << QFLE3_CHIP_E2_OFST)
#define QFLE3_CHIP_MASK_E3	(1 << QFLE3_CHIP_E3_OFST)
#define QFLE3_CHIP_MASK_E3B0	(1 << QFLE3_CHIP_E3B0_OFST)

#define QFLE3_CHIP_MASK_ALL	((1 << QFLE3_CHIP_MAX_OFST) - 1)
#define QFLE3_CHIP_MASK_E1X	(QFLE3_CHIP_MASK_E1 | QFLE3_CHIP_MASK_E1H)

VMK_ReturnStatus qfle3_selftest_register(qfle3_adapter *adapter)
{
   int idx, i;
   VMK_ReturnStatus rc = VMK_FAILURE;
   u32 wr_val = 0, hw;
   int port = QFLE3_PORT(adapter);
   static const struct {
      u32 hw;
      u32 offset0;
      u32 offset1;
      u32 mask;
   } reg_tbl[] = {
/* 0 */     { QFLE3_CHIP_MASK_ALL,
         BRB1_REG_PAUSE_LOW_THRESHOLD_0,  4, 0x000003ff },
      { QFLE3_CHIP_MASK_ALL,
         DORQ_REG_DB_ADDR0,      4, 0xffffffff },
      { QFLE3_CHIP_MASK_E1X,
         HC_REG_AGG_INT_0,    4, 0x000003ff },
      { QFLE3_CHIP_MASK_ALL,
         PBF_REG_MAC_IF0_ENABLE,    4, 0x00000001 },
      { QFLE3_CHIP_MASK_E1X | QFLE3_CHIP_MASK_E2 | QFLE3_CHIP_MASK_E3,
         PBF_REG_P0_INIT_CRD,    4, 0x000007ff },
      { QFLE3_CHIP_MASK_E3B0,
         PBF_REG_INIT_CRD_Q0,    4, 0x000007ff },
      { QFLE3_CHIP_MASK_ALL,
         PRS_REG_CID_PORT_0,     4, 0x00ffffff },
      { QFLE3_CHIP_MASK_ALL,
         PXP2_REG_PSWRQ_CDU0_L2P,   4, 0x000fffff },
      { QFLE3_CHIP_MASK_ALL,
         PXP2_REG_RQ_CDU0_EFIRST_MEM_ADDR, 8, 0x0003ffff },
      { QFLE3_CHIP_MASK_ALL,
         PXP2_REG_PSWRQ_TM0_L2P,    4, 0x000fffff },
/* 10 */ { QFLE3_CHIP_MASK_ALL,
         PXP2_REG_RQ_USDM0_EFIRST_MEM_ADDR, 8, 0x0003ffff },
      { QFLE3_CHIP_MASK_ALL,
         PXP2_REG_PSWRQ_TSDM0_L2P,  4, 0x000fffff },
      { QFLE3_CHIP_MASK_ALL,
         QM_REG_CONNNUM_0,    4, 0x000fffff },
      { QFLE3_CHIP_MASK_ALL,
         TM_REG_LIN0_MAX_ACTIVE_CID,   4, 0x0003ffff },
      { QFLE3_CHIP_MASK_ALL,
         SRC_REG_KEYRSS0_0,      40, 0xffffffff },
      { QFLE3_CHIP_MASK_ALL,
         SRC_REG_KEYRSS0_7,      40, 0xffffffff },
      { QFLE3_CHIP_MASK_ALL,
         XCM_REG_WU_DA_SET_TMR_CNT_FLG_CMD00, 4, 0x00000001 },
      { QFLE3_CHIP_MASK_ALL,
         XCM_REG_WU_DA_CNT_CMD00,   4, 0x00000003 },
      { QFLE3_CHIP_MASK_ALL,
         XCM_REG_GLB_DEL_ACK_MAX_CNT_0,   4, 0x000000ff },
      { QFLE3_CHIP_MASK_ALL,
         NIG_REG_LLH0_T_BIT,     4, 0x00000001 },
/* 20 */ { QFLE3_CHIP_MASK_E1X | QFLE3_CHIP_MASK_E2,
         NIG_REG_EMAC0_IN_EN,    4, 0x00000001 },
      { QFLE3_CHIP_MASK_E1X | QFLE3_CHIP_MASK_E2,
         NIG_REG_BMAC0_IN_EN,    4, 0x00000001 },
      { QFLE3_CHIP_MASK_ALL,
         NIG_REG_XCM0_OUT_EN,    4, 0x00000001 },
      { QFLE3_CHIP_MASK_ALL,
         NIG_REG_BRB0_OUT_EN,    4, 0x00000001 },
      { QFLE3_CHIP_MASK_ALL,
         NIG_REG_LLH0_XCM_MASK,     4, 0x00000007 },
      { QFLE3_CHIP_MASK_ALL,
         NIG_REG_LLH0_ACPI_PAT_6_LEN,  68, 0x000000ff },
      { QFLE3_CHIP_MASK_ALL,
         NIG_REG_LLH0_ACPI_PAT_0_CRC,  68, 0xffffffff },
      { QFLE3_CHIP_MASK_ALL,
         NIG_REG_LLH0_DEST_MAC_0_0, 160, 0xffffffff },
      { QFLE3_CHIP_MASK_ALL,
         NIG_REG_LLH0_DEST_IP_0_1,  160, 0xffffffff },
      { QFLE3_CHIP_MASK_ALL,
         NIG_REG_LLH0_IPV4_IPV6_0,  160, 0x00000001 },
/* 30 */ { QFLE3_CHIP_MASK_ALL,
         NIG_REG_LLH0_DEST_UDP_0,   160, 0x0000ffff },
      { QFLE3_CHIP_MASK_ALL,
         NIG_REG_LLH0_DEST_TCP_0,   160, 0x0000ffff },
      { QFLE3_CHIP_MASK_ALL,
         NIG_REG_LLH0_VLAN_ID_0, 160, 0x00000fff },
      { QFLE3_CHIP_MASK_E1X | QFLE3_CHIP_MASK_E2,
         NIG_REG_XGXS_SERDES0_MODE_SEL,   4, 0x00000001 },
      { QFLE3_CHIP_MASK_ALL,
         NIG_REG_LED_CONTROL_OVERRIDE_TRAFFIC_P0, 4, 0x00000001},
      { QFLE3_CHIP_MASK_ALL,
         NIG_REG_STATUS_INTERRUPT_PORT0,  4, 0x07ffffff },
      { QFLE3_CHIP_MASK_E1X | QFLE3_CHIP_MASK_E2,
         NIG_REG_XGXS0_CTRL_EXTREMOTEMDIOST, 24, 0x00000001 },
      { QFLE3_CHIP_MASK_E1X | QFLE3_CHIP_MASK_E2,
         NIG_REG_SERDES0_CTRL_PHY_ADDR,   16, 0x0000001f },

      { QFLE3_CHIP_MASK_ALL, 0xffffffff, 0, 0x00000000 }
   };

   if (!qfle3_is_nvm_accessible(adapter)) {
      QFLE3_DBG(QFLE3_DBG_UPLINK,
         "cannot access eeprom when the interface is down\n");
      return rc;
   }

   if (CHIP_IS_E1(adapter))
      hw = QFLE3_CHIP_MASK_E1;
   else if (CHIP_IS_E1H(adapter))
      hw = QFLE3_CHIP_MASK_E1H;
   else if (CHIP_IS_E2(adapter))
      hw = QFLE3_CHIP_MASK_E2;
   else if (CHIP_IS_E3B0(adapter))
      hw = QFLE3_CHIP_MASK_E3B0;
   else /* e3 A0 */
      hw = QFLE3_CHIP_MASK_E3;

   /* Repeat the test twice:
    * First by writing 0x00000000, second by writing 0xffffffff
    */
   for (idx = 0; idx < 2; idx++) {

      switch (idx) {
      case 0:
         wr_val = 0;
         break;
      case 1:
         wr_val = 0xffffffff;
         break;
      }

      for (i = 0; reg_tbl[i].offset0 != 0xffffffff; i++) {
         u32 offset, mask, save_val, val;
         if (!(hw & reg_tbl[i].hw))
            continue;

         offset = reg_tbl[i].offset0 + port*reg_tbl[i].offset1;
         mask = reg_tbl[i].mask;

         save_val = REG_RD(adapter, offset);

         REG_WR(adapter, offset, wr_val & mask);

         val = REG_RD(adapter, offset);

         /* Restore the original register's value */
         REG_WR(adapter, offset, save_val);

         /* verify value is as expected */
         if ((val & mask) != (wr_val & mask)) {
            QFLE3_DBG(QFLE3_DBG_UPLINK,
               "offset 0x%x: val 0x%x != 0x%x mask 0x%x\n",
               offset, val, wr_val, mask);
            goto test_reg_exit;
         }
      }
   }

   rc = VMK_OK;

test_reg_exit:
   return rc;
}
VMK_ReturnStatus qfle3_selftest_clock(qfle3_adapter *adapter)
{
   return VMK_OK;
}
struct code_entry {
	u32 sram_start_addr;
	u32 code_attribute;
#define CODE_IMAGE_TYPE_MASK			0xf0800003
#define CODE_IMAGE_VNTAG_PROFILES_DATA		0xd0000003
#define CODE_IMAGE_LENGTH_MASK			0x007ffffc
#define CODE_IMAGE_TYPE_EXTENDED_DIR		0xe0000000
	u32 nvm_start_addr;
};
#define EXTENDED_DIR_EXISTS(code)					  \
	((code & CODE_IMAGE_TYPE_MASK) == CODE_IMAGE_TYPE_EXTENDED_DIR && \
	 (code & CODE_IMAGE_LENGTH_MASK) != 0)
#define CODE_ENTRY_EXTENDED_DIR_IDX	15
#define NVRAM_DIR_OFFSET		0x14
#define MAX_IMAGES_IN_EXTENDED_DIR          64

static int qfle3_test_nvram_dir(qfle3_adapter *adapter,
				struct code_entry *entry,
				u8 *buff)
{
	int size = entry->code_attribute & CODE_IMAGE_LENGTH_MASK;
	u32 type = entry->code_attribute & CODE_IMAGE_TYPE_MASK;
	int rc;

	/* Zero-length images and AFEX profiles do not have CRC */
	if (size == 0 || type == CODE_IMAGE_VNTAG_PROFILES_DATA)
		return 0;

	rc = qfle3_nvram_crc(adapter, entry->nvm_start_addr, size, buff);
	if (rc)
		QFLE3_DBG(QFLE3_DBG_UPLINK,
		   "image %x has failed crc test (rc %d)\n", type, rc);

	return rc;
}

static VMK_ReturnStatus qfle3_test_dir_entry(qfle3_adapter *adapter, u32 addr, u8 *buff)
{
	int rc;
	struct code_entry entry;

	rc = qfle3_nvram_read32(adapter, addr, (u32 *)&entry, sizeof(entry));
	if (rc)
		return rc;

	return qfle3_test_nvram_dir(adapter, &entry, buff);
}


static int qfle3_test_nvram_ext_dirs(qfle3_adapter *adapter, u8 *buff)
{
	u32 rc, cnt, dir_offset = NVRAM_DIR_OFFSET;
	struct code_entry entry;
	int i;

	rc = qfle3_nvram_read32(adapter,
				dir_offset +
				sizeof(entry) * CODE_ENTRY_EXTENDED_DIR_IDX,
				(u32 *)&entry, sizeof(entry));
	if (rc)
		return rc;

	if (!EXTENDED_DIR_EXISTS(entry.code_attribute))
		return 0;

	rc = qfle3_nvram_read32(adapter, entry.nvm_start_addr,
				&cnt, sizeof(u32));
	if (rc)
		return rc;

	dir_offset = entry.nvm_start_addr + 8;

	for (i = 0; i < cnt && i < MAX_IMAGES_IN_EXTENDED_DIR; i++) {
		rc = qfle3_test_dir_entry(adapter, dir_offset +
					      sizeof(struct code_entry) * i,
					  buff);
		if (rc)
			return rc;
	}

	return 0;
}

static VMK_ReturnStatus qfle3_test_nvram_dirs(qfle3_adapter *adapter, u8 *buff)
{
	VMK_ReturnStatus rc;
   u32 dir_offset = NVRAM_DIR_OFFSET;
	int i;

	QFLE3_DBG(QFLE3_DBG_NVM, "NVRAM DIRS CRC test-set\n");

	for (i = 0; i < CODE_ENTRY_EXTENDED_DIR_IDX; i++) {
		rc = qfle3_test_dir_entry(adapter, dir_offset +
					      sizeof(struct code_entry) * i,
					  buff);
		if (rc)
			return rc;
	}

	return qfle3_test_nvram_ext_dirs(adapter, buff);
}

VMK_ReturnStatus qfle3_selftest_nvram(qfle3_adapter *adapter)
{
   const struct crc_pair nvram_tbl[] = {
      {     0,  0x14 }, /* bootstrap */
      {  0x14,  0xec }, /* dir */
      { 0x100, 0x350 }, /* manuf_info */
      { 0x450,  0xf0 }, /* feature_info */
      { 0x640,  0x64 }, /* upgrade_key_info */
      { 0x708,  0x70 }, /* manuf_key_info */
      {     0,     0 }
   };
   const struct crc_pair nvram_tbl2[] = {
      { 0x7e8, 0x350 }, /* manuf_info2 */
      { 0xb38,  0xf0 }, /* feature_info */
      {     0,     0 }
   };

   u8 *buf;
   VMK_ReturnStatus rc = VMK_OK;
   u32 magic;

   if (QFLE3_NOMCP(adapter))
      return 0;

   buf = qfle3_heap_alloc(CRC_BUFF_SIZE);
   if (!buf) {
      QFLE3_DBG(QFLE3_DBG_UPLINK, "vmk_HeapAlloc failed\n");
      rc = VMK_NO_MEMORY;
      goto test_nvram_exit;
   }

   rc = qfle3_nvram_read32(adapter, 0, &magic, sizeof(magic));
   if (rc != VMK_OK) {
      QFLE3_DBG(QFLE3_DBG_UPLINK,
         "magic value read (rc %d)\n", rc);
      goto test_nvram_exit;
   }

   if (magic != 0x669955aa) {
      QFLE3_DBG(QFLE3_DBG_UPLINK,
         "wrong magic value (0x%08x)\n", magic);
      rc = VMK_FAILURE;
      goto test_nvram_exit;
   }

   QFLE3_DBG(QFLE3_DBG_UPLINK, "Port 0 CRC test-set\n");
   rc = qfle3_test_nvram_tbl(adapter, nvram_tbl, buf);
   if (rc)
      goto test_nvram_exit;

   if (!CHIP_IS_E1x(adapter) && !CHIP_IS_57811xx(adapter)) {
      u32 hide = SHMEM_RD(adapter, dev_info.shared_hw_config.config2) &
            SHARED_HW_CFG_HIDE_PORT1;

      if (!hide) {
         QFLE3_DBG(QFLE3_DBG_UPLINK,
            "Port 1 CRC test-set\n");
         rc = qfle3_test_nvram_tbl(adapter, nvram_tbl2, buf);
         if (rc)
            goto test_nvram_exit;
      }
   }

   rc = qfle3_test_nvram_dirs(adapter, buf);

test_nvram_exit:
   qfle3_heap_free(buf);
   return rc;
}

u8 qfle3_link_test(qfle3_adapter *adapter, u8 is_serdes)
{
	u8 rc = 0;

	if (!QFLE3_NOMCP(adapter)) {
		qfle3_acquire_phy_lock(adapter);
		rc = elink_test_link(&adapter->link_params, &adapter->link_vars,
				     is_serdes);
		qfle3_release_phy_lock(adapter);
	} else
	      if (!CHIP_REV_IS_SLOW(adapter))
		QFLE3_ERR("Bootcode is missing - can not test link\n");

	return rc;
}

static void qfle3_wait_for_link(qfle3_adapter *adapter, u8 link_up, u8 is_serdes)
{
	int cnt = 1400;

	if (link_up) {
		while (qfle3_link_test(adapter, is_serdes) && cnt--)
			vmk_DelayUsecs(20);

		if (cnt <= 0 && qfle3_link_test(adapter, is_serdes))
			QFLE3_DBG(QFLE3_DBG_UPLINK,"Timeout waiting for link up\n");

		cnt = 1400;
		while (!adapter->link_vars.link_up && cnt--)
			vmk_DelayUsecs(20);

		if (cnt <= 0 && !adapter->link_vars.link_up)
			QFLE3_DBG(QFLE3_DBG_UPLINK,
			   "Timeout waiting for link init\n");
	}
}

vmk_PktHandle *qfle3_alloc_lb_packet(qfle3_adapter *adapter, int size)
{
   vmk_PktHandle *pkt;
   char * mapped_data;
   VMK_ReturnStatus status;
   int i;
   status = vmk_PktAlloc(size, &pkt); 
   if (status != VMK_OK) {
      QFLE3_INFO( "Failed to allocate pkt for loopback test\n");
      return NULL;
   }

   mapped_data = (char *) vmk_PktFrameMappedPointerGet(pkt);

   /* Create the lb pkt */
   vmk_Memset(mapped_data, '0', size);
   /* Initialize src and dst MAC */
   vmk_Memcpy(mapped_data, &adapter->hwMacAddr, 6);
   vmk_Memcpy(mapped_data, &adapter->hwMacAddr, 6);
   vmk_Memset(mapped_data+6+6, 0x77, ETH_HLEN - 12);

   for (i = ETH_HLEN; i < size; i++)
	mapped_data[i] = (unsigned char) (i & 0xff);

   if ((vmk_PktFrameLenSet(pkt, 128)) != VMK_OK) {
       vmk_PktRelease(pkt);
       return NULL;
   }
   return pkt;
}


static VMK_ReturnStatus qfle3_run_loopback(qfle3_adapter *adapter, int loopback_mode)
{
   vmk_PktList pktlist;
   struct qfle3_fastpath *fp_rx = &adapter->fp[0];
   struct qfle3_fastpath *fp_tx = &adapter->fp[adapter->num_rxqs_drv];
   VMK_ReturnStatus rc = VMK_FAILURE;
   vmk_uint32 rxDone = 0;
   vmk_uint32 txDone = 0;
   vmk_UplinkQueueID vmkqid;
   u32 qid;
   u8 cos;
        /* check the loopback mode */
   switch (loopback_mode) {
   case QFLE3_PHY_LOOPBACK:
        if (adapter->link_params.loopback_mode != ELINK_LOOPBACK_XGXS) {
                QFLE3_INFO("PHY loopback not supported\n");
                return VMK_FAILURE;
        }
        break;
   case QFLE3_MAC_LOOPBACK:
        if (CHIP_IS_E3(adapter)) {
                int cfg_idx = qfle3_get_link_cfg_idx(adapter);
                if (adapter->port.supported[cfg_idx] &
                    (ELINK_SUPPORTED_10000baseT_Full |
                     ELINK_SUPPORTED_20000baseMLD2_Full |
                     ELINK_SUPPORTED_20000baseKR2_Full))
                        adapter->link_params.loopback_mode = ELINK_LOOPBACK_XMAC;
                else
                        adapter->link_params.loopback_mode = ELINK_LOOPBACK_UMAC;
        } else
                adapter->link_params.loopback_mode = ELINK_LOOPBACK_BMAC;

        elink_phy_init(&adapter->link_params, &adapter->link_vars);
        break;
   case QFLE3_EXT_LOOPBACK:
        if (adapter->link_params.loopback_mode != ELINK_LOOPBACK_EXT) {
                QFLE3_INFO(
                   "Can't configure external loopback\n");
                return VMK_FAILURE;
        }
        break;
   default:
        QFLE3_DBG(QFLE3_DBG_UPLINK, "Command parameters not supported\n");
        return VMK_FAILURE;
   }

   qfle3_tq_pause(adapter,0);
   vmk_DelayUsecs(100);
   // flush out the rx queue first
   vmk_PktListAlloc(&pktlist);
   vmk_PktListInit(pktlist);
   if(qfle3_has_rx_work(fp_rx)){
      rxDone = qfle3_rxeof(adapter, fp_rx, 1024, NULL, VMK_FALSE, pktlist);
   }
   vmk_PktListFree(pktlist);
        /* prepare the loopback packet */
   adapter->lb_pkt = qfle3_alloc_lb_packet(adapter, 128);
   if (VMK_UNLIKELY(!adapter->lb_pkt)) {
                QFLE3_DBG(QFLE3_DBG_UPLINK, "Unable to qfle3_alloc_lb_packet\n");
                goto test_loopback_exit;
   }

   vmkqid = vmk_PktQueueIDGet(adapter->lb_pkt);
   qid = vmk_UplinkQueueIDVal(vmkqid);
     if (qid >= adapter->num_txqs_vmk) {
        VMK_ASSERT(qid == 0);
     }


   qfle3_xmit_pkt(adapter, qid, adapter->lb_pkt);

   vmk_DelayUsecs(100);
   if (qfle3_has_tx_work(fp_tx)){
      for_each_cos_in_tx_queue(fp_tx, cos) {
         txDone += qfle3_txeof(adapter, fp_tx->txdata_ptr[cos], 1024, VMK_FALSE);
      }
   }

   vmk_PktListAlloc(&pktlist);
   vmk_PktListInit(pktlist);
   if(qfle3_has_rx_work(fp_rx)){
      rxDone = qfle3_rxeof(adapter, fp_rx, 1024, NULL, VMK_FALSE, pktlist);
   }
   vmk_PktListFree(pktlist);
   QFLE3_INFO("LOOPBACK TEST, sent %d pkts and received %d pkts\n", txDone, rxDone);
   if (txDone == rxDone)
       rc = VMK_OK;
   else
       rc = VMK_FAILURE;


   qfle3_tq_resume(adapter,0);
test_loopback_exit:
   adapter->link_params.loopback_mode = ELINK_LOOPBACK_NONE;
   return rc;
}

static VMK_ReturnStatus qfle3_test_loopback(qfle3_adapter *adapter)
{
        int res;
        VMK_ReturnStatus rc = VMK_OK;

        if (QFLE3_NOMCP(adapter))
                return VMK_FAILURE;

        if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_QUIESCED))
                return VMK_FAILURE;

        u8 is_serdes =
                (adapter->link_vars.link_status & LINK_STATUS_SERDES_LINK) > 0;

        /* Internal loopback */
        qfle3_initial_phy_init(adapter, QFLE3_LOAD_DIAG);
        qfle3_wait_for_link(adapter, 1, is_serdes);

        qfle3_int_disable(adapter);
        qfle3_netpoll_disable(adapter);
        qfle3_acquire_phy_lock(adapter);
        res = qfle3_run_loopback(adapter, QFLE3_PHY_LOOPBACK);
        if (res) {
                QFLE3_DBG(QFLE3_DBG_UPLINK, "  PHY loopback failed  (res %d)\n", res);
                rc |= QFLE3_PHY_LOOPBACK_FAILED;
        }

        res = qfle3_run_loopback(adapter, QFLE3_MAC_LOOPBACK);
        if (res) {
                QFLE3_DBG(QFLE3_DBG_UPLINK, "  MAC loopback failed  (res %d)\n", res);
                rc |= QFLE3_MAC_LOOPBACK_FAILED;
        }

        qfle3_release_phy_lock(adapter);
        qfle3_netpoll_enable(adapter);
        qfle3_int_enable(adapter);

#if 0
        /* external loopback */
        qfle3_initial_phy_init(adapter, QFLE3_LOAD_LOOPBACK_EXT);
        qfle3_wait_for_link(adapter, 1, is_serdes);
        qfle3_int_disable(adapter);
        qfle3_netpoll_disable(adapter);
        res = qfle3_run_loopback(adapter, QFLE3_EXT_LOOPBACK);
        if (res) {
                QFLE3_DBG(QFLE3_DBG_UPLINK, "EXT loopback failed  (res %d)\n", rc);
                rc |= QFLE3_EXT_LOOPBACK_FAILED;
        } else
            QFLE3_INFO("EXT Loopback passed\n");
        qfle3_netpoll_enable(adapter);
        qfle3_int_enable(adapter);
#endif
        /* Restore to Normal */
        qfle3_initial_phy_init(adapter, QFLE3_LOAD_NORMAL);
        qfle3_wait_for_link(adapter, 1, is_serdes);

        return rc;
}

/* Selftest implementing functions for qedrntv_selftest_cmd_t */
qfle3_selftest_fntbl_t qfle3_selftest_ecore[QFLE3_SELFTEST_MAX] = {
   {
   qfle3_selftest_memory},    // QFLE3_SELFTEST_MEMORY
   {
   qfle3_selftest_interrupt}, // QFLE3_SELFTEST_INTERRUPT
   {
   qfle3_selftest_register},  // QFLE3_SELFTEST_REGISTER
   {
   qfle3_selftest_clock},     // QFLE3_SELFTEST_CLOCK
   {
   qfle3_selftest_nvram},     // QFLE3_SELFTEST_NVRAM
   {
   qfle3_test_loopback},      // lookback test
};

qfle3_selftest_fntbl_t *qfle3_get_selftest(qfle3_selftest_cmd_t selftest_cmd)
{
   
   if ((selftest_cmd < 0) || (selftest_cmd >= QFLE3_SELFTEST_MAX)) {
      return NULL;
   }
   return &qfle3_selftest_ecore[selftest_cmd];
}

VMK_ReturnStatus
qfle3_selftest_len(vmk_AddrCookie drv_data, vmk_uint32 * len)
{
   //qedentv_udev_t *udev = (qedentv_udev_t *) drv_data.ptr;

   *len = QFLE3_SELFTEST_MAX;
   return VMK_OK;
}
VMK_ReturnStatus
qfle3_selftest_run(vmk_AddrCookie drv_data,
                     vmk_Bool online,
                     vmk_Bool * passed,
                     vmk_UplinkSelfTestResult *resultBuf,
                     vmk_UplinkSelfTestString *stringsBuf)
{
   qfle3_adapter *adapter = (qfle3_adapter *)drv_data.ptr;
   qfle3_selftest_cmd_t cmd;
//   int rc;

   /* Initialise with success for all tests. */
   vmk_Memset(resultBuf, 0, sizeof(vmk_UplinkSelfTestResult) *
                                              QFLE3_SELFTEST_MAX);

   /* Copy test names to output buffer. */
   vmk_Memcpy(stringsBuf, selftest_strings, 32 * QFLE3_SELFTEST_MAX);

   *passed = VMK_TRUE;

   /* Run all of the self-tests. */
   for (cmd = 0; cmd < QFLE3_SELFTEST_MAX; cmd++) {
      if (qfle3_selftest_ecore[cmd].fn_ptr != NULL) {

//         rc = (qfle3_selftest_ecore[cmd].fn_ptr)(adapter);
//         if (rc != ECORE_SUCCESS){
//            resultBuf[cmd] = 1;
//            *passed = VMK_FALSE;
//         }
      }
      
      QFLE3_INFO("Selftest: %s=> %d\n",
                       stringsBuf[cmd], (int)resultBuf[cmd]);
   }

   return VMK_OK;
}

static vmk_UplinkSelfTestOps qfle3_self_test_ops = {
   .selfTestResultLenGet = qfle3_selftest_len,
   .selfTestRun = qfle3_selftest_run,
};

static VMK_ReturnStatus
qfle3_uplink_caps_register(qfle3_adapter * adapter)
{
   VMK_ReturnStatus status = VMK_OK;
   vmk_UplinkQueueLROConstraints lroConstraints;

   status = vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_SG_TX, NULL);
   VMK_ASSERT((status == VMK_OK),"SG_TX register failed");

   status = vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_MULTI_PAGE_SG, NULL);
   VMK_ASSERT((status == VMK_OK),"MPAGE_SG register failed");

   status = vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_MESSAGE_LEVEL,
				  &qfle3_message_level_ops);
   VMK_ASSERT((status == VMK_OK),"message level register failed");

   status = vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_LINK_STATUS_SET,
				  qfle3_linkstatus_setcb);
   VMK_ASSERT((status == VMK_OK),"message level register failed");

   status = vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_PAUSE_PARAMS, &qfle3_pauseparam);

   VMK_ASSERT((status == VMK_OK),"Pause Params register failed");

   status = vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_WAKE_ON_LAN, &qfle3_wol);

   VMK_ASSERT((status == VMK_OK),"Wake-On-LAN register failed");

   status = vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_COALESCE_PARAMS, &qfle3_coalesce);

   VMK_ASSERT((status == VMK_OK),"Coalesce Params register failed");

   status = vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_RING_PARAMS, &qfle3_ring);

   VMK_ASSERT((status == VMK_OK),"Ring Params register failed");

   if (adapter->hw_vlan_en) {
      status = vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_VLAN_TX_INSERT, NULL);
      VMK_ASSERT((status == VMK_OK),"vlan tx insert register failed");

      status = vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_VLAN_RX_STRIP, NULL);
      VMK_ASSERT((status == VMK_OK),"vlan rx strip register failed");
   }
   /*
    * Netdump, firmware dump, EEPROM dump?
    */
   status = vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_NETWORK_DUMP, &qfle3_netdump_ops);
   VMK_ASSERT((status == VMK_OK),"netdump register failed");

   // vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_REGDUMP, &qfle3RegDumpOps);
   /*
    * Private stats, accessed by "vsish -e get /net/pNics/vmnicxx/stats"
    */
   vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_PRIV_STATS, &qfle3_priv_stats_ops);
   VMK_ASSERT((status == VMK_OK),"private stats register failed");

   if (adapter->offloadflags & QFLE3_FLAGS_CSUM_OFFLOAD_ENABLED) {
      status = vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_IPV4_CSO, NULL);
      VMK_ASSERT((status == VMK_OK),"IPv4 CSO register failed");
      status = vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_IPV6_CSO, NULL);
      VMK_ASSERT((status == VMK_OK),"IPv6 CSO register failed");

   }

#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)
   if (adapter->offloadflags & QFLE3_FLAGS_TSO_ENABLED) {
      status = vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_IPV4_TSO, &tso_cons);
      VMK_ASSERT((status == VMK_OK),"IPv4 TSO register failed");
      status = vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_IPV6_TSO, &tso_cons);
      VMK_ASSERT((status == VMK_OK),"IPv6 TSO register failed");
   }
#endif

   if (adapter->offloadflags & QFLE3_FLAGS_VXLAN_OFFLOAD_ENABLED) {
      status = vmk_UplinkCapRegister(adapter->uplink,
#if (VMKAPI_REVISION >= VMK_REVISION_FROM_NUMBERS(2, 5, 0, 0))
				     VMK_UPLINK_CAP_VXLAN_OFFLOAD,
#else
				     VMK_UPLINK_CAP_ENCAP_OFFLOAD,
#endif
				     &qfle3EncapOffloadOps);
      VMK_ASSERT((status == VMK_OK),"ENCAP OFFLOAD register failed");
   }

   /*
    * Register Netqueue capability only if multiple TX queues need to be suported.
    */
   if (adapter->num_txqs_vmk > 1) {
      vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_MULTI_QUEUE,
			    &qfle3_multiqueue_ops);
      VMK_ASSERT((status == VMK_OK),"Multiple queue register failed");
   }

   if (adapter->lro_enable) {
      lroConstraints.supportedTypes = VMK_UPLINK_QUEUE_LRO_TYPE_IPV4 |
	 VMK_UPLINK_QUEUE_LRO_TYPE_IPV6;
      vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_LRO, &lroConstraints);
      VMK_ASSERT((status == VMK_OK),"Multiple queue register failed");
   }
#if (ESX_DDK_VERSION >= 2017)
   if (adapter->num_rss_engines > 0) {
      status = vmk_UplinkQueueRegisterFeatureOps(adapter->uplink,
                                                 VMK_UPLINK_QUEUE_FEAT_RSS_GENERIC,
                                                 &qfle3_rss_generic_ops);
      
         VMK_ASSERT(status == VMK_OK, "Failed to register RSS ops");
      }
#else
   if (adapter->num_rssqs_nd > 1) {
      status = vmk_UplinkQueueRegisterFeatureOps(adapter->uplink,
						 VMK_UPLINK_QUEUE_FEAT_RSS_DYN,
						 &qfle3_rss_dynamic_ops);
      
         VMK_ASSERT(status == VMK_OK, "Failed to register RSS ops");
      }
#endif

   status = vmk_UplinkCapRegister(adapter->uplink,
				  VMK_UPLINK_CAP_REGDUMP,
				  &qfel3_regdump_ops);

   VMK_ASSERT(status == VMK_OK, "Failed to register regdump ops");
#if (VMKAPI_REVISION >= VMK_API_2_4_0_0)

   vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_ADVERTISE_MODES,
                         &qfle3_adv_modes_ops);
#endif
   status = vmk_UplinkCapRegister(adapter->uplink,
		   VMK_UPLINK_CAP_EEPROM,
		   &qfle3_eeprom_ops);
   VMK_ASSERT(status == VMK_OK);

   status = vmk_UplinkCapRegister(adapter->uplink,
		   VMK_UPLINK_CAP_TRANSCEIVER_TYPE,
		   &qfle3_xcvr_ops);
   VMK_ASSERT(status == VMK_OK);

   status = vmk_UplinkCapRegister(adapter->uplink,
		   VMK_UPLINK_CAP_CABLE_TYPE,
		   &qfle3_cable_type_ops);
   VMK_ASSERT(status == VMK_OK);

#ifdef QFLE3_SRIOV
   if(adapter->vfdb){
      status = vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_SRIOV, NULL);
      VMK_ASSERT(status == VMK_OK);
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Setting SRIOV CAP\n");
   } else {
      QFLE3_DBG(QFLE3_DBG_IOV, "SRIOV: Not Enabled hence not Setting SRIOV CAP\n");
   }
#endif //QFLE3_SRIOV

   // self test
   status = vmk_UplinkCapRegister(adapter->uplink,
                                  VMK_UPLINK_CAP_SELF_TEST,
                                  &qfle3_self_test_ops);
   VMK_ASSERT(status == VMK_OK);

   if (QFLE3_IS_GENEVE_OFFLOAD_ENABLED(adapter)) {
      status =
         vmk_UplinkCapRegister(adapter->uplink, VMK_UPLINK_CAP_GENEVE_OFFLOAD,
                               &qfle3_geneve_offload_ops);
      VMK_ASSERT(status == VMK_OK, "Failed to register Geneve capability\n");
   }

   return VMK_OK;
}


static VMK_ReturnStatus
qfle3_uplink_associate(vmk_AddrCookie driverData, vmk_Uplink uplink)
{
   vmk_int32 i;
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   VMK_ReturnStatus status;
   vmk_Name poll_name;

   adapter->uplink = uplink;
   adapter->uplinkName = vmk_UplinkNameGet(uplink);
   QFLE3_DBG(QFLE3_DBG_KERNEL,
	     "Uplink.associate with uplink %p", uplink);
   if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING)){
      QFLE3_ERR("Uplink is Resetting, Rejecting Uplink Associate");
      return VMK_FAILURE;
   }
   for (i = 0; i < QFLE3_TOTAL_ETH_QUEUES(adapter); i++) {
      status = vmk_NameFormat(&poll_name, "%s_np[%d]",
			      vmk_NameToString(&adapter->uplinkName), i);
      if (status != VMK_OK) {
         QFLE3_ERR("Error in creating poll name.\n");
      }

      status = vmk_NetPollRegisterUplink(adapter->fp[i].netpoll,
					 uplink,
					 poll_name,
					 i ? VMK_FALSE : VMK_TRUE);

      if (status != VMK_OK) {
         QFLE3_ERR("Failed to register RX netpoll\n");
      }
   }

   status = vmk_UplinkSetWatchdogTimeout(adapter->uplink, 60 * 1000);
   if (status != VMK_OK) {
	   QFLE3_WARN("Failed to set watchdog timeout to 10 seconds.\n");
   }
   status =
      vmk_MgmtRegisterInstanceCallbacks(vmkmgmt_api_handler, (vmk_uint64)adapter,
                                        vmk_ModuleCurrentID,
                                        qfle3_mod_info.heapID,
                                        &adapter->uplinkName,
                                        QFLE3_NUM_CALLBACKS,
                                        qfle3_mgmt_callbacks);
   if (status != VMK_OK) {
      QFLE3_ERR("Error in registering instance to the management handler.\n");
   }
   
   status = qfle3_uplink_caps_register(adapter);
   
   QFLE3_DBG(QFLE3_DBG_KERNEL, "Uplink.associate Done");
   return status;
}

static VMK_ReturnStatus
qfle3_uplink_disassociate(vmk_AddrCookie driverData)
{
   vmk_int32 i;
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   VMK_ReturnStatus status = VMK_OK;
   QFLE3_DBG(QFLE3_DBG_KERNEL,  "Uplink.diassociate");
   if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING)){
      QFLE3_DBG(QFLE3_DBG_UPLINK, "Uplink is Resetting, Rejecting Uplink Disassciate");
      return VMK_FAILURE;
   }

   for (i = 0; i < QFLE3_TOTAL_ETH_QUEUES(adapter); i++) {
      vmk_NetPollUnregisterUplink(adapter->fp[i].netpoll);
   }

   adapter->uplink = NULL;
   vmk_NameInitialize(&adapter->uplinkName, "");
   vmk_MgmtUnregisterInstanceCallbacks(vmkmgmt_api_handler, (vmk_uint64)adapter);
   
   QFLE3_DBG(QFLE3_DBG_KERNEL,  "Uplink.diassociate, Done");
   return status;
}

static vmk_uint8 *uplink_cap_string[39] = {
   "CAP_0",
   "SG_TX",
   "SG_RX",
   "MULTI_PAGE_SG",
   "IPV4_CSO",
   "IPV6_CSO",
   "IPV6_EXT_CSO",
   "VLAN_RX_STRIP",
   "VLAN_TX_INSERT",
   "IPV4_TSO",
   "IPV6_TSO",
   "IPV6_EXT_TSO",
   "MOD_TX_HDRS",
   "NO_SCHEDULER",
   "PRIV_STATS",
   "LINK_STATUS_SET",
   "MAC_ADDR_SET",
   "COALESCE_PARAMS",
   "VLAN_FILTER",
   "WAKE_ON_LAN",
   "NETWORK_DUMP",
   "MULTI_QUEUE",
   "DCB",
   "UPT",
   "SRIOV",
   "ENCAP_OFFLOAD",
   "OFFLOAD_CONSTRAINTS",
   "EEPROM_DUMP",
   "REG_DUMP",
   "SELF_TEST",
   "PAUSE_PARAMS",
   "RESTART_NEG",
   "LRO",
   "CABLE_TYPE",
   "PHY_ADDRESS",
   "TRANSCEIVER_TYPE",
   "MESSAGE_LEVEL",
   "RING_PARAMS",
   "GENEVE_OFFLOAD"
};

vmk_uint8 *qfle3_cap_decoder(qfle3_adapter *adapter, vmk_UplinkCap cap)
{
   if (cap <= 0 || cap > 38) {
      QFLE3_INFO("Incorrect capability %d", cap);
      return NULL;
   }
   return uplink_cap_string[cap];
}

#if 0
static VMK_ReturnStatus
qfle3_change_vlan_config(qfle3_adapter *adapter,
						 vmk_UplinkCap cap, vmk_Bool en)
{
   vmk_uint16 sm_state;
   QFLE3_SMCMD_STATUS cmd_status;
	
   sm_state = qfle3_sm_getstate(adapter);
	
   // Check to make sure we are in IOSTARTED STATE AND that the adapter is not in
   // the process of resetting, otherwise, cache the value and return to process it
   // at a later time;  for now we'll only check the adapter state and deal with
   // reset later
   adapter->uplinkCachedData.en_vlan = en;
   adapter->uplinkCachedNewData |= QFLE3_UPLINK_VLAN_CONFIG_CHANGED;
   QFLE3_DBG(QFLE3_DBG_SM,
		   "VLAN %s, caching vlan change\n", en ? "enable" : "disable");
   QFLE3_DBG(QFLE3_DBG_SM,
		   "uplinkCachedNewdata 0x%x\n",
		   adapter->uplinkCachedNewData);
	
   if (sm_state == QFLE3_SM_IOSTARTED)
   {
      if (vmk_BitVectorAtomicTestAndSet(adapter->state, QFLE3_STATE_BIT_RESETTING)) {
		  return VMK_OK;
      }
	
      //qfle3_link_down(adapter);
      cmd_status = qfle3_sm_cmd(adapter, QFLE3_SMCMD_UPLINKRESET, 0);
	
      if (cmd_status == QFLE3_SMCMD_STATUS_COMPLETED){
		  return VMK_OK;
      }else if (cmd_status == QFLE3_SMCMD_STATUS_WRONGSTATE){
		  QFLE3_ERR("Changing VLAN config in the wrong state %s", qfle3_sm_get_string(sm_state));
		  return VMK_FAILURE;
      }else {
		  QFLE3_ERR("ERROR changing VLAN config");
		  return VMK_FAILURE;
      }
   }
   return VMK_OK;
}
#endif

static VMK_ReturnStatus
qfle3_uplink_cap_enable(vmk_AddrCookie driverData, vmk_UplinkCap cap)
{
	qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
	QFLE3_DBG(QFLE3_DBG_KERNEL,  "Uplink.cap_enable %s",
			qfle3_cap_decoder(adapter, cap));
	VMK_ReturnStatus status = VMK_OK;

#if 0
	switch (cap) {
		case VMK_UPLINK_CAP_VLAN_RX_STRIP:
			status = qfle3_change_vlan_config(adapter, cap, VMK_TRUE);
			break;
		case VMK_UPLINK_CAP_VLAN_TX_INSERT:
			//Nothing needs to be done in driver.
		default:
			break;
	}

	if (status == VMK_OK) {
		QFLE3_DBG(QFLE3_DBG_UPLINK,  "Enabled capability: %s",
				qfle3_cap_decoder(adapter, cap));
	} else {
		QFLE3_DBG(QFLE3_DBG_UPLINK,  "Failed to enabled capability: %s",
				qfle3_cap_decoder(adapter, cap));
	}
#endif

	return status;

}

static VMK_ReturnStatus
qfle3_uplink_cap_disable(vmk_AddrCookie driverData, vmk_UplinkCap cap)
{
	qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
	QFLE3_DBG(QFLE3_DBG_KERNEL,  "Uplink.cap_disable %s", qfle3_cap_decoder(adapter, cap));
	VMK_ReturnStatus status = VMK_OK;

#if 0
	switch (cap) {
		case VMK_UPLINK_CAP_VLAN_RX_STRIP:
			status = qfle3_change_vlan_config(adapter, cap, VMK_FALSE);
			break;
		case VMK_UPLINK_CAP_VLAN_TX_INSERT:
			//Nothing needs to be done in driver.
		default:
			break;
	}

	if (status == VMK_OK) {
		QFLE3_DBG(QFLE3_DBG_UPLINK,  "Disabled capability: %s",
				qfle3_cap_decoder(adapter, cap));
	} else {
		QFLE3_DBG(QFLE3_DBG_UPLINK,  "Failed to disable capability: %s",
				qfle3_cap_decoder(adapter, cap));
	}
#endif

	return status;
}

VMK_ReturnStatus
qfle3_interrupt_ack(void *handlerData, vmk_IntrCookie intrCookie)
{
   return VMK_OK;
}

VMK_ReturnStatus
qfle3_single_interrupt_ack(void *handlerData, vmk_IntrCookie intrCookie)
{
   qfle3_adapter *adapter = (qfle3_adapter *) handlerData;
   vmk_uint32 intr_sts = qfle3_ack_int(adapter);

   /*
    * the interrupt is not for us
    */
   if (VMK_UNLIKELY(intr_sts == 0)) {
      return VMK_NOT_THIS_DEVICE;
   }

   vmk_CPUMemFenceReadWrite();
   adapter->intr_sts |= intr_sts;
   vmk_CPUMemFenceReadWrite();

   return VMK_OK;
}

/*
 * When a TPA aggregation is completed, loop through the individual mbufs
 * of the aggregation, combining them into a single mbuf which will be sent
 * up the stack. Refill all freed SGEs with mbufs as we go along.
 */
static int
qfle3_fill_frag_mbuf(qfle3_adapter * adapter,
		     struct qfle3_fastpath *fp,
		     struct qfle3_sw_tpa_info *tpa_info,
		     vmk_uint16 queue,
		     vmk_uint16 pages,
		     vmk_PktHandle * pkt,
		     struct eth_end_agg_rx_cqe *cqe, vmk_uint16 cqe_idx)
{
   vmk_uint32 frag_len, frag_size, i;
   vmk_uint16 sge_idx;
   vmk_int32 rc = 0;
   vmk_int32 j;
   vmk_MPN frag_page;

   frag_size = le16toh(cqe->pkt_len) - tpa_info->len_on_bd;

   QFLE3_DBG(QFLE3_DBG_LRO,
	     "fp[%02d].tpa[%02d] TPA fill len_on_bd=%d frag_size=%d pages=%d\n",
	     fp->qid, queue, tpa_info->len_on_bd, frag_size, pages);

   /*
    * make sure the aggregated frame is not too big to handle
    */
   if (pages > 8 * PAGES_PER_SGE) {
      QFLE3_DBG(QFLE3_DBG_LRO, "fp[%02d].sge[0x%04x] has too many pages (%d)! "
		"pkt_len=%d len_on_bd=%d frag_size=%d\n",
		fp->qid, cqe_idx, pages, le16toh(cqe->pkt_len),
		tpa_info->len_on_bd, frag_size);
      VMK_ASSERT(pages <= 8);
      //return (-1);
   }

   /*
    * Scan through the scatter gather list pulling individual mbufs into a
    * single mbuf for the host stack.
    */
   for (i = 0, j = 0; i < pages; i += PAGES_PER_SGE, j++) {
      sge_idx = RX_SGE(le16toh(cqe->sgl_or_raw_data.sgl[j]));

      /*
       * Firmware gives the indices of the SGE as if the ring is an array
       * (meaning that the "next" element will consume 2 indices).
       */
      frag_len = min(frag_size, (vmk_uint32) (SGE_PAGES));

      QFLE3_DBG(QFLE3_DBG_LRO, "fp[%02d].tpa[%02d] TPA fill i=%d j=%d "
		"sge_idx=%d frag_size=%d frag_len=%d\n",
		fp->qid, queue, i, j, sge_idx, frag_size, frag_len);

      frag_page = fp->rx_sge_mbuf_chain[sge_idx].page;

      /*
       * allocate a new mbuf for the SGE
       */
      rc = qfle3_alloc_rx_sge_mbuf(fp, sge_idx);
      if (rc) {
	 /*
	  * Leave all remaining SGEs in the ring!
	  */

         QFLE3_ERR("fp[%02d].tpa[%02d] Could not alloc sge mbuf\n", fp->qid, queue);
	 return (rc);
      }

      /*
       * update the fragment length
       */
      //m_frag->m_len = frag_len;

      /*
       * concatenate the fragment to the head mbuf
       */
      //m_cat(m, m_frag);
      //fp->eth_q_stats.mbuf_alloc_sge--;
      rc = vmk_PktPageAppend(pkt,
			     frag_page, vmk_PktFrameLenGet(pkt), 0, frag_len);
      VMK_ASSERT(rc == VMK_OK);
      /*
       * update the TPA mbuf size and remaining fragment size
       */
      //m->m_pkthdr.len += frag_len;
      vmk_PktFrameLenSet(pkt, vmk_PktFrameLenGet(pkt) + frag_len);
      frag_size -= frag_len;
   }
   QFLE3_DBG(QFLE3_DBG_LRO,
	     "fp[%02d].tpa[%02d] TPA fill done frag_size=%d\n",
	     fp->qid, queue, frag_size);

   return (rc);
}


inline void
qfle3_update_rx_prod(struct qfle3_adapter *adapter,
		     struct qfle3_fastpath *fp,
		     vmk_uint16 rx_bd_prod, vmk_uint16 rx_cq_prod, vmk_uint16 rx_sge_prod)
{
   struct ustorm_eth_rx_producers rx_prods = { 0 };
   vmk_uint32 i;

   /*
    * update producers
    */
   rx_prods.bd_prod = rx_bd_prod;
   rx_prods.cqe_prod = rx_cq_prod;
   rx_prods.sge_prod = rx_sge_prod;

   /*
    * Make sure that the BD and SGE data is updated before updating the
    * producers since FW might read the BD/SGE right after the producer
    * is updated.
    * This is only applicable for weak-ordered memory model archs such
    * as IA-64. The following barrier is also mandatory since FW will
    * assumes BDs must have buffers.
    */

   vmk_CPUMemFenceReadWrite();
   for (i = 0; i < (sizeof(rx_prods) / 4); i++) {
      REG_WR(adapter,
	     (fp->ustorm_rx_prods_offset + (i * 4)), ((vmk_uint32 *) & rx_prods)[i]);
   }
   vmk_CPUMemFenceReadWrite();

   QFLE3_DBG(QFLE3_DBG_RX,
	     "RX fp[%d]: wrote prods bd_prod=%d cqe_prod=%d sge_prod=%d\n",
	     fp->qid, rx_bd_prod, rx_cq_prod, rx_sge_prod);
}

static inline void
qfle3_clear_sge_mask_next_elems(struct qfle3_fastpath *fp)
{
   int i, j;

   for (i = 1; i <= RX_SGE_NUM_PAGES; i++) {
      int idx = RX_SGE_TOTAL_PER_PAGE * i - 1;

      for (j = 0; j < 2; j++) {
	 BIT_VEC64_CLEAR_BIT(fp->sge_mask, idx);
	 idx--;
      }
   }
}

inline void
qfle3_init_sge_ring_bit_mask(struct qfle3_fastpath *fp)
{
   /*
    * set the mask to all 1's, it's faster to compare to 0 than to 0xf's
    */
   vmk_Memset(fp->sge_mask, 0xff, sizeof(fp->sge_mask));

   /*
    * Clear the two last indices in the page to 1. These are the indices that
    * correspond to the "next" element, hence will never be indicated and
    * should be removed from the calculations.
    */
   qfle3_clear_sge_mask_next_elems(fp);
}

static inline void
qfle3_update_last_max_sge(struct qfle3_fastpath *fp, vmk_uint16 idx)
{
   vmk_uint16 last_max = fp->last_max_sge;

   if (SUB_S16(idx, last_max) > 0) {
      fp->last_max_sge = idx;
   }
}

#define BIT_VEC64_ELEM_ONE_MASK ((vmk_uint64)(~0))


static inline void
qfle3_update_sge_prod(qfle3_adapter * adapter,
		      struct qfle3_fastpath *fp,
		      vmk_uint16 sge_len, struct eth_end_agg_rx_cqe *cqe)
{
   vmk_uint16 last_max, last_elem, first_elem;
   vmk_uint16 delta = 0;
   vmk_uint16 i;

   if (!sge_len) {
      return;
   }

   /*
    * first mark all used pages
    */
   for (i = 0; i < sge_len; i++) {
      BIT_VEC64_CLEAR_BIT(fp->sge_mask, RX_SGE(le16toh(cqe->sgl_or_raw_data.sgl[i])));
   }

   QFLE3_DBG(QFLE3_DBG_LRO,
	     "fp[%02d] fp_cqe->sgl[%d] = %d\n",
	     fp->qid, sge_len - 1, le16toh(cqe->sgl_or_raw_data.sgl[sge_len - 1]));

   /*
    * assume that the last SGE index is the biggest
    */
   qfle3_update_last_max_sge(fp, le16toh(cqe->sgl_or_raw_data.sgl[sge_len - 1]));

   last_max = RX_SGE(fp->last_max_sge);
   last_elem = last_max >> BIT_VEC64_ELEM_SHIFT;
   first_elem = RX_SGE(fp->rx_sge_prod) >> BIT_VEC64_ELEM_SHIFT;

   /*
    * if ring is not full
    */
   if (last_elem + 1 != first_elem) {
      last_elem++;
   }

   /*
    * now update the prod
    */
   for (i = first_elem; i != last_elem; i = RX_SGE_NEXT_MASK_ELEM(i)) {
      if (VMK_LIKELY(fp->sge_mask[i])) {
	 break;
      }

      fp->sge_mask[i] = BIT_VEC64_ELEM_ONE_MASK;
      delta += BIT_VEC64_ELEM_SZ;
   }

   if (delta > 0) {
      fp->rx_sge_prod += delta;
      /*
       * clear page-end entries
       */
      qfle3_clear_sge_mask_next_elems(fp);
   }

   QFLE3_DBG(QFLE3_DBG_LRO,
	     "fp[%02d] fp->last_max_sge=%d fp->rx_sge_prod=%d\n",
	     fp->qid, fp->last_max_sge, fp->rx_sge_prod);
}

void
qfle3_free_rx_bd_chain(struct qfle3_fastpath *fp)
{
   qfle3_adapter *adapter = fp->adapter;
   qfle3_rxbuf_info *rx_buf;
   int i;
   vmk_int32 rx_bd_total = RX_BD_TOTAL;

   if (IS_FCOE_FP(fp) || IS_OOO_FP(fp) || IS_FWD_FP(fp)) {
      rx_bd_total = RX_BD_TOTAL_PER_PAGE * FCOE_NUM_RX_PAGES_DEF;
   }

   /*
    * free all mbufs and unload all maps
    */
   for (i = 0; i < rx_bd_total; i++) {
      rx_buf = &fp->rxbuf_chain[i];
      if (rx_buf->dmaAddr) {
   	 qfle3_dma_unmap(adapter, rx_buf->dmaAddr, rx_buf->len,
   			 VMK_DMA_DIRECTION_TO_MEMORY, adapter->dmaEngine_rxbuf);
   	 rx_buf->dmaAddr = 0;
      }

      if (rx_buf->pkt != NULL) {
	 vmk_PktRelease(rx_buf->pkt);
	 rx_buf->pkt = NULL;
	 //fp->eth_q_stats.mbuf_alloc_rx--;
      }
	  rx_buf->bufType = QFLE3_RX_BUF_NONE;
	  rx_buf->len = 0;
   }
}

void
qfle3_free_tpa_pool(struct qfle3_fastpath *fp)
{
   qfle3_adapter *adapter = fp->adapter;
   qfle3_rxbuf_info *rx_tpa_bd;;
   int i, max_agg_queues;

   max_agg_queues = MAX_AGG_QS(adapter);

   /*
    * release all mbufs and unload all DMA maps in the TPA pool
    */
   for (i = 0; i < max_agg_queues; i++) {
      rx_tpa_bd = &fp->rx_tpa_info[i].bd;
      if (rx_tpa_bd->dmaAddr) {
	 qfle3_dma_unmap(adapter, rx_tpa_bd->dmaAddr,
			 rx_tpa_bd->len,
			 VMK_DMA_DIRECTION_TO_MEMORY, adapter->dmaEngine_rxbuf);
	 rx_tpa_bd->dmaAddr = 0;
      }

      if (rx_tpa_bd->pkt != NULL) {
	 vmk_PktRelease(rx_tpa_bd->pkt);
	 rx_tpa_bd->pkt = NULL;
	 //fp->eth_q_stats.mbuf_alloc_tpa--;
      }
	  rx_tpa_bd->bufType = QFLE3_RX_BUF_NONE;
	  rx_tpa_bd->len = 0;
   }
}

void
qfle3_free_sge_chain(struct qfle3_fastpath *fp)
{
   qfle3_adapter *adapter = fp->adapter;
   int i;

   /*
    * rree all mbufs and unload all maps
    */
   for (i = 0; i < RX_SGE_TOTAL; i++) {
      if (fp->rx_sge_mbuf_chain[i].dmaAddr) {
	 qfle3_page_unmap_and_free(adapter,
				   fp->rx_sge_mbuf_chain[i].page,
				   fp->rx_sge_mbuf_chain[i].dmaAddr);
	 fp->rx_sge_mbuf_chain[i].dmaAddr = 0;
      }

	  fp->rx_sge_mbuf_chain[i].bufType = QFLE3_RX_BUF_NONE;
	  fp->rx_sge_mbuf_chain[i].len = 0;
      fp->rx_sge_mbuf_chain[i].page = 0;
      //fp->eth_q_stats.mbuf_alloc_sge--;
   }
}


VMK_ReturnStatus
qfle3_alloc_rx_bd_mbuf(struct qfle3_fastpath *fp, vmk_uint16 index)
{
   qfle3_rxbuf_info *rx_buf;
   struct qfle3_adapter *adapter = fp->adapter;
   vmk_PktHandle *newpkt = NULL;
   vmk_IOA ioaddr;
   struct eth_rx_bd *rx_bd;
   VMK_ReturnStatus status = 0;

   /*
    * allocate the new RX BD mbuf
    */
   status = qfle3_pkt_alloc_and_map(adapter, fp->rx_buf_alloc_size,
				    adapter->dmaEngine_rxbuf, &newpkt, &ioaddr);
   if (status != VMK_OK) {
      QFLE3_ERR("fp[%d], allocate #%d pkt failed", fp->qid, index);
      return status;
   }

   rx_buf = &fp->rxbuf_chain[index];
   if (rx_buf->dmaAddr) {
      qfle3_dma_unmap(adapter, rx_buf->dmaAddr, rx_buf->len,
		      VMK_DMA_DIRECTION_TO_MEMORY, adapter->dmaEngine_rxbuf);
      rx_buf->dmaAddr = 0;
   }

   rx_buf->pkt = newpkt;
   rx_buf->dmaAddr = ioaddr;
   rx_buf->len = fp->rx_buf_alloc_size;
   rx_buf->bufType = QFLE3_RX_BUF_PKT;

   rx_bd = &fp->rx_chain[index];
   rx_bd->addr_hi = htole32(U64_HI(ioaddr));
   rx_bd->addr_lo = htole32(U64_LO(ioaddr));

   return (status);
}



/*
 * Allocate an mbuf and assign it to the receive scatter gather chain. The
 * caller must take care to save a copy of the existing mbuf in the SG mbuf
 * chain.
 */
int
qfle3_alloc_rx_sge_mbuf(struct qfle3_fastpath *fp, vmk_uint16 index)
{
   qfle3_adapter *adapter = fp->adapter;
   vmk_MPN new_mpn;
   qfle3_rxbuf_info *sge_buf;
   struct eth_rx_sge *sge;
   vmk_IOA ioaddr;
   VMK_ReturnStatus status = 0;

   /*
    * allocate a new SGE mbuf
    */
   status = qfle3_page_alloc_and_map(adapter, adapter->dmaEngine, &new_mpn, &ioaddr);
   if (status != VMK_OK)
      return (status);

   sge_buf = &fp->rx_sge_mbuf_chain[index];
   if (sge_buf->dmaAddr) {
      qfle3_dma_unmap(adapter, sge_buf->dmaAddr, VMK_PAGE_SIZE,
		      VMK_DMA_DIRECTION_TO_MEMORY, adapter->dmaEngine);
   }

   sge_buf->dmaAddr = ioaddr;
   sge_buf->bufType = QFLE3_RX_BUF_PAGE;
   sge_buf->len = SGE_PAGE_SIZE;
   /*
    * save the mbuf and mapping info for a future packet
    */
   sge_buf->page = new_mpn;

   sge = &fp->rx_sge_chain[index];
   sge->addr_hi = htole32(U64_HI(ioaddr));
   sge->addr_lo = htole32(U64_LO(ioaddr));

   return (status);
}

int
qfle3_alloc_rx_tpa_mbuf(struct qfle3_fastpath *fp, int queue)
{
   struct qfle3_sw_tpa_info *tpa_info = &fp->rx_tpa_info[queue];
   qfle3_adapter *adapter = fp->adapter;
   vmk_PktHandle *newpkt;
   vmk_IOA ioaddr;
   VMK_ReturnStatus status = 0;

   /*
    * allocate the new TPA mbuf
    */
   status = qfle3_pkt_alloc_and_map(adapter, fp->rx_buf_alloc_size,
				    adapter->dmaEngine_rxbuf, &newpkt, &ioaddr);
   if (status != VMK_OK) {
      QFLE3_ERR("fp[%d], allocate #%d TPA pkt failed", fp->qid, queue);
      return (status);
   }
   if (tpa_info->bd.dmaAddr) {
      qfle3_dma_unmap(adapter, tpa_info->bd.dmaAddr,
		      tpa_info->bd.len,
		      VMK_DMA_DIRECTION_TO_MEMORY, adapter->dmaEngine_rxbuf);
   }
   tpa_info->bd.pkt = newpkt;
   tpa_info->bd.dmaAddr = ioaddr;
   tpa_info->bd.len = fp->rx_buf_alloc_size;
   tpa_info->bd.bufType = QFLE3_RX_BUF_PKT;
   return (status);
}
void qfle3_init_txdata(struct qfle3_adapter *adapter,
				     struct qfle3_fp_txdata *txdata, vmk_uint32 cid,
				     int txq_index, vmk_uint16 *tx_cons_sb,
				     struct qfle3_fastpath *fp)
{

	txdata->cid = cid;
	txdata->txq_index = txq_index;
	txdata->tx_cons_sb = tx_cons_sb;
	txdata->parent_fp = fp;
	txdata->tx_ring_size = IS_FCOE_FP(fp) ? MAX_TX_AVAIL : adapter->tx_ring_size;

	QFLE3_DBG(QFLE3_DBG_UPLINK, "created tx data cid %d, txq %d\n",
	   txdata->cid, txdata->txq_index);
}


VMK_ReturnStatus
qfle3_alloc_fp_buffers(struct qfle3_fastpath * fp)
{
   vmk_int32 j;
   VMK_ReturnStatus rc = 0;
   int ring_prod, cqe_ring_prod;
   qfle3_adapter *adapter = fp->adapter;
   vmk_int32 max_agg_queues;
   vmk_int32 rx_bd_total = RX_BD_TOTAL;
   vmk_int32 rx_bd_usable = RX_BD_USABLE;
   vmk_int32 rcq_total = RCQ_TOTAL;

   if (IS_FCOE_FP(fp) || IS_OOO_FP(fp) || IS_FWD_FP(fp)) {
      rx_bd_total = FCOE_NUM_RX_PAGES_DEF * RX_BD_TOTAL_PER_PAGE;
      rx_bd_usable = RX_BD_USABLE_PER_PAGE * FCOE_NUM_RX_PAGES_DEF;
      rcq_total = RCQ_TOTAL_PER_PAGE * FCOE_NUM_RX_PAGES_DEF *CQE_BD_REL;
   }

   ring_prod = cqe_ring_prod = 0;
   VMK_ASSERT((fp->rxbuf_chain != NULL), "rxbuf_chain is %ld", (vmk_uintptr_t) fp->rxbuf_chain);
   vmk_Memset(fp->rxbuf_chain, 0, rx_bd_total * sizeof(qfle3_rxbuf_info));


   /*
    * allocate buffers for the RX BDs in RX BD chain
    */
   for (j = 0; j < rx_bd_usable; j++) {
      vmk_uint8 tryagain_alloc = 5;

      try_once_again:
      rc = qfle3_alloc_rx_bd_mbuf(fp, ring_prod);
      if (rc != 0) {
         if (tryagain_alloc--)
            goto try_once_again;
         goto qfle3_alloc_fp_buffers_error;
      }

      ring_prod = RX_BD_NEXT(ring_prod);
      cqe_ring_prod = RCQ_NEXT(cqe_ring_prod);
   }

   fp->rx_bd_prod = ring_prod;
   fp->rx_cq_prod = min_t(vmk_uint16, rcq_total, cqe_ring_prod);

   if (fp->tpa_enable) {
      max_agg_queues = ETH_MAX_AGGREGATION_QUEUES_E1H_E2;

      /*
       * fill the TPA pool
       */
      for (j = 0; j < max_agg_queues; j++) {
         rc = qfle3_alloc_rx_tpa_mbuf(fp, j);
         if (rc != 0) {
            QFLE3_ERR( "mbuf alloc fail for fp[%02d] TPA queue %d\n",
               fp->qid, j);
            fp->tpa_enable = VMK_FALSE;
            goto qfle3_alloc_fp_buffers_error;
         }

         fp->rx_tpa_info[j].state = QFLE3_TPA_STATE_STOP;
      }

      /*
       * fill the RX SGE chain
       */
      ring_prod = 0;
      for (j = 0; j < RX_SGE_USABLE; j++) {
         rc = qfle3_alloc_rx_sge_mbuf(fp, ring_prod);
         if (rc != 0) {
            QFLE3_DBG(QFLE3_DBG_LOAD, "mbuf alloc fail for fp[%02d] SGE %d\n",
               fp->qid, ring_prod);
            fp->tpa_enable = VMK_FALSE;
            ring_prod = 0;
            goto qfle3_alloc_fp_buffers_error;
         }

         ring_prod = RX_SGE_NEXT(ring_prod);
      }

      fp->rx_sge_prod = ring_prod;
   }


   return VMK_OK;

qfle3_alloc_fp_buffers_error:

   /*
    * unwind what was already allocated
    */
    
   QFLE3_ERR("Error allocating fp buffers for fp[%d], unwinding", fp->qid);
   qfle3_free_rx_bd_chain(fp);
   qfle3_free_tpa_pool(fp);
   qfle3_free_sge_chain(fp);
   return VMK_FAILURE;
}

#define TPA_TSTAMP_OPT_LEN   (12)

static vmk_uint16 qfle3_get_tpa_header_len(vmk_PktHandle *pkt,
					   vmk_uint16 parsing_flags,
					   vmk_uint8 hw_vlan)
{
   /* TPA aggregation won't have either IP options or TCP options
    * other than timestamp or IPv6 extension headers.
    */
   vmk_uint16 hdrs_len = sizeof(vmk_EthHdr) + sizeof(vmk_TCPHdr);

   if (GET_FLAG(parsing_flags, PARSING_FLAGS_OVER_ETHERNET_PROTOCOL) ==
       PRS_FLAG_OVERETH_IPV6) {
      hdrs_len += sizeof(vmk_IPv6Hdr);
   } else {
      hdrs_len += sizeof(vmk_IPv4Hdr);
   }

   if ((hw_vlan == 0) && (parsing_flags & PARSING_FLAGS_VLAN))
      hdrs_len += sizeof(vmk_VLANHdr);
   /* Check if there was a TCP timestamp, if there is it's will
    * always be 12 bytes length: nop nop kind length echo val.
    *
    * Otherwise FW would close the aggregation.
    */
   if (parsing_flags & PARSING_FLAGS_TIME_STAMP_EXIST_FLAG)
      hdrs_len += TPA_TSTAMP_OPT_LEN;
   return (hdrs_len);
}


/*
 * The aggregation on the current TPA queue has completed. Pull the individual
 * mbuf fragments together into a single mbuf, perform all necessary checksum
 * calculations, and send the resuting mbuf to the stack.
 */
static int
qfle3_tpa_stop(qfle3_adapter * adapter,
	       struct qfle3_fastpath *fp,
	       struct qfle3_sw_tpa_info *tpa_info,
	       vmk_uint16 queue,
	       vmk_uint16 pages, struct eth_end_agg_rx_cqe *cqe, vmk_uint16 cqe_idx)
{
   vmk_PktHandle *pkt;
   int rc = 0;

   QFLE3_DBG(QFLE3_DBG_LRO,
	     "fp[%02d].tpa[%02d] pad=%d pkt_len=%d pages=%d vlan=%d\n",
	     fp->qid, queue, tpa_info->placement_offset,
	     le16toh(cqe->pkt_len), pages, tpa_info->vlan_tag);

   pkt = tpa_info->bd.pkt;

   /*
    * allocate a replacement before modifying existing mbuf
    */
   rc = qfle3_alloc_rx_tpa_mbuf(fp, queue);
   if (rc) {
      /*
       * drop the frame and log an error
       */
      // fp->eth_q_stats.rx_soft_errors++;

      QFLE3_ERR("fp[%02d].tpa[%02d] Could not alloc tpa mbuf\n", fp->qid, queue);
      fp->drv_stats.rx_Drops++;
      adapter->drv_stats.rx_Drops++;
      goto qfle3_tpa_stop_exit;
   }

   /*
    * we have a replacement, fixup the current mbuf
    */
   //m_adj(m, tpa_info->placement_offset);
   //m->m_pkthdr.len = m->m_len = tpa_info->len_on_bd;
   vmk_PktFrameLenSet(pkt, tpa_info->len_on_bd + tpa_info->placement_offset);
   vmk_PktPushHeadroom(pkt, tpa_info->placement_offset);
   /*
    * mark the checksums valid (taken care of by the firmware)
    */
   //fp->eth_q_stats.rx_ofld_frames_csum_ip++;
   //fp->eth_q_stats.rx_ofld_frames_csum_tcp_udp++;

#if 0                           //I don't know what to do here for vmkernel
   m->m_pkthdr.csum_data = 0xffff;
   m->m_pkthdr.csum_flags |= (CSUM_IP_CHECKED |
			      CSUM_IP_VALID | CSUM_DATA_VALID | CSUM_PSEUDO_HDR);
#endif
   //by default hardware Check Sum Offload for RX.
 //  vmk_PktSetCsumVfd(pkt);
   /*
    * aggregate all of the SGEs into a single mbuf
    */
   rc = qfle3_fill_frag_mbuf(adapter, fp, tpa_info, queue, pages, pkt, cqe, cqe_idx);
   if (rc) {
      /*
       * drop the packet and log an error
       */
      //fp->eth_q_stats.rx_soft_errors++;
      fp->drv_stats.rx_Drops++;
      adapter->drv_stats.rx_Drops++;
      vmk_PktRelease(pkt);
   } else {
      vmk_uint32 mss = 0;
      vmk_uint32 hdrs_len = 0;
      if ((adapter->hw_vlan_en) && (tpa_info->parsing_flags & PARSING_FLAGS_VLAN)) {
	 //m->m_pkthdr.ether_vtag = tpa_info->vlan_tag;
	 //m->m_flags |= M_VLANTAG;
	 vmk_PktVlanIDSet(pkt, tpa_info->vlan_tag & VLAN_VID_MASK);
      }
      vmk_PktSetCsumVfd(pkt);
      VMK_ASSERT(le16toh(cqe->pkt_len) == vmk_PktFrameLenGet(pkt));
      adapter->drv_stats.rx_pkts++;
      adapter->drv_stats.rx_bytes += le16toh(cqe->pkt_len);
      fp->drv_stats.rx_pkts++;
      fp->drv_stats.rx_bytes += le16toh(cqe->pkt_len);
      fp->drv_stats_priv.tpa_rx_bytes += le16toh(cqe->pkt_len);
      fp->drv_stats_priv.tpa_rx_pkts++;

      if (le16toh(cqe->pkt_len) - tpa_info->len_on_bd) {
         hdrs_len = qfle3_get_tpa_header_len(pkt, tpa_info->parsing_flags, adapter->hw_vlan_en);
         mss = tpa_info->len_on_bd - hdrs_len;

         vmk_PktSetLargeTcpPacket(pkt, mss);
      }

      vmk_NetPollRxPktQueue(fp->netpoll, pkt);

      QFLE3_DBG(QFLE3_DBG_RX | QFLE3_DBG_LRO, "fp[%02d].tpa[%02d] Upload LRO packet into stack, len %d, mss %d\n", fp->qid, queue, le16toh(cqe->pkt_len), mss);
   }

  qfle3_tpa_stop_exit:

   fp->rx_tpa_info[queue].state = QFLE3_TPA_STATE_STOP;
   fp->rx_tpa_queue_used &= ~(1 << queue);

   return (rc);
}


/*
 * The current mbuf is part of an aggregation. Move the mbuf into the TPA
 * aggregation queue, put an empty mbuf back onto the receive chain, and mark
 * the current aggregation queue as in-progress.
 */
static void
qfle3_tpa_start(qfle3_adapter * adapter,
		struct qfle3_fastpath *fp,
		vmk_uint16 queue,
		vmk_uint16 cons, vmk_uint16 prod, struct eth_fast_path_rx_cqe *cqe)
{
   qfle3_rxbuf_info tmp_bd;
   struct eth_rx_bd *rx_bd;
   vmk_int32 max_agg_queues;
   struct qfle3_sw_tpa_info *tpa_info = &fp->rx_tpa_info[queue];

   QFLE3_DBG(QFLE3_DBG_LRO, "fp[%02d].tpa[%02d] TPA START "
	     "cons=%d prod=%d\n", fp->qid, queue, cons, prod);

   max_agg_queues = MAX_AGG_QS(adapter);

   VMK_ASSERT(queue < max_agg_queues);
   VMK_ASSERT(tpa_info->state == QFLE3_TPA_STATE_STOP);

   /*
    * copy the existing mbuf and mapping from the TPA pool
    */
   tmp_bd = tpa_info->bd;

   if (tmp_bd.pkt == NULL) {
      QFLE3_DBG(QFLE3_DBG_LRO, "fp[%02d].tpa[%02d] mbuf not allocated!\n",
		fp->qid, queue);
      /*
       * XXX Error handling?
       */
      return;
   }

   /*
    * change the TPA queue to the start state
    */
   tpa_info->state = QFLE3_TPA_STATE_START;
   tpa_info->placement_offset = cqe->placement_offset;
   tpa_info->parsing_flags = le16toh(cqe->pars_flags.flags);
   tpa_info->vlan_tag = le16toh(cqe->vlan_tag);
   tpa_info->len_on_bd = le16toh(cqe->len_on_bd);

   fp->rx_tpa_queue_used |= (1 << queue);

   /*
    * move the received mbuf and mapping to TPA pool
    */
   tpa_info->bd = fp->rxbuf_chain[cons];

   /*
    * update the Rx SW BD with the mbuf info from the TPA pool
    */
   fp->rxbuf_chain[cons] = tmp_bd;

   /*
    * update the Rx BD with the empty mbuf phys address from the TPA pool
    */
   rx_bd = &fp->rx_chain[cons];
   rx_bd->addr_hi = htole32(U64_HI(tmp_bd.dmaAddr));
   rx_bd->addr_lo = htole32(U64_LO(tmp_bd.dmaAddr));
}
#if (ESX_DDK_VERSION >= 2017)
static void
qfle3_set_rxhash(struct qfle3_adapter * adapter, struct qfle3_fastpath *fp,
                 vmk_PktHandle *pkt,
                 struct eth_fast_path_rx_cqe *cqe)
#else
static void
qfle3_set_rxhash(struct qfle3_fastpath *fp,
		 vmk_PktHandle *pkt,
		 struct eth_fast_path_rx_cqe *cqe)
#endif
{
   vmk_uint32 hvalue = 0;
   VMK_ReturnStatus status;
   vmk_PktRssType htype = VMK_PKT_RSS_TYPE_NONE;
#if (ESX_DDK_VERSION >= 2017)
   vmk_UplinkSharedQueueData *sqd;
   vmk_UplinkQueueID          vmkqid;
   sqd = QFLE3_RX_QUEUE_SHARED_DATA(adapter);
#endif

   if (fp->is_defq_rss)
      return;

   /* Get Toeplitz hash from CQE */
   if ((cqe->status_flags & ETH_FAST_PATH_RX_CQE_RSS_HASH_FLG)) {
      enum eth_rss_hash_type rxhash_type;

      rxhash_type = cqe->status_flags & ETH_FAST_PATH_RX_CQE_RSS_HASH_TYPE;

      switch (rxhash_type) {
      case IPV4_HASH_TYPE:
	 htype = VMK_PKT_RSS_TYPE_IPV4;
	 break;
      case TCP_IPV4_HASH_TYPE:
	 htype = VMK_PKT_RSS_TYPE_IPV4_TCP;
	 break;
      case IPV6_HASH_TYPE:
	 htype = VMK_PKT_RSS_TYPE_IPV6;
	 break;
      case TCP_IPV6_HASH_TYPE:
	 htype = VMK_PKT_RSS_TYPE_IPV6_TCP;
	 break;
      default:
	 hvalue = 0;
	 htype = VMK_PKT_RSS_TYPE_NONE;
	 return;
      }

      fp->drv_stats_priv.rssq_rx_bytes += vmk_PktFrameLenGet(pkt);

      hvalue = le32toh(cqe->rss_hash_result);

      status = vmk_PktRssHashSet(pkt, hvalue, htype);
      if (VMK_UNLIKELY(status != VMK_OK)) {
	 QFLE3_WARN("RssHashSet failed with status: %s\n",
		    vmk_StatusToString(status));
      }
#if (ESX_DDK_VERSION >= 2017)
      vmkqid = sqd[fp->qid].qid;
      status = vmk_PktQueueIDSet(pkt, vmkqid);
      if (VMK_UNLIKELY(status != VMK_OK)) {
         QFLE3_ERR("Pkt Queue ID set failed with status: %s\n",
                    vmk_StatusToString(status));
      }
#endif
   }
   return;
}


void qfle3_verify_lb_packet(struct qfle3_adapter *adapter, vmk_PktHandle *pkt)
{
        vmk_PktHandle *lb_pkt = adapter->lb_pkt;
        char *loopback_pkt , *orig;

        loopback_pkt = (char *) vmk_PktFrameMappedPointerGet(pkt);
        orig = (char *) vmk_PktFrameMappedPointerGet(lb_pkt);
        if (vmk_Memcmp(orig, loopback_pkt, 128)) {
                QFLE3_ERR("LB packet mismatch!!\n");
        } else
           QFLE3_INFO("Verified LB packet !!\n");

        adapter->lb_rcvd++;
        vmk_PktRelease(pkt);
        adapter->lb_pkt = NULL;
}
void validate_fcoe_pkt(vmk_PktHandle *pkt, struct eth_fast_path_rx_cqe *cqe_fp)
{
   vmk_uint32 len;
   vmk_uint16 *etherType;
   vmk_uint8 *eof;
   unsigned char buffer[64];
   vmk_uint32 * wbp;
   vmk_uint16 ethType;


   len = le16toh(cqe_fp->pkt_len_or_gro_seg_len);

   vmk_PktCopyBytesOut((void *)buffer,64,0,pkt);
   etherType = (vmk_uint16 *)&buffer[12];
   if (*etherType == VMK_ETH_TYPE_VLAN_NBO){
          etherType += 4;
   }
   ethType = *etherType;
//   vmk_LogMessage("Recv etherTYPe 0x%04x\n", ethType);
   if (ethType == VMK_ETH_TYPE_FCOE_NBO){
      wbp= (vmk_uint32 *)buffer;
      vmk_LogMessage("XX %08x %08x %08x %08x %08x %08x %08x %08x",
       bswap32(wbp[0]),bswap32(wbp[1]),bswap32(wbp[2]),bswap32(wbp[3]),bswap32(wbp[4]),bswap32(wbp[5]),bswap32(wbp[6]),bswap32(wbp[7]));
      vmk_LogMessage("XX %08x %08x %08x %08x %08x %08x %08x %08x\n",
      bswap32(wbp[8]),bswap32(wbp[9]),bswap32(wbp[10]), bswap32(wbp[11]),bswap32(wbp[12]),bswap32(wbp[13]),bswap32(wbp[14]),bswap32(wbp[15]));

      vmk_PktCopyBytesOut((void *)buffer,16,len - 16,pkt);
      eof = (vmk_uint8 *)&buffer[12];
      if ((*eof != 0x41) && (*eof != 0x42)) {
         // panic

         vmk_LogMessage("PANIC!!!!!  FCoE frame length error!  length %d, eof%02x, print packet from %d\n", len, *eof, len - 16);

         VMK_ASSERT(0,"PANIC" );
      }

//      vmk_LogMessage("FC_EOF %02x\n", *eof);
   }
}

#define QFLE3_SEED_CQE(cqe_fp) (cqe_fp->marker = 0xFFFFFFFF)
vmk_uint32
qfle3_rxeof(struct qfle3_adapter * adapter,
	    struct qfle3_fastpath * fp, vmk_uint32 rx_budget, vmk_Bool *sp_update,
	    vmk_Bool isPanic, vmk_PktList pktList)

{
   vmk_uint16 bd_cons, bd_prod, bd_prod_fw, comp_ring_cons;
   vmk_uint16 hw_cq_cons, sw_cq_cons, sw_cq_prod;
   int rx_pkts = 0;
   int rc;
   vmk_PktHandle *pkt = NULL;
   vmk_Bool csum_failure = VMK_FALSE;

   vmk_SpinlockLock(fp->fp_lock);
   /*
    * CQ "next element" is of the size of the regular element
    */
   hw_cq_cons = le16toh(*fp->rx_cq_cons_sb);
   if ((hw_cq_cons & RCQ_USABLE_PER_PAGE) == RCQ_USABLE_PER_PAGE) {
      hw_cq_cons++;
   }

   bd_cons = fp->rx_bd_cons;
   bd_prod = fp->rx_bd_prod;
   bd_prod_fw = bd_prod;
   sw_cq_cons = fp->rx_cq_cons;
   sw_cq_prod = fp->rx_cq_prod;
   if (IS_FWD_FP(fp))
      QFLE3_DBG(QFLE3_DBG_POLL, "NET POLL for cnicqs FWD%d\n", fp->qid);

   QFLE3_DBG(QFLE3_DBG_RX,
	     "fp[%02d] Rx START hw_cq_cons=%u sw_cq_cons=%u\n",
	     fp->qid, hw_cq_cons, sw_cq_cons);

   if (sw_cq_cons == hw_cq_cons){
      if (hw_cq_cons == 0)
         QFLE3_ERR("Netpoll Woke Up, but nothing to process HW CQ CONS 0x%x", hw_cq_cons);
      
      vmk_SpinlockUnlock(fp->fp_lock);
      return 0;
   }
   
   while (sw_cq_cons != hw_cq_cons) {
      qfle3_rxbuf_info *rx_buf = NULL;
      union eth_rx_cqe *cqe;
      struct eth_fast_path_rx_cqe *cqe_fp;
      vmk_uint8 cqe_fp_flags;
      enum eth_rx_cqe_type cqe_fp_type;
      vmk_uint16 len = 0, pad = 0;
      comp_ring_cons = RCQ(sw_cq_cons);
      bd_prod = RX_BD(bd_prod);
      bd_cons = RX_BD(bd_cons);

      cqe = &fp->rcq_chain[comp_ring_cons];
      cqe_fp = &cqe->fast_path_cqe;
      cqe_fp_flags = cqe_fp->type_error_flags;
      cqe_fp_type = cqe_fp_flags & ETH_FAST_PATH_RX_CQE_TYPE;

      QFLE3_DBG(QFLE3_DBG_RX,
		"fp[%02d] vlan=%d Rx hw_cq_cons=%d hw_sw_cons=%d "
		"BD prod=%d cons=%d CQE type=0x%x err=0x%x "
		"status=0x%x rss_hash=0x%x len=%u\n",
		fp->qid, le16toh(cqe_fp->vlan_tag),
		hw_cq_cons,
		sw_cq_cons,
		bd_prod,
		bd_cons,
		CQE_TYPE(cqe_fp_flags),
		cqe_fp_flags,
		cqe_fp->status_flags,
		le32toh(cqe_fp->rss_hash_result),
		le16toh(cqe_fp->pkt_len_or_gro_seg_len));

      /*
       * is this a slowpath msg?
       */
      if (VMK_UNLIKELY(CQE_TYPE_SLOW(cqe_fp_type))) {
         QFLE3_DBG(QFLE3_DBG_RX, "Slow path cqe %p type %d", cqe, cqe_fp_type);
	 qfle3_sp_event(fp, cqe);
	 if (sp_update)
	    *sp_update = VMK_TRUE;
	 goto next_cqe;
      }

      if ((fp->qid >= QFLE3_NUM_RX_ETH_QUEUES(adapter)) && !IS_FCOE_IDX(fp->qid)) {
         vmk_Panic("??receive pkt in tx queue %d!!", fp->qid);
	 goto next_cqe;
      }
      rx_buf = &fp->rxbuf_chain[bd_cons];

      if (!CQE_TYPE_FAST(cqe_fp_type)) {
	 struct qfle3_sw_tpa_info *tpa_info;
	 vmk_uint16 frag_size, pages;
	 vmk_uint8 queue;

#if 0
	 /*
	  * sanity check
	  */
	 if (!fp->tpa_enable &&
	     (CQE_TYPE_START(cqe_fp_type) || CQE_TYPE_STOP(cqe_fp_type))) {
	    BLOGE(sc, "START/STOP packet while !tpa_enable type (0x%x)\n",
		  CQE_TYPE(cqe_fp_type));
	 }
#endif

	 if (CQE_TYPE_START(cqe_fp_type)) {
	    qfle3_tpa_start(adapter, fp, cqe_fp->queue_index, bd_cons, bd_prod, cqe_fp);
	    pkt = NULL;           /* packet not ready yet */
	    goto next_rx;
	 }

	 VMK_ASSERT(CQE_TYPE_STOP(cqe_fp_type));

	 queue = cqe->end_agg_cqe.queue_index;
	 tpa_info = &fp->rx_tpa_info[queue];

	 QFLE3_DBG(QFLE3_DBG_LRO, "fp[%02d].tpa[%02d] TPA STOP\n", fp->qid, queue);

	 frag_size = (le16toh(cqe->end_agg_cqe.pkt_len) - tpa_info->len_on_bd);
	 pages = SGE_PAGE_ALIGN(frag_size) >> SGE_PAGE_SHIFT;

         if (qfle3_tpa_stop(adapter, fp, tpa_info, queue, pages,
               &cqe->end_agg_cqe, comp_ring_cons) == 0) {
            rx_pkts++;
         }

	 qfle3_update_sge_prod(adapter, fp, pages, &cqe->end_agg_cqe);
	 goto next_cqe;
      }

      /*
       * non TPA
       */

      /*
       * is this an error packet?
       */
      if (VMK_UNLIKELY(cqe_fp_flags & ETH_FAST_PATH_RX_CQE_PHY_DECODE_ERR_FLG)) {
	 QFLE3_DBG(QFLE3_DBG_RX, "flags 0x%x rx packet %u\n", cqe_fp_flags, sw_cq_cons);
	 //fp->eth_q_stats.rx_soft_errors++;
	 fp->drv_stats.rx_Errors++;
	 adapter->drv_stats.rx_Errors++;
	 goto next_rx;
      }

      len = le16toh(cqe_fp->pkt_len_or_gro_seg_len);
      pad = cqe_fp->placement_offset;
      QFLE3_DBG(QFLE3_DBG_RX,"rcq_chain[%d]  rx chain[%d], packet length %d\n",(int)comp_ring_cons, (int)bd_cons, len);
      pkt = rx_buf->pkt;

//      validate_fcoe_pkt(pkt,cqe_fp);
      if (VMK_UNLIKELY(pkt == NULL)) {
	 QFLE3_DBG(QFLE3_DBG_RX, "No mbuf in rx chain descriptor %d for fp[%02d]\n",
		   bd_cons, fp->qid);
	 adapter->drv_stats.rx_Errors++;
	 goto next_rx;
      }

      /*
       * XXX double copy if packet length under a threshold
       */

      /*
       * If all the buffer descriptors are filled with mbufs then fill in
       * the current consumer index with a new BD. Else if a maximum Rx
       * buffer limit is imposed then fill in the next producer index.
       */
      rc = qfle3_alloc_rx_bd_mbuf(fp, bd_cons);
      if (rc != 0) {
         QFLE3_ERR( "mbuf alloc fail for fp[%02d] rx chain[%d], dropping packet\n",
		   fp->qid, (int)bd_cons);
	 //fp->eth_q_stats.rx_soft_errors++;
	 pkt = NULL;
	 fp->drv_stats.rx_Drops++;
	 adapter->drv_stats.rx_Drops++;
	 goto next_rx;
      }

      vmk_PktFrameLenSet(pkt, len + pad);
      vmk_PktPushHeadroom(pkt, pad);


//      QFLE3_DBG(QFLE3_DBG_RX, "pkt len = %d and pad = %d\n", len, pad);

      if (VMK_UNLIKELY(len < 60)) {
	 QFLE3_DBG(QFLE3_DBG_RX, "RX pkt len %d < 60", len);
	 vmk_PktFrameLenSet(pkt, 60);
      }

      /*
       * Hardware always do checksum for the incoming frames.
       */
      /*
       * validate checksum if offload enabled
       */
      /*
       * check for a valid IP frame
       */
      if (!(cqe->fast_path_cqe.status_flags &
	    ETH_FAST_PATH_RX_CQE_IP_XSUM_NO_VALIDATION_FLG)) {
	 //m->m_pkthdr.csum_flags |= CSUM_IP_CHECKED;
	 if (VMK_UNLIKELY(cqe_fp_flags & ETH_FAST_PATH_RX_CQE_IP_BAD_XSUM_FLG)) {
	    QFLE3_DBG(QFLE3_DBG_RX, "RX IP CSUM failed");     //fp->eth_q_stats.rx_hw_csum_errors++;
	    csum_failure = VMK_TRUE;
	    fp->drv_stats.rx_Errors++;
	    adapter->drv_stats.rx_Errors++;
	    goto next_rx;
	 }
      }
      /*
       * check for a valid TCP/UDP frame
       */
      if (!(cqe->fast_path_cqe.status_flags &
	    ETH_FAST_PATH_RX_CQE_L4_XSUM_NO_VALIDATION_FLG)) {
	 if (VMK_UNLIKELY(cqe_fp_flags & ETH_FAST_PATH_RX_CQE_L4_BAD_XSUM_FLG)) {
	    //fp->eth_q_stats.rx_hw_csum_errors++;
	    QFLE3_DBG(QFLE3_DBG_RX, "RX L4 CSUM failed");
	    csum_failure = VMK_TRUE;
	    adapter->drv_stats.rx_Errors++;
	    fp->drv_stats.rx_Errors++;
	    goto next_rx;
	 } else {
	    //fp->eth_q_stats.rx_ofld_frames_csum_tcp_udp++;
	    //m->m_pkthdr.csum_data = 0xFFFF;
	    //m->m_pkthdr.csum_flags |= (CSUM_DATA_VALID |
	    //                           CSUM_PSEUDO_HDR);
	    QFLE3_DBG(QFLE3_DBG_RX, "RX L4 CSUM Verified");
	    vmk_PktSetCsumVfd(pkt);
	 }
      }

      /*
       * if there is a VLAN tag then flag that info
       */
      //if (vmk_UplinkCapIsEnabled(adapter->uplink,
		//		    VMK_UPLINK_CAP_VLAN_RX_STRIP)) {
		if ((!IS_FCOE_FP(fp)) && adapter->hw_vlan_en) {
   	 if (cqe->fast_path_cqe.pars_flags.flags & PARSING_FLAGS_VLAN) {
   	    //m->m_pkthdr.ether_vtag = cqe->fast_path_cqe.vlan_tag;
   	    //m->m_flags |= M_VLANTAG;
   	    //VMK_ASSERT((cqe->fast_path_cqe.pars_flags.flags & PARSING_FLAGS_VLAN) == 0);
   	    QFLE3_DBG(QFLE3_DBG_RX, "RX VLAN stripped, bd_cons %d, bd_prod %d, bd_prod_fw %d",
   		      (int)bd_cons, (int)bd_prod, (int)bd_prod_fw);
   	    vmk_PktVlanIDSet(pkt, cqe->fast_path_cqe.vlan_tag & VLAN_VID_MASK);
   	 }
      }



     next_rx:

      bd_cons = RX_BD_NEXT(bd_cons);
      bd_prod = RX_BD_NEXT(bd_prod);
      bd_prod_fw = RX_BD_NEXT(bd_prod_fw);

      if (csum_failure == VMK_TRUE) {
	 vmk_PktRelease(pkt);
	 pkt = NULL;
	 adapter->drv_stats.rx_Drops++;
	 csum_failure = VMK_FALSE;
      }
      {
//         count_protocol_pkt(adapter, pkt, VMK_FALSE);
      }
      /*
       * pass the frame to the stack
       */
      if (VMK_LIKELY(pkt != NULL)) {
	 //ifp->if_ipackets++;

         if (IS_FCOE_FP(fp)) {
#ifdef QFLE3_CNIC
            if (CNIC_LOADED(adapter)){
	 rx_pkts++;
	 adapter->drv_stats.rx_pkts++;
	 adapter->drv_stats.rx_bytes += vmk_PktFrameLenGet(pkt);
	 fp->drv_stats.rx_pkts++;
	 fp->drv_stats.rx_bytes += vmk_PktFrameLenGet(pkt);
      adapter->cnicOps->cnicLL2Rx(adapter->cnic_data, pkt);
      QFLE3_DBG(QFLE3_DBG_RX, "fp[%02d]. Upload packet into CNIC, len %d, pad %d\n", fp->qid, len, pad);
            } else {
               vmk_PktRelease(pkt);
               QFLE3_INFO( "fp[%02d]. Dropping packet due to CNIC not loaded\n", fp->qid);
            }
#endif
    }else {
   	 if(VMK_UNLIKELY(isPanic)) {
   	    vmk_PktListAppendPkt(pktList, pkt);
   	 }else{
#if (ESX_DDK_VERSION >= 2017)
            qfle3_set_rxhash(adapter, fp, pkt, cqe_fp);
#else
   	    qfle3_set_rxhash(fp, pkt, cqe_fp);
#endif
            if (fp->is_defq_rss) {
               fp->drv_stats_priv.rssq_rx_bytes += vmk_PktFrameLenGet(pkt);
               }
               if (VMK_LIKELY(!adapter->lb_pkt)) {
                  if (!fp->pollDisabled) {
                     rx_pkts++;
                     adapter->drv_stats.rx_pkts++;
                     adapter->drv_stats.rx_bytes += vmk_PktFrameLenGet(pkt);
                     fp->drv_stats.rx_pkts++;
                     fp->drv_stats.rx_bytes += vmk_PktFrameLenGet(pkt);
                     if (fp->netpoll) {
                        vmk_NetPollRxPktQueue(fp->netpoll, pkt);
                        QFLE3_DBG(QFLE3_DBG_RX, "fp[%02d]. Upload packet into stack, len %d, pad %d\n", fp->qid, len, pad);
                     }
                  } else {
                     vmk_PktRelease(pkt);
                  }
               } else {
		rx_pkts++;
         	adapter->drv_stats.rx_pkts++;
         	adapter->drv_stats.rx_bytes += vmk_PktFrameLenGet(pkt);
         	fp->drv_stats.rx_pkts++;
	      	fp->drv_stats.rx_bytes += vmk_PktFrameLenGet(pkt);
	        qfle3_verify_lb_packet(adapter, pkt);
   	 }
            }
    }
      }

     next_cqe:

      sw_cq_prod = RCQ_NEXT(sw_cq_prod);
      sw_cq_cons = RCQ_NEXT(sw_cq_cons);
      
		/* mark CQE as free */
		QFLE3_SEED_CQE(cqe_fp);

      /*
       * limit spinning on the queue
       */
      if (rx_pkts == rx_budget) {
	 //fp->eth_q_stats.rx_budget_reached++;
	 break;
      }
   }                            /* while work to do */

   fp->rx_bd_cons = bd_cons;
   fp->rx_bd_prod = bd_prod_fw;
   fp->rx_cq_cons = sw_cq_cons;
   fp->rx_cq_prod = sw_cq_prod;

   /*
    * Update producers
    */
   qfle3_update_rx_prod(adapter, fp, bd_prod_fw, sw_cq_prod, fp->rx_sge_prod);

   //fp->eth_q_stats.rx_pkts += rx_pkts;
   //fp->eth_q_stats.rx_calls++;

   vmk_SpinlockUnlock(fp->fp_lock);

   return (rx_pkts);
}

static vmk_uint16
qfle3_release_tx_pkt(struct qfle3_adapter *adapter,
               struct qfle3_fp_txdata *txdata, vmk_uint16 idx, vmk_Bool isPanic)
{
   qfle3_txbuf_info *tx_buf = &txdata->txbuf_chain[idx];
   struct eth_tx_start_bd *tx_start_bd;
   vmk_uint16 bd_idx = TX_BD(tx_buf->first_bd);
   vmk_uint16 new_cons;
   int nbd, i;

   /*
    * unmap the mbuf from non-paged memory
    */
   for (i = 0; i < tx_buf->nsegs; i++) {
      qfle3_dma_unmap(adapter,
		      tx_buf->tx_segs[i].ioa,
		      tx_buf->tx_segs[i].len,
		      VMK_DMA_DIRECTION_FROM_MEMORY, adapter->dmaEngine);
      tx_buf->tx_segs[i].ioa = 0;
      tx_buf->tx_segs[i].len = 0;
      QFLE3_DBG(QFLE3_DBG_TX, "fp[%02d][%d]. txbuf [%02d]ummapped. sge[%02d]\n", txdata->parent_fp->qid, txdata->cos, idx, i);
   }
   tx_start_bd = &txdata->tx_chain[bd_idx].start_bd;
   nbd = le16toh(tx_start_bd->nbd) - 1;

   new_cons = (tx_buf->first_bd + nbd);

   /*
    * free the mbuf
    */
   if (tx_buf->pkt != NULL) {
   if (VMK_LIKELY(tx_buf->pkt != NULL)) {
      adapter->drv_stats.tx_done_pkts++;
         txdata->parent_fp->drv_stats.tx_done_pkts++;
      if (VMK_UNLIKELY(isPanic)) {
	 VMK_ASSERT(vmk_SystemCheckState(VMK_SYSTEM_STATE_PANIC));
	 vmk_PktReleasePanic(tx_buf->pkt);
      } else {
        if (VMK_LIKELY(!adapter->lb_pkt))
               vmk_NetPollQueueCompPkt(txdata->parent_fp->netpoll, tx_buf->pkt);
        else
            vmk_PktRelease(tx_buf->pkt);
	 //fp->eth_q_stats.mbuf_alloc_tx--;
            adapter->drv_stats_priv.tx_pkt_completed++;
      }
   } else {
      ;                         //fp->eth_q_stats.tx_chain_lost_mbuf++;
   }
   } else {
      QFLE3_ERR("Tx Pkt is null fp[%d][%d] tx_chain[%d]\n", txdata->parent_fp->qid, txdata->cos, idx);
   }
   tx_buf->pkt = NULL;
   tx_buf->first_bd = 0;
   tx_buf->nsegs = 0;
   tx_buf->flags = 0;
   tx_buf->mapType = QFLE3_MAP_INVALID;

   return (new_cons);
}


static inline void
qfle3_update_fp_sb_idx(struct qfle3_fastpath *fp)
{
   /*
    * status block is written to by the chip
    */
   fp->fp_hc_idx = fp->sb_running_index[SM_RX_ID];
}

static inline vmk_int32
   qfle3_tx_queue_has_work_cos(struct qfle3_fp_txdata *txdata)
{
   vmk_uint16 hw_cons;

   /*vmk_LogMessage("tx_cons_sb is %d tx_pkt_cons is %d", 
      le16toh(*fp->tx_cons_sb), fp->tx_pkt_cons);
   */
   /*
    * status block fields can change
    */
      hw_cons = le16toh(*txdata->tx_cons_sb);
      return (hw_cons != txdata->tx_pkt_cons);
}


static inline vmk_uint8
qfle3_has_tx_work_cos(struct qfle3_fp_txdata *txdata)
{
   /*
    * expand this for multi-cos if ever supported
    */
   return (qfle3_tx_queue_has_work(txdata)) ? VMK_TRUE : VMK_FALSE;
}

static inline int
qfle3_has_rx_work(struct qfle3_fastpath *fp)
{
   vmk_uint16 rx_cq_cons_sb;
   struct qfle3_adapter *adapter = fp->adapter;

   VMK_ASSERT(fp->rx_cq_cons_sb, " no rx_cq_cons_sb");
   /*
    * status block fields can change
    */
   rx_cq_cons_sb = le16toh(*fp->rx_cq_cons_sb);
   if ((rx_cq_cons_sb & RCQ_MAX) == RCQ_MAX)
      rx_cq_cons_sb++;
   
   //vmk_LogMessage("rx_cq_cons_sb is %d driver is %d", rx_cq_cons_sb, fp->rx_cq_cons);
   return (fp->rx_cq_cons != rx_cq_cons_sb);
}

void
qfle3_unmap_txbuf(qfle3_adapter * adapter, qfle3_txbuf_info *tbi, vmk_uint32 index, vmk_uint32 qid)
{
   int i;

   if (tbi->mapType == QFLE3_MAP_ELEM) {
      QFLE3_DBG(QFLE3_DBG_TX,"Caught a leak fp[%d]txbuf, nsegs %d", qid, tbi->nsegs);
      /*
       * unmap the mbuf from non-paged memory
       */
      for (i = 0; i < tbi->nsegs; i++) {
	 if (tbi->tx_segs[i].ioa && tbi->tx_segs[i].len) {
	    qfle3_dma_unmap(adapter,
			    tbi->tx_segs[i].ioa,
			    tbi->tx_segs[i].len,
			    VMK_DMA_DIRECTION_FROM_MEMORY, adapter->dmaEngine);

	    QFLE3_DBG(QFLE3_DBG_TX,"Caught a leak fp[%d] txbuf [%d]", qid, index);
	 }
	 tbi->tx_segs[i].ioa = 0;
	 tbi->tx_segs[i].len = 0;
      }
   }
   tbi->nsegs = 0;
   tbi->mapType = QFLE3_MAP_INVALID;       /* to help debugging */
}

/* processes transmit completions */
static vmk_uint32
qfle3_txeof(struct qfle3_adapter *adapter, struct qfle3_fp_txdata *txdata,
	    vmk_uint32 budget, vmk_Bool isPanic)
{
   vmk_uint16 bd_cons, hw_cons, sw_cons, pkt_cons;
   vmk_uint16 tx_bd_avail;
   vmk_uint32 completed = 0;
   struct qfle3_fastpath *fp = txdata->parent_fp;

   if (budget == 0)
      return completed;
   
   bd_cons = txdata->tx_bd_cons;
   hw_cons = le16toh(*txdata->tx_cons_sb);
   sw_cons = txdata->tx_pkt_cons;

   while (sw_cons != hw_cons) {
      pkt_cons = TX_BD(sw_cons);

      QFLE3_DBG(QFLE3_DBG_TX,
		"Tx Comp: fp[%d]: hw_cons=%u sw_cons=%u pkt_cons=%u\n",
		fp->qid, hw_cons, sw_cons, pkt_cons);

      bd_cons = qfle3_release_tx_pkt(adapter, txdata, pkt_cons, isPanic);
      completed++;
      sw_cons++;
      //if ((txdata->cos == FIRST_TX_COS_INDEX) && (completed == budget))
      if (completed >= budget)
         break;
   }

   txdata->tx_pkt_cons = sw_cons;
   txdata->tx_bd_cons = bd_cons;

   QFLE3_DBG(QFLE3_DBG_TX,
	     "Tx Done: fp[%d][%d]: hw_cons=%u sw_cons=%u sw_prod=%u completed %d\n",
	     fp->qid, txdata->cos, hw_cons, txdata->tx_pkt_cons, txdata->tx_pkt_prod, completed);

   tx_bd_avail = qfle3_tx_avail(adapter, txdata);

   //if (!IS_FCOE_FP(fp) && (txdata->cos == FIRST_TX_COS_INDEX)){
   if (!IS_FCOE_FP(fp)){
      if(fp->soft_state.tq_stopped_by_tx && adapter->linkUp){
         if ((tx_bd_avail > BXE_TX_CLEANUP_THRESHOLD) && completed) {
            int tx_qid = fp->qid - QFLE3_DEFAULT_TX_QID(adapter);
            if (qfle3_tq_stopped(adapter, tx_qid)){
   	       vmk_SpinlockLock(fp->fp_lock);
               qfle3_tq_resume(adapter, tx_qid);
               vmk_SpinlockUnlock(fp->fp_lock);
	    }
         }
      }
   }

   if (txdata->tx_pkt_prod != txdata->tx_pkt_cons) {
      /*
       * reset the watchdog timer if there are pending transmits
       */
      // fp->watchdog_timer = BXE_TX_TIMEOUT;
      QFLE3_DBG(QFLE3_DBG_TX,
      "Tx Tmot: fp[%d][%d]: hw_cons=%u sw_cons=%u sw_prod=%u completed %d\n",
      fp->qid, txdata->cos, hw_cons, txdata->tx_pkt_cons, txdata->tx_pkt_prod, completed);
      return (completed);
   } else {
      /*
       * clear watchdog when there are no pending transmits
       */
      // fp->watchdog_timer = 0;
      return (completed);
   }
}

vmk_Bool
qfle3_netpoll_rxcb(vmk_AddrCookie priv, vmk_uint32 budget)
{
   struct qfle3_fastpath *fp;
   struct qfle3_adapter *adapter;
   vmk_uint32 rxDone = 0;
   vmk_uint32 ration = 24;
   vmk_uint32 rxSingle = 0;

   fp = (struct qfle3_fastpath *) priv.ptr;
   adapter = fp->adapter;

   if (!fp->fp_state)
      return VMK_FALSE;

   /*
    * update the fastpath index
    */
   QFLE3_DBG(QFLE3_DBG_POLL, "rxq fp %d poll, budget %d\n", fp->qid, budget);

      vmk_CPUMemFenceReadWrite();
      qfle3_update_fp_sb_idx(fp);
      vmk_CPUMemFenceReadWrite();
   
   /* Tx interrupts should never come on RxQ. If it does then it is an error.
    *
    */
   if (VMK_UNLIKELY(qfle3_has_tx_work(fp))) {
      vmk_Panic("rx queue got tx intr");
   }
   if (!adapter->sw_fcoe) {
      if (qfle3_has_rx_work(fp)) {
      rxDone = qfle3_rxeof(adapter, fp, budget, NULL, VMK_FALSE, NULL);
      QFLE3_DBG(QFLE3_DBG_POLL, "rxq fp %d poll, rxDone %d\n", fp->qid, rxDone);
      }
   } else {
      while (qfle3_has_rx_work(fp) && (rxDone < budget)) {
         rxSingle = qfle3_rxeof(adapter, fp, (budget>ration)? ration: budget, NULL, VMK_FALSE, NULL);
         rxDone += rxSingle;
         // to prevent dead loops
         if (!rxSingle)
            break;
      } ;

      QFLE3_DBG(QFLE3_DBG_POLL, "rxq fp %d poll, rxDone %d\n", fp->qid, rxDone);
   }

   /*
    * Here we write the fastpath index taken before doing any tx or rx work.
    * It is very well possible other hw events occurred up to this point and
    * they were actually processed accordingly above. Since we're going to
    * write an older fastpath index, an interrupt is coming which we might
    * not do any work in.
    */
    
   if (rxDone < budget) {
      //qfle3_enable_all_intrs(adapter);
      qfle3_ack_sb(adapter, fp->igu_sb_id, USTORM_ID,
		   le16toh(fp->fp_hc_idx), IGU_INT_ENABLE, 1);
      return VMK_FALSE;
   }
   return VMK_TRUE;

}


vmk_Bool
qfle3_netpoll_txrxcb(vmk_AddrCookie priv, vmk_uint32 budget)
{
   struct qfle3_fastpath *fp;
   struct qfle3_adapter *adapter;
   vmk_uint32 txDone = 0;
   vmk_Bool sp_update = VMK_FALSE;
   vmk_uint8 cos;
   vmk_uint32 ration = 24;
   vmk_uint32 completed=0;
   vmk_uint32 budget_left=0;
   vmk_uint32 completed_single=0;

   fp = (struct qfle3_fastpath *) priv.ptr;
   adapter = fp->adapter;

   /* only allow netpoll on fp that are setup in hardware and software */
#ifdef QFLE3_CNIC
   if (IS_FWD_FP(fp))
      QFLE3_DBG(QFLE3_DBG_POLL, "NET POLL for cnicqs FWD%d\n", fp->qid);

   if(fp->pollDisabled){
      vmk_LogMessage("netpoll already disabled, returning\n");
      return VMK_OK;
   }


   /*
    * update the fastpath index
    */
   QFLE3_DBG(QFLE3_DBG_POLL, "txq fp %d poll, budget %d\n", fp->qid, budget);
   if (!IS_FCOE_FP(fp)) {
#endif
      vmk_CPUMemFenceReadWrite();
      qfle3_update_fp_sb_idx(fp);
//   QFLE3_DBG(QFLE3_DBG_POLL, "tx_cons_sb %p  tx_pkt_cons %d \n", fp->tx_cons_sb, fp->tx_pkt_cons);
      vmk_CPUMemFenceReadWrite();
   }
   if (fp->max_cos <= 1) {
   if (qfle3_has_tx_work(fp)) {
         txDone += qfle3_txeof(adapter, fp->txdata_ptr[0], budget, VMK_FALSE);
         QFLE3_DBG(QFLE3_DBG_POLL, "txq fp %d:%d poll, budget %d, txDone %d\n", fp->qid, 0, budget, txDone);
      }
   } else {// split the budget over 3 queues
      while((budget-txDone) > 0) {
         budget_left = budget-txDone;
         completed = 0;
         for_each_cos_in_tx_queue(fp, cos){
            if (qfle3_tx_queue_has_work(fp->txdata_ptr[cos])) {
               completed_single = qfle3_txeof(adapter, fp->txdata_ptr[cos], (budget_left>ration) ? ration:(budget_left), VMK_FALSE);
            }
            budget_left-=completed_single;
            completed += completed_single;
            completed_single=0;
         }
         if (!completed)
            break;
         txDone += completed;
      }
   }

      QFLE3_DBG(QFLE3_DBG_POLL, "txq fp %d poll, budget %d, txDone %d\n", fp->qid, budget, txDone);
   /*
    * Rx interrupts should not come on TxQ except for slow path and FCOE traffic 
    * so hint the compiler that his is a less likely path
    */
   if (VMK_UNLIKELY(qfle3_has_rx_work(fp))) {
      (void) qfle3_rxeof(adapter, fp, 1024, &sp_update, VMK_FALSE, NULL);
   }

   /*
    * Here we write the fastpath index taken before doing any tx or rx work.
    * It is very well possible other hw events occurred up to this point and
    * they were actually processed accordingly above. Since we're going to
    * write an older fastpath index, an interrupt is coming which we might
    * not do any work in.
    */
   if ((txDone < budget) || (sp_update == VMK_TRUE)) {
      //qfle3_enable_all_intrs(adapter);
      if (!IS_FCOE_FP(fp)) {
         qfle3_ack_sb(adapter, fp->igu_sb_id, USTORM_ID,
   		   le16toh(fp->fp_hc_idx), IGU_INT_ENABLE, 1);
      }
      return VMK_FALSE;
   }
   return VMK_TRUE;

}

void
qfle3_msix_rx(void *handlerData, vmk_IntrCookie intrCookie)
{
   struct qfle3_fastpath *fp = (struct qfle3_fastpath *) handlerData;
   qfle3_adapter *adapter = fp->adapter;
   VMK_ReturnStatus poll_status;

   QFLE3_DBG(QFLE3_DBG_INTR,
     "MSI_RX got an MSI-X interrupt on IDX:SB [fp %d fw_sd %d igusb %d]\n", 
    fp->qid, fp->fw_sb_id, fp->igu_sb_id);

   qfle3_ack_sb(adapter, fp->igu_sb_id, USTORM_ID, 0, IGU_INT_DISABLE, 0);
   if (!fp->netpoll)
       QFLE3_ERR("Interrupt received on FP[%d] without netpoll", fp->qid);
   poll_status = vmk_NetPollActivate(fp->netpoll);
   if(poll_status != VMK_OK) {
      QFLE3_ERROR("Failed to activate rx poll");
   }

}

static void
qfle3_msix_tx(void *handlerData, vmk_IntrCookie intrCookie)
{
   struct qfle3_fastpath *fp = (struct qfle3_fastpath *) handlerData;
   qfle3_adapter *adapter = fp->adapter;
   VMK_ReturnStatus poll_status;

   QFLE3_DBG(QFLE3_DBG_INTR,
   		   "MSI_TX got an MSI-X interrupt on IDX:SB [fp %d fw_sd %d igusb %d]\n", 
   		   fp->qid, fp->fw_sb_id, fp->igu_sb_id);

   qfle3_ack_sb(adapter, fp->igu_sb_id, USTORM_ID, 0, IGU_INT_DISABLE, 0);

   if (!fp->netpoll)
       QFLE3_ERR("Interrupt received on FP[%d] without netpoll", fp->qid);
   poll_status = vmk_NetPollActivate(fp->netpoll);
   if(poll_status != VMK_OK) {
      QFLE3_ERROR("Failed to activate poll");
   }
}
void
qfle3_sp_interrupt(void *handlerData, vmk_IntrCookie intrCookie)
{
   qfle3_adapter *adapter = (qfle3_adapter *) handlerData;
   qfle3_ack_sb(adapter, adapter->igu_dsb_id, USTORM_ID, 0,
		IGU_INT_DISABLE, 0);

   QFLE3_DBG(QFLE3_DBG_INTR, "Schedule sp helper function\n");
   qfle3_schedule_helper(adapter, adapter->sp_helper, qfle3_sp_helper_func, 0);
}

void
qfle3_single_interrupt(void *handlerData, vmk_IntrCookie intrCookie)
{
   qfle3_adapter *adapter = (qfle3_adapter *) handlerData;
   struct qfle3_fastpath *fp;
   vmk_uint16 mask;
   vmk_uint32 i;
   VMK_ReturnStatus poll_status;
   /*
    * 0 for ustorm, 1 for cstorm
    * the bits returned from ack_int() are 0-15
    * bit 0 = attention status block
    * bit 1 = fast path status block
    * a mask of 0x2 or more = tx/rx event
    * a mask of 1 = slow path event
    */

   QFLE3_DBG(QFLE3_DBG_INTR, "interrupt received with status = 0x%x", adapter->intr_sts);
   /*
    * the interrupt is not for us
    */

   for (i = 0; i < QFLE3_TOTAL_ETH_QUEUES(adapter); i++) {
      fp = &adapter->fp[i];
      mask = (0x2 << (fp->qid + CNIC_SUPPORT(adapter)));
//      mask = (0x2 << (fp->qid));
      vmk_CPUMemFenceReadWrite();
      if (adapter->intr_sts & mask) {
         vmk_CPUMemFenceReadWrite();
         adapter->intr_sts &= ~mask;
         vmk_CPUMemFenceReadWrite();
         
         QFLE3_DBG(QFLE3_DBG_INTR, "single interrupt on qid %d \n", fp->qid);
         /*
         * acknowledge and disable further fastpath interrupts
         */
         //qfle3_ack_sb(adapter, fp->igu_sb_id, USTORM_ID, 0, IGU_INT_DISABLE, 0);
         if(!fp->netpoll || fp->pollDisabled)
          continue;
         QFLE3_DBG(QFLE3_DBG_INTR, "starting netpoll on qid %d \n", fp->qid);
         poll_status = vmk_NetPollActivate(fp->netpoll);

         if(poll_status != VMK_OK) {
          QFLE3_ERROR("Failed to activate poll. intr_status %x", adapter->intr_sts);
         }
      }
   }

   vmk_CPUMemFenceReadWrite();
   /* slow path handling */
   if (VMK_UNLIKELY(adapter->intr_sts & 0x1)) {
      vmk_CPUMemFenceReadWrite();
      adapter->intr_sts &= ~0x1;
      vmk_CPUMemFenceReadWrite();
      /*
       * acknowledge and disable further slowpath interrupts
       */
      //qfle3_ack_sb(adapter, adapter->igu_dsb_id, USTORM_ID, 0, IGU_INT_DISABLE, 0);

      /*
       * schedule slowpath handler
       */
      QFLE3_DBG(QFLE3_DBG_INTR, "Schedule sp helper function\n");
      qfle3_schedule_helper(adapter, adapter->sp_helper, qfle3_sp_helper_func, 0);
   }
}

void
qfle3_netpoll_enable(qfle3_adapter * adapter)
{
   int i, qid;
   struct qfle3_fastpath *fp;
   FOREACH_ETH_QUEUE(i,qid,fp) 
      if (fp->netpoll && fp->pollDisabled && fp->fp_state) {
         vmk_NetPollEnable(fp->netpoll);
         netpoll_count++;
         fp->pollDisabled = VMK_FALSE;
         QFLE3_INFO("enabled netpoll for q_index %d count%d \n", i, netpoll_count);
      }
   }
}

void
qfle3_netpoll_disable(qfle3_adapter * adapter)
{
   int i, qid;
   struct qfle3_fastpath *rxq;
   struct qfle3_fastpath *txq;

   FOREACH_RX_ETH_QUEUE(i,qid,rxq) 
      if (rxq->netpoll && !rxq->pollDisabled) {
         netpoll_count--;
         QFLE3_INFO("disable netpoll for q_index %d count %d\n", qid, netpoll_count);
         vmk_NetPollDisable(rxq->netpoll);
         vmk_NetPollFlushRx(rxq->netpoll);
         rxq->pollDisabled = VMK_TRUE;
      }
   }

   FOREACH_TX_ETH_QUEUE(i,qid,txq) 
      if (txq->netpoll && !txq->pollDisabled) {
         netpoll_count--;
         QFLE3_INFO("disable netpoll for q_index %d count %d\n", qid, netpoll_count);
         vmk_NetPollDisable(txq->netpoll);
         
         vmk_NetPollFlushRx(txq->netpoll);
         txq->pollDisabled = VMK_TRUE;
      }
   }
}

void
qfle3_unset_netpoll_interrupts(qfle3_adapter * adapter)
{
   struct qfle3_fastpath *fp;
   int i, qid;
   qfle3_intr *intr = &adapter->intr;

   if (intr->intrType != QFLE3_IT_MSIX)
      return;

   FOREACH_RX_ETH_QUEUE(i,qid,fp) 
      if (fp->netpoll) {
         vmk_NetPollInterruptUnSet(fp->netpoll);
      }
   }
   
   FOREACH_TX_ETH_QUEUE(i,qid,fp) 
      if (fp->netpoll) {
         vmk_NetPollInterruptUnSet(fp->netpoll);
      }
   }
}


void
qfle3_unregister_interrupt(qfle3_adapter * adapter,     // IN: adapter
			   vmk_uint32 idx)      // IN: which cookie
{
   VMK_ReturnStatus status;
   qfle3_intr *intr = &adapter->intr;

   status = vmk_IntrUnregister(vmk_ModuleCurrentID, intr->cookies[idx], intr->handlerData[idx]);
   if (status != VMK_OK)
      QFLE3_DBG(QFLE3_DBG_INTR, "Unregister interrupt # %d, failed 0x%x\n", idx, status);
   else
      QFLE3_DBG(QFLE3_DBG_INTR, "Unregister interrupt # %d success\n", idx);
}


void
qfle3_unregister_fp_interrupts(qfle3_adapter * adapter)
{
   qfle3_intr *intr = &adapter->intr;
   int i;
   if (intr->intrType != QFLE3_IT_MSIX)
      return;
#ifdef QFLE3_CNIC
//   if (adapter->flags & OWN_CNIC_IRQ)
//      qfle3_unregister_interrupt(adapter, 1);
//   
#endif
   for (i = 1+CNIC_SUPPORT(adapter); i < intr->numIntrs; i++) {
      
      QFLE3_DBG(QFLE3_DBG_INTR, "Unregister interrupt # %d\n", i);
      qfle3_unregister_interrupt(adapter, i);
   }
   
}

VMK_ReturnStatus
qfle3_register_fp_interrupts(qfle3_adapter * adapter)
{
   VMK_ReturnStatus status = VMK_OK;
   qfle3_intr *intr = &adapter->intr;
   struct qfle3_fastpath *rxq;
   struct qfle3_fastpath *txq;
   int fp_num;
   int i, offset;

   /* MSI/INTX have just one interrupt which has
      already been enabled in slow path */
   if (intr->intrType != QFLE3_IT_MSIX) {
      return VMK_OK;
   }
   offset = 1;
   
//   if (CNIC_SUPPORT(adapter)){
//      rxq = qfle3_ooo_fp(adapter);
//      vmk_NameFormat(&rxq->name, "%s-cnic-%d",
//           QFLE3_NAME_TO_STRING(adapter->pdev_name), rxq->qid);
//      status = qfle3_register_interrupt(adapter, offset,
//               rxq->name, rxq,
//               qfle3_interrupt_ack, qfle3_msix_rx);
//   }
   offset += CNIC_SUPPORT(adapter);
   //skip cnic interrupt
   FOREACH_RX_ETH_QUEUE(i,fp_num,rxq) 
      vmk_NameFormat(&rxq->name, "%s-fp-%d",
		     QFLE3_NAME_TO_STRING(adapter->pdev_name), fp_num);
      status = qfle3_register_interrupt(adapter, offset,
					rxq->name, rxq,
					qfle3_interrupt_ack, qfle3_msix_rx);
      if (status != VMK_OK) {
   	 QFLE3_ERR("Failed to initiate intrCookie #%d 0x%x (%x)", offset,
   		   intr->cookies[offset], status);
   	 goto fail_reg;
      }
      
//      QFLE3_INFO("initiate intrCookie #%d 0x%x (%x) for RX queue", offset,
//           intr->cookies[offset], status);
      offset++;
   }

   FOREACH_TX_ETH_QUEUE(i,fp_num,txq) 
      vmk_NameFormat(&txq->name, "%s-fp-%d",
		     QFLE3_NAME_TO_STRING(adapter->pdev_name), fp_num);
      status = qfle3_register_interrupt(adapter, offset,
					txq->name, txq,
					qfle3_interrupt_ack, qfle3_msix_tx);
      if (status != VMK_OK) {
   	 QFLE3_ERR("Failed to initiate intrCookie #%d 0x%x (%x)", offset,
   		   intr->cookies[offset], status);
   	 goto fail_reg;
      }
      
//      QFLE3_INFO("initiate intrCookie #%d 0x%x (%x) for TX queue", offset,
//           intr->cookies[offset], status);
      offset++;
   }
   VMK_ASSERT(intr->numIntrs == offset);
   
   return VMK_OK;

fail_reg:
   qfle3_unregister_fp_interrupts(adapter);
   return status;
}
#if 0
static VMK_ReturnStatus
qfle3_register_msix_interrupts(qfle3_adapter * adapter)
{
   VMK_ReturnStatus status;
   struct qfle3_fastpath *rxq;
   qfle3_intr *intr = &adapter->intr;
   int i, vector = 0;

   /*
    * MSIX vector 0 is used by default status block, and slow path;
    * * fastpath status block, Tx/Rx started from vector 1
    */
   for (i = 0; i < adapter->num_rxqs_vmk; i++) {
      rxq = &adapter->fp[i];
      vmk_NameFormat(&rxq->name, "%s-fp-%d",
		     QFLE3_NAME_TO_STRING(adapter->pdev_name), i);
      status = qfle3_register_interrupt(adapter, (vector + 1),
					rxq->name, rxq,
					qfle3_interrupt_ack, qfle3_msix_rx);
      if (status != VMK_OK) {
	 QFLE3_ERR("Failed to initiate intrCookie 0x%x (%x)",
		   intr->cookies[vector], status);
	 goto fail_reg;
      }
   }

   return VMK_OK;

  fail_reg:
   for (i = vector - 1; i >= 0; i++) {
      qfle3_unregister_interrupt(adapter, i);
   }
   return status;

}
#endif
VMK_ReturnStatus
qfle3_register_interrupt(qfle3_adapter * adapter,       // IN: adapter
			 vmk_uint32 cookieIdx,  // IN: which interrupt cookie
			 vmk_Name name, // IN: interrupt name
			 void *handlerData,     // IN: private data for handler
			 vmk_IntrAcknowledge ack,       // IN: acknowledging function
			 vmk_IntrHandler handler)       // IN: interrupt handler
{
   qfle3_intr *intr = &adapter->intr;
   vmk_IntrProps props;
   VMK_ReturnStatus status;
   vmk_IntrCookie cookie = intr->cookies[cookieIdx];

   props.device = adapter->device;
   vmk_NameCopy(&props.deviceName, &name);
   props.attrs = VMK_INTR_ATTRS_ENTROPY_SOURCE;
   props.acknowledgeInterrupt = ack;
   props.handler = handler;
   props.handlerData = handlerData;
   status = vmk_IntrRegister(vmk_ModuleCurrentID, cookie, &props);
   if (status != VMK_OK) {
      QFLE3_ERR("Failed to register intrCookie 0x%x (%x)", cookie, status);
      goto fail_intr_reg;
   }

   intr->handlerData[cookieIdx] = handlerData;

  fail_intr_reg:
   return status;
}



VMK_ReturnStatus
qfle3_set_netpoll_interrupts(qfle3_adapter * adapter)
{
   VMK_ReturnStatus status;
   struct qfle3_fastpath *fp;
   qfle3_intr *intr = &adapter->intr;
   vmk_IntrCookie cookie;
   int i=0, fp_num, offset;

   if (intr->intrType != QFLE3_IT_MSIX)
      return VMK_OK;

   offset = 1;
   offset = 1 + CNIC_SUPPORT(adapter);
   FOREACH_ETH_QUEUE(i,fp_num,fp) 
      if (fp->netpoll) {
         if (intr->intrType != QFLE3_IT_MSIX)
            cookie = intr->cookies[0];
         else
            cookie = intr->cookies[offset];
         status = vmk_NetPollInterruptSet(fp->netpoll, cookie);
         if (status != VMK_OK) {
            QFLE3_ERR("Failed to set associated interrupt cookie #%d 0x%x"
               "with fp[%d] netpoll", offset, cookie, fp_num);
            goto fail_set_poll;
         }
      }
      offset++;
   }
   return VMK_OK;

  fail_set_poll:
   for (i--; i >= 0; i--) {
      fp = &adapter->fp[i];
      if (fp->netpoll) {
	 vmk_NetPollInterruptUnSet(fp->netpoll);
      }
   }
   return status;
}

VMK_ReturnStatus
qfle3_unmask_fp_interrupts(qfle3_adapter * adapter)
{
   VMK_ReturnStatus status;
   qfle3_intr *intr = &adapter->intr;
   struct qfle3_fastpath *fp;
   vmk_IntrCookie cookie;
   int i, fp_num, offset;

   if (intr->intrType != QFLE3_IT_MSIX) {
      return VMK_OK;
   }
   /* Slowpath interrupts have already been enabled
    * so start from 1
    */
   offset = 1;
   offset += CNIC_SUPPORT(adapter);

   FOREACH_ETH_QUEUE(i,fp_num,fp) 
      cookie = intr->cookies[offset];
      status = vmk_IntrEnable(cookie);
      if (status != VMK_OK) {
   	 QFLE3_ERR("Failed to enable intrCookie[%d] 0x%x (%x)", offset, cookie, status);
   	 goto fail_intr_enable;
      }
      
      QFLE3_DBG(QFLE3_DBG_QUEUE, "Interrupt enabled on fp %d intr num %d\n", fp_num, offset);
      offset++;
   }
   return VMK_OK;
fail_intr_enable:
   for (; offset >= 1; offset--) {
      vmk_IntrDisable(intr->cookies[offset]);
   }
   return status;
}
VMK_ReturnStatus
qfle3_activate_dev(qfle3_adapter * adapter)
{
   VMK_ReturnStatus status;
   int i,fp_num;
   struct qfle3_fastpath *fpq;
   vmk_UplinkSharedQueueData *sqd = NULL;
   vmk_UplinkSharedQueueInfo *sqi;
   sqi = &adapter->uplinkQueueInfo;


   vmk_LogMessage("Active_dev Entering");

   /*
    * Start default TX/RX queues
    */
   status = qfle3_rx_queue_start(adapter, sqi->defaultRxQueueID);
   if (status != VMK_OK) {
      QFLE3_ERR("Failed to start default RX queue (%x)\n", status);
      goto failed_start_rq;
   }
   status = qfle3_tx_queue_start(adapter, sqi->defaultTxQueueID);
   if (status != VMK_OK) {
      QFLE3_ERR("Failed to start default RX queue (%x)\n", status);
      goto failed_start_tq;
   }

   //* loop through all the rx queues and start the ones that are marked started in the uplink shared area
   FOREACH_RX_VMK_QUEUE(i,fp_num,fpq)
      sqd = QFLE3_RX_QUEUE_SHARED_DATA(adapter);
      sqd += i;
      if ((sqd->flags & VMK_UPLINK_QUEUE_FLAG_IN_USE) &&
          !(sqd->flags & VMK_UPLINK_QUEUE_FLAG_DEFAULT) &&
          fpq->soft_state.q_started_by_vmk &&
          (fpq->fp_state == 0)){
         QFLE3_DBG(QFLE3_DBG_QUEUE, "Reallocing RX q %d\n", i);
         qfle3_sm_q_cmd (adapter, QFLE3_SMCMD_CREATE_Q, i, QFLE3_SM_PARAM_RXQ);
      }
      if (/*(sqd->state == VMK_UPLINK_QUEUE_STATE_STARTED) && */
         !(sqd->flags & VMK_UPLINK_QUEUE_FLAG_DEFAULT) &&
         fpq->soft_state.q_started_by_vmk &&
         (fpq->fp_state == QFLE3_SM_Q_CREATED)) {
         QFLE3_DBG(QFLE3_DBG_QUEUE, "Restarting RX q %d\n", i);
         qfle3_sm_q_cmd (adapter, QFLE3_SMCMD_START_Q, i, QFLE3_SM_PARAM_RXQ);
      }
   }
   //* loop through all the Tx queues and start the ones that are marked started in the uplink shared area
   FOREACH_TX_VMK_QUEUE(i,fp_num,fpq)
      sqd = QFLE3_TX_QUEUE_SHARED_DATA(adapter);
      sqd += i;
      if ((sqd->flags & VMK_UPLINK_QUEUE_FLAG_IN_USE) &&
         !(sqd->flags & VMK_UPLINK_QUEUE_FLAG_DEFAULT) &&
         fpq->soft_state.q_started_by_vmk &&
          (fpq->fp_state == 0)){
         QFLE3_DBG(QFLE3_DBG_QUEUE, "Reallocing TX q %d\n", i);
         qfle3_sm_q_cmd (adapter, QFLE3_SMCMD_CREATE_Q, i, QFLE3_SM_PARAM_TXQ);
      }
      if (/*(sqd->state == VMK_UPLINK_QUEUE_STATE_STARTED) &&*/
         !(sqd->flags & VMK_UPLINK_QUEUE_FLAG_DEFAULT) &&
         fpq->soft_state.q_started_by_vmk &&
         (fpq->fp_state == QFLE3_SM_Q_CREATED)) {
         QFLE3_DBG(QFLE3_DBG_QUEUE, "Restarting TX q %d\n", i);
         qfle3_sm_q_cmd (adapter, QFLE3_SMCMD_START_Q, i, QFLE3_SM_PARAM_TXQ);
      }
   }

#define IS_PF(x) (x)!=NULL
   if (IS_PF(adapter) && adapter->pending_max) {
        qfle3_update_max_mf_config(adapter, adapter->pending_max);
    }

   if (adapter->port.pmf) {
	   if (!adapter->phy_initialized) {
		   status = qfle3_initial_phy_init(adapter, 0);
		   if (status != VMK_OK) {
			   QFLE3_ERROR("initial phy init failure, aborting\n");
		   }
		   adapter->phy_initialized = 1;
	   }
      qfle3_acquire_phy_lock(adapter);
      elink_period_func(&adapter->link_params, &adapter->link_vars);
      qfle3_release_phy_lock(adapter);
      qfle3_link_status_update(adapter);
   } else
      qfle3_link_status_update(adapter);

   adapter->rx_mode = QFLE3_RX_MODE_ALLMULTI;
   qfle3_set_rx_mode(adapter);
   
   vmk_BitVectorAtomicTestAndSet(adapter->state, QFLE3_STATE_BIT_IOSTARTED);
   vmk_LogMessage("Active_dev Exiting");
   return VMK_OK;


failed_start_tq:
   qfle3_rx_queue_stop(adapter, sqi->defaultRxQueueID);
failed_start_rq:

   QFLE3_DBG(QFLE3_DBG_UPLINK,  "Active_dev FAILED");
   return VMK_FAILURE;
}



static VMK_ReturnStatus
qfle3_uplink_start_io(vmk_AddrCookie driverData)
{
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   QFLE3_SMCMD_STATUS cmd_status;
   vmk_UplinkSharedQueueInfo *sqi;
   sqi = &adapter->uplinkQueueInfo;

   QFLE3_DBG(QFLE3_DBG_KERNEL,  "Uplink.startIO");

   if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING)){
      QFLE3_INFO("Uplink is Resetting while Start IO received");
   }

   cmd_status = qfle3_sm_cmd (adapter, QFLE3_SMCMD_RESUMEIO, 0);
   if (cmd_status == QFLE3_SMCMD_STATUS_COMPLETED){

      // Is this code realy necessary?
	   QFLE3_DBG(QFLE3_DBG_SM, "uplinkCachedNewdata 0x%x\n", adapter->uplinkCachedNewData);
	   if (adapter->uplinkCachedNewData){
		   QFLE3_INFO( "Uplink info changed while adapter is starting \n");
		   vmk_BitVectorSet(adapter->state, QFLE3_STATE_BIT_RESETTING);

		   qfle3_link_down(adapter);
		   cmd_status = qfle3_sm_cmd (adapter, QFLE3_SMCMD_UPLINKRESET, 0);
		   //qfle3_link_up(adapter);
		   vmk_BitVectorClear(adapter->state, QFLE3_STATE_BIT_RESETTING);
	   }
      QFLE3_DBG(QFLE3_DBG_KERNEL,  "Uplink.startIO, Done");
	   return VMK_OK;
   } else {
      QFLE3_ERR( "Uplink.startIO, Error");
      return VMK_FAILURE;
   }

}

void
qfle3_link_down(qfle3_adapter * adapter)
{
   vmk_LinkStatus linkStatus;
   vmk_UplinkSharedData *sd;
   QFLE3_DBG(QFLE3_DBG_KERNEL,  "Mark uplink sd down");
   QFLE3_DBG(QFLE3_DBG_STATS,"STATS_EVENT_STOP\n");
   qfle3_stats_handle(adapter, STATS_EVENT_STOP);

   adapter->linkSpeed = 0;
   adapter->linkUp = VMK_FALSE;
   linkStatus.speed = 0;
   linkStatus.state = VMK_LINK_STATE_DOWN;
   linkStatus.duplex = VMK_LINK_DUPLEX_HALF;

   sd = &adapter->uplinkSharedData;
   QFLE3_SHARED_DATA_WRITE_BEGIN(adapter);
   vmk_Memcpy(&sd->link, &linkStatus, sizeof(linkStatus));
   QFLE3_SHARED_DATA_WRITE_END(adapter);

   vmk_UplinkUpdateLinkState(adapter->uplink, &linkStatus);

   vmk_Memset(&adapter->last_reported_link, 0, sizeof(struct qfle3_link_report_data));
   adapter->phy_initialized = 0;
#ifdef QFLE3_CNIC
   if (CNIC_LOADED(adapter)) {
      adapter->cnicOps->cnicLinkUpdate(adapter->cnic_data, 2, 0);
      
      QFLE3_INFO("CNIC Link state reported\n");
   }
#endif
   QFLE3_DBG(QFLE3_DBG_KERNEL,  "Mark uplink sd down, DONE");

}

void
qfle3_link_up(qfle3_adapter * adapter)
{
   vmk_LinkStatus linkStatus;
   vmk_UplinkSharedData *sd;

   adapter->linkSpeed = 0;
   adapter->linkUp = VMK_TRUE;
   linkStatus.speed = 0;
   linkStatus.state = VMK_LINK_STATE_UP;
   linkStatus.duplex = VMK_LINK_DUPLEX_FULL;

   sd = &adapter->uplinkSharedData;
   QFLE3_SHARED_DATA_WRITE_BEGIN(adapter);
   vmk_Memcpy(&sd->link, &linkStatus, sizeof(linkStatus));
   QFLE3_SHARED_DATA_WRITE_END(adapter);

   vmk_UplinkUpdateLinkState(adapter->uplink, &linkStatus);

   vmk_Memset(&adapter->last_reported_link, 0, sizeof(struct qfle3_link_report_data));
}

void
qfle3_mask_fp_interrupts(qfle3_adapter * adapter)
{
   qfle3_intr *intr = &adapter->intr;
   vmk_IntrCookie cookie;
   int i,fp_num, offset;
   struct qfle3_fastpath *fpq;
   /* Slowpath interrupts will be masked later * so start from 1
    */
   offset = 1;
//#ifdef QFLE3_CNIC
//   cookie = intr->cookies[1];
//   vmk_IntrDisable(cookie);
//   if (intr->intrType != QFLE3_IT_INTX) {
//      vmk_IntrSync(cookie);
//   }
//#endif   
   offset = 1+CNIC_SUPPORT(adapter);
   FOREACH_ETH_QUEUE(i,fp_num,fpq)
      cookie = intr->cookies[offset];
      /*
       * It is OK to call vmk_IntrDisable even on
       * shared interrupt as vmkernel supports
       * nesting interrupt vector mask/unmask.
       * However, it is probably not a good idea
       * to sync on shared interrupt.
       */
      vmk_IntrDisable(cookie);
      if (intr->intrType != QFLE3_IT_INTX) {
         vmk_IntrSync(cookie);
      }
      offset++;
   }
   
}

void
qfle3_deactive_dev(qfle3_adapter * adapter)
{
   int i,fp_num;
   struct qfle3_fastpath *fpq;
   vmk_UplinkSharedQueueData *sqd = NULL;
   vmk_UplinkSharedData *sd = NULL;
   vmk_UplinkSharedQueueInfo *sqi;
   sqi = &adapter->uplinkQueueInfo;
   VMK_ReturnStatus status;

   QFLE3_INFO("Deactive_dev Entering");
   if (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_QUIESCED)) {
      QFLE3_ERR("Adapter state is quiesced\n");
      return;
   }
   
   if (!vmk_BitVectorAtomicTestAndClear(adapter->state, QFLE3_STATE_BIT_IOSTARTED)){
      QFLE3_ERR("Uplink state is quiesced\n");
      return;
   }
   QFLE3_INFO("marking link down\n");
   ECORE_SET_BIT(QFLE3_LINK_REPORT_LINK_DOWN,
         &adapter->last_reported_link.link_report_flags);

#if 0
   for (i = adapter->num_rxqs_drv; i < adapter->num_ethqs; i++) {
      vmk_SpinlockLock(adapter->fp[i].fp_lock);
      vmk_SpinlockUnlock(adapter->fp[i].fp_lock);
   }
#endif

   /*
    * issue stop comand to hardware
    */
   sd = &adapter->uplinkSharedData;

   FOREACH_TX_VMK_QUEUE(i,fp_num,fpq)
      sqd = QFLE3_TX_QUEUE_SHARED_DATA(adapter);
      sqd += i;
      if ((sqd->flags & VMK_UPLINK_QUEUE_FLAG_IN_USE) &&
         !(sqd->flags & VMK_UPLINK_QUEUE_FLAG_DEFAULT)) {
	 QFLE3_DBG(QFLE3_DBG_SM, "Invalidating  TX q %d\n", i);
         status = vmk_UplinkQueueInvalidate(adapter->uplink, sqd->qid);
	 if (status == VMK_FAILURE) {
	 	QFLE3_ERR("Invalidating queue not supported\n");
	 }
         if (fpq->fp_state == QFLE3_SM_Q_STARTED)
            qfle3_sm_q_cmd (adapter, QFLE3_SMCMD_STOP_Q, i, QFLE3_SM_PARAM_TXQ);
         if ((fpq->fp_state == QFLE3_SM_Q_CREATED) &&
             !(sqd->flags & VMK_UPLINK_QUEUE_FLAG_DEFAULT))
            qfle3_sm_q_cmd (adapter, QFLE3_SMCMD_REMOVE_Q, i, QFLE3_SM_PARAM_TXQ);
	 if(i);
	    qfle3_tq_free(adapter,sqd->qid);
      }
   }

   FOREACH_RX_VMK_QUEUE(i,fp_num,fpq)
      sqd = QFLE3_RX_QUEUE_SHARED_DATA(adapter);
      sqd += i;
      if ((sqd->flags & VMK_UPLINK_QUEUE_FLAG_IN_USE) &&
         !(sqd->flags & VMK_UPLINK_QUEUE_FLAG_DEFAULT)) {
            QFLE3_DBG(QFLE3_DBG_SM, "Invalidating  RX q %d\n", i);
            status = vmk_UplinkQueueInvalidate(adapter->uplink, sqd->qid);
         if (status == VMK_FAILURE) {
            QFLE3_ERR("Invalidating queue not supported\n");
         }
         if (fpq->fp_state == QFLE3_SM_Q_STARTED)
            qfle3_sm_q_cmd (adapter, QFLE3_SMCMD_STOP_Q, i, QFLE3_SM_PARAM_RXQ);
         if ((fpq->fp_state == QFLE3_SM_Q_CREATED) &&
             !(sqd->flags & VMK_UPLINK_QUEUE_FLAG_DEFAULT))
            qfle3_sm_q_cmd (adapter, QFLE3_SMCMD_REMOVE_Q, i, QFLE3_SM_PARAM_RXQ);
	 if (i)
            qfle3_rq_free(adapter,sqd->qid);
      }
   }

   /*
    * Mark default TX/RX queues stopped. Already stopped
    */
   qfle3_rx_queue_stop(adapter, sqi->defaultRxQueueID);
   qfle3_tx_queue_stop(adapter, sqi->defaultTxQueueID);
   adapter->stats_pending = 0;
   
   QFLE3_INFO("Deactive_dev Exiting");

}

static VMK_ReturnStatus
qfle3_uplink_quiesce_io(vmk_AddrCookie driverData)
{
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   QFLE3_SMCMD_STATUS cmd_status;
   vmk_uint32 wait_for_reset=50;

   QFLE3_DBG(QFLE3_DBG_KERNEL, "Uplink.quiesce_io");
   while (vmk_BitVectorTest(adapter->state, QFLE3_STATE_BIT_RESETTING)){
      QFLE3_ERR("Waiting to QuiesceIO, uplink is resetting");
      vmk_WorldSleep(1000);
      if (!wait_for_reset--){
         vmk_BitVectorAtomicTestAndSet(adapter->state, QFLE3_STATE_BIT_ERROR);
         QFLE3_ERR("Uplink Reset timed out");
         return VMK_FAILURE;
      }
         
   }
   cmd_status = qfle3_sm_cmd (adapter, QFLE3_SMCMD_PAUSEIO, 0);
   if (cmd_status != QFLE3_SMCMD_STATUS_COMPLETED){
      QFLE3_DBG(QFLE3_DBG_UPLINK, "Uplink.quiesce_io: Error Encountered");
   }
   
   QFLE3_DBG(QFLE3_DBG_KERNEL, "Uplink.quiesce_io, Done");
   return VMK_OK;
}


static VMK_ReturnStatus
qfle3_uplink_reset(vmk_AddrCookie driverData)
{
   qfle3_adapter *adapter = (qfle3_adapter *) driverData.ptr;
   vmk_uint16  sm_state;
   QFLE3_SMCMD_STATUS cmd_status;
	
   QFLE3_DBG(QFLE3_DBG_KERNEL, "Uplink.reset");
	
   sm_state = qfle3_sm_getstate(adapter);
   QFLE3_DBG(QFLE3_DBG_SM, "Uplink.uplinkReset state is %s\n", qfle3_sm_get_string(sm_state));
	
   if (vmk_BitVectorAtomicTestAndSet(adapter->state, QFLE3_STATE_BIT_RESETTING)) {
      return VMK_FAILURE;
   }
	
   qfle3_link_down(adapter);
   cmd_status = qfle3_sm_cmd(adapter, QFLE3_SMCMD_UPLINKRESET, 0);
	
   if (cmd_status == QFLE3_SMCMD_STATUS_COMPLETED){
      
      QFLE3_DBG(QFLE3_DBG_KERNEL, "Uplink.reset, Done");
      return VMK_OK;
   }else if (cmd_status == QFLE3_SMCMD_STATUS_WRONGSTATE){
      QFLE3_ERR("Setting MTU in the wrong state %s", qfle3_sm_get_string(sm_state));
      return VMK_FAILURE;
   }else {
      QFLE3_ERR("ERROR Setting MTU");
      return VMK_FAILURE;
   }
}


inline void
qfle3_enable_intr(qfle3_adapter * adapter,      // IN: adapter
		  unsigned idx) // IN: interrupt index
{
   //Todo
}

inline void
qfle3_disable_intr(qfle3_adapter * adapter,     // IN: adapter
		   vmk_uint16 idx)      // IN: interrupt index
{
   //Todo
}

VMK_ReturnStatus
qfle3_geneve_port_update(vmk_AddrCookie cookie, vmk_uint16 port_num)
{
   int status = VMK_OK;
   qfle3_adapter *adapter = (qfle3_adapter *)cookie.ptr;
   struct ecore_func_state_params func_params = {NULL};
   struct ecore_func_switch_update_params *switch_update_params =
           &func_params.params.switch_update;
   int rc;

   port_num = vmk_BE16ToCPU(port_num);

   QFLE3_INFO("Geneve UDP port update: new:%u old:%u\n",
                                     port_num, adapter->geneve_udp_dst_port);

   SET_BIT(RAMROD_COMP_WAIT, &func_params.ramrod_flags);
   SET_BIT(RAMROD_RETRY, &func_params.ramrod_flags);

   func_params.f_obj = &adapter->func_obj;
   func_params.cmd = ECORE_F_CMD_SWITCH_UPDATE;

   SET_BIT(ECORE_F_UPDATE_TUNNEL_CFG_CHNG,
           &switch_update_params->changes);

   switch_update_params->geneve_dst_port = port_num;

   /* Enable RSS based on updated port number. */
   SET_BIT(ECORE_F_UPDATE_TUNNEL_INNER_RSS,
           &switch_update_params->changes);

   rc = ecore_func_state_change(adapter, &func_params);
   if (rc) {
      QFLE3_ERR("Failed to change Geneve dst port to %d (rc = 0x%x)\n",
                port_num, rc);
      status = VMK_FAILURE;
   }

   /* save the updated GENEVE UDP port number. */
   adapter->geneve_udp_dst_port = port_num;

   return status;
}

//inline void
//qfle3_enable_all_intrs(qfle3_adapter * adapter)
//{
//   int i;

//   for (i = 0; i < adapter->intr.numEthIntrs; i++) {
//      qfle3_enable_intr(adapter, i);
//   }
//}

//inline void
//qfle3_disable_all_intrs(qfle3_adapter * adapter)
//{
//   int i;

//   for (i = 0; i < adapter->intr.numEthIntrs; i++) {
//      qfle3_disable_intr(adapter, i);
//   }
//}

