from typing import Optional, List

from pydantic import BaseModel
from sqlalchemy import Column, Integer, ForeignKey, String, Double, Boolean
from sqlalchemy.orm import relationship

from ..base import Base


class ThreadHotspotRecord(Base):
    __tablename__ = 'thread_hotspot_record'
    id = Column(Integer, primary_key=True, index=True)
    name = Column(String, nullable=True)
    value = Column(Double, nullable=True)
    duration = Column(Double, nullable=True)
    sample_count = Column(Integer, nullable=True)
    start_time = Column(Double, nullable=True)
    extra_1 = Column(String, nullable=True)
    extra_2 = Column(String, nullable=True)

    thread_hotspot_collection_id = Column(Integer, ForeignKey('thread_hotspot_collection.id'))
    thread_hotspot_collection = relationship('ThreadHotspotCollection', back_populates='thread_hotspot_records')


class ThreadHotspotRecordInDB(BaseModel):
    id: int
    name: Optional[str]
    value: Optional[float]
    duration: Optional[float]
    sample_count: Optional[int]
    start_time: Optional[float]
    extra_1: Optional[str]
    extra_2: Optional[str]

    class Config:
        from_attributes = True


class ThreadHotspotRecordCreate(BaseModel):
    name: Optional[str]
    value: Optional[float]
    duration: Optional[float]
    sample_count: Optional[int]
    start_time: Optional[float]
    extra_1: Optional[str]
    extra_2: Optional[str]

    class Config:
        from_attributes = True


class ThreadHotspotCollection(Base):
    __tablename__ = 'thread_hotspot_collection'
    id = Column(Integer, primary_key=True, index=True)

    type = Column(String)
    name = Column(String, default='')
    start_time_ns = Column(Double, default=0)
    end_time_ns = Column(Double, default=0)
    raw_total_samples = Column(Integer, nullable=True)
    raw_total_duration = Column(Double, nullable=True)
    sample_string = Column(String, default='Count')
    start_time_string = Column(String, default='Start Time (ns)')
    duration_string = Column(String, default='Duration (ms)')
    unit_string = Column(String, default='% of Frame Duration')
    name_string = Column(String, default='')
    is_duration = Column(Boolean, nullable=True)
    extra_1_string = Column(String, nullable=True)
    extra_2_string = Column(String, nullable=True)

    view_type = Column(String, default='table')

    region_thread_id = Column(Integer, ForeignKey('region_thread.id'))
    region_thread = relationship('RegionThread', back_populates='thread_hotspot_collections')

    thread_hotspot_records = relationship(ThreadHotspotRecord, back_populates='thread_hotspot_collection')


class ThreadHotspotCollectionInDB(BaseModel):
    id: int

    type: str
    name: str
    start_time_ns: float
    end_time_ns: float
    raw_total_samples: Optional[int]
    raw_total_duration: Optional[float]
    sample_string: str
    start_time_string: str
    duration_string: str
    unit_string: str
    name_string: str
    is_duration: bool
    extra_1_string: Optional[str]
    extra_2_string: Optional[str]
    view_type: str

    thread_hotspot_records: List[ThreadHotspotRecordInDB]

    class Config:
        from_attributes = True


class ThreadHotspotCollectionBaseCreate(BaseModel):
    type: str
    name: Optional[str]
    start_time_ns: float
    end_time_ns: float
    raw_total_samples: Optional[int]
    raw_total_duration: Optional[float]
    sample_string: str
    start_time_string: Optional[str]
    duration_string: str
    unit_string: str
    name_string: str
    is_duration: bool
    extra_1_string: Optional[str]
    extra_2_string: Optional[str]
    view_type: Optional[str]

    class Config:
        from_attributes = True


class ThreadHotspotCollectionCreate(ThreadHotspotCollectionBaseCreate):
    thread_hotspot_records: List[ThreadHotspotRecordCreate]

    class Config:
        from_attributes = True
