"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useLoadToCache = exports.useLoadTablesToCache = exports.useLoadTableColumnsToCache = exports.useLoadDatabasesToCache = exports.useLoadAccelerationsToCache = exports.updateToCache = exports.updateTablesToCache = exports.updateTableColumnsToCache = exports.updateDatabasesToCache = exports.updateAccelerationsToCache = exports.createLoadQuery = void 0;
var _react = require("react");
var _constants = require("../constants");
var _types = require("../types");
var _query_session_utils = require("../utils/query_session_utils");
var _shared = require("../utils/shared");
var _use_polling = require("../utils/use_polling");
var _sql = require("../requests/sql");
var _cache_manager = require("./cache_manager");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const updateDatabasesToCache = (dataSourceName, pollingResult, dataSourceMDSId) => {
  const cachedDataSource = _cache_manager.CatalogCacheManager.getOrCreateDataSource(dataSourceName, dataSourceMDSId);
  const currentTime = new Date().toUTCString();
  if (!pollingResult) {
    _cache_manager.CatalogCacheManager.addOrUpdateDataSource({
      ...cachedDataSource,
      databases: [],
      lastUpdated: currentTime,
      status: _types.CachedDataSourceStatus.Failed,
      ...(dataSourceMDSId && {
        dataSourceMDSId
      })
    }, dataSourceMDSId);
    return;
  }
  const combinedData = (0, _shared.combineSchemaAndDatarows)(pollingResult.schema, pollingResult.datarows);
  const newDatabases = combinedData.map(row => ({
    name: row.namespace,
    tables: [],
    lastUpdated: '',
    status: _types.CachedDataSourceStatus.Empty
  }));
  _cache_manager.CatalogCacheManager.addOrUpdateDataSource({
    ...cachedDataSource,
    databases: newDatabases,
    lastUpdated: currentTime,
    status: _types.CachedDataSourceStatus.Updated,
    ...(dataSourceMDSId && {
      dataSourceMDSId
    })
  }, dataSourceMDSId);
};
exports.updateDatabasesToCache = updateDatabasesToCache;
const updateTablesToCache = (dataSourceName, databaseName, pollingResult, dataSourceMDSId) => {
  try {
    const cachedDatabase = _cache_manager.CatalogCacheManager.getDatabase(dataSourceName, databaseName, dataSourceMDSId);
    const currentTime = new Date().toUTCString();
    if (!pollingResult) {
      _cache_manager.CatalogCacheManager.updateDatabase(dataSourceName, {
        ...cachedDatabase,
        tables: [],
        lastUpdated: currentTime,
        status: _types.CachedDataSourceStatus.Failed
      }, dataSourceMDSId);
      return;
    }
    const combinedData = (0, _shared.combineSchemaAndDatarows)(pollingResult.schema, pollingResult.datarows);
    const newTables = combinedData.filter(row => !_constants.SPARK_HIVE_TABLE_REGEX.test(row.information)).map(row => ({
      name: row.tableName
    }));
    _cache_manager.CatalogCacheManager.updateDatabase(dataSourceName, {
      ...cachedDatabase,
      tables: newTables,
      lastUpdated: currentTime,
      status: _types.CachedDataSourceStatus.Updated
    }, dataSourceMDSId);
  } catch (error) {
    // eslint-disable-next-line no-console
    console.error(error);
  }
};
exports.updateTablesToCache = updateTablesToCache;
const updateAccelerationsToCache = (dataSourceName, pollingResult, dataSourceMDSId) => {
  const currentTime = new Date().toUTCString();
  if (!pollingResult) {
    _cache_manager.CatalogCacheManager.addOrUpdateAccelerationsByDataSource({
      name: dataSourceName,
      accelerations: [],
      lastUpdated: currentTime,
      status: _types.CachedDataSourceStatus.Failed,
      ...(dataSourceMDSId && {
        dataSourceMDSId
      })
    });
    return;
  }
  const combinedData = (0, _shared.combineSchemaAndDatarows)(pollingResult.schema, pollingResult.datarows);
  const newAccelerations = combinedData.map(row => ({
    flintIndexName: row.flint_index_name,
    type: row.kind === 'mv' ? 'materialized' : row.kind,
    database: row.database,
    table: row.table,
    indexName: row.index_name,
    autoRefresh: row.auto_refresh,
    status: row.status
  }));
  _cache_manager.CatalogCacheManager.addOrUpdateAccelerationsByDataSource({
    name: dataSourceName,
    accelerations: newAccelerations,
    lastUpdated: currentTime,
    status: _types.CachedDataSourceStatus.Updated,
    ...(dataSourceMDSId && {
      dataSourceMDSId
    })
  });
};
exports.updateAccelerationsToCache = updateAccelerationsToCache;
const updateTableColumnsToCache = (dataSourceName, databaseName, tableName, pollingResult, dataSourceMDSId) => {
  try {
    if (!pollingResult) {
      return;
    }
    const cachedDatabase = _cache_manager.CatalogCacheManager.getDatabase(dataSourceName, databaseName, dataSourceMDSId);
    const currentTime = new Date().toUTCString();

    // @ts-expect-error TS2322 TODO(ts-error): fixme
    const combinedData = (0, _shared.combineSchemaAndDatarows)(pollingResult.schema, pollingResult.datarows);
    const tableColumns = [];
    for (const row of combinedData) {
      if (row.col_name === _constants.SPARK_PARTITION_INFO) {
        break;
      }
      tableColumns.push({
        fieldName: row.col_name,
        dataType: row.data_type
      });
    }
    const newTables = cachedDatabase.tables.map(ts => ts.name === tableName ? {
      ...ts,
      columns: tableColumns
    } : {
      ...ts
    });
    if (cachedDatabase.status === _types.CachedDataSourceStatus.Updated) {
      _cache_manager.CatalogCacheManager.updateDatabase(dataSourceName, {
        ...cachedDatabase,
        tables: newTables,
        lastUpdated: currentTime,
        status: _types.CachedDataSourceStatus.Updated
      }, dataSourceMDSId);
    }
  } catch (error) {
    // eslint-disable-next-line no-console
    console.error(error);
  }
};
exports.updateTableColumnsToCache = updateTableColumnsToCache;
const updateToCache = (pollResults, loadCacheType, dataSourceName, databaseName, tableName, dataSourceMDSId) => {
  switch (loadCacheType) {
    case 'databases':
      updateDatabasesToCache(dataSourceName, pollResults, dataSourceMDSId);
      break;
    case 'tables':
      updateTablesToCache(dataSourceName, databaseName, pollResults, dataSourceMDSId);
      break;
    case 'accelerations':
      updateAccelerationsToCache(dataSourceName, pollResults, dataSourceMDSId);
      break;
    case 'tableColumns':
      updateTableColumnsToCache(dataSourceName, databaseName, tableName, pollResults, dataSourceMDSId);
    default:
      break;
  }
};
exports.updateToCache = updateToCache;
const createLoadQuery = (loadCacheType, dataSourceName, databaseName, tableName) => {
  let query;
  switch (loadCacheType) {
    case 'databases':
      query = `SHOW SCHEMAS IN ${(0, _shared.addBackticksIfNeeded)(dataSourceName)}`;
      break;
    case 'tables':
      query = `SHOW TABLE EXTENDED IN ${(0, _shared.addBackticksIfNeeded)(dataSourceName)}.${(0, _shared.addBackticksIfNeeded)(databaseName)} LIKE '*'`;
      break;
    case 'accelerations':
      query = `SHOW FLINT INDEX in ${(0, _shared.addBackticksIfNeeded)(dataSourceName)}`;
      break;
    case 'tableColumns':
      query = `DESC ${(0, _shared.addBackticksIfNeeded)(dataSourceName)}.${(0, _shared.addBackticksIfNeeded)(databaseName)}.${(0, _shared.addBackticksIfNeeded)(tableName)}`;
      break;
    default:
      query = '';
      break;
  }
  return query;
};
exports.createLoadQuery = createLoadQuery;
const useLoadToCache = (loadCacheType, http, notifications) => {
  const sqlService = new _sql.SQLService(http);
  const [currentDataSourceName, setCurrentDataSourceName] = (0, _react.useState)('');
  const [currentDatabaseName, setCurrentDatabaseName] = (0, _react.useState)('');
  const [currentTableName, setCurrentTableName] = (0, _react.useState)('');
  const [loadStatus, setLoadStatus] = (0, _react.useState)(_types.DirectQueryLoadingStatus.INITIAL);
  const dataSourceMDSClientId = (0, _react.useRef)('');
  const {
    data: pollingResult,
    loading: _pollingLoading,
    error: pollingError,
    startPolling,
    stopPolling: stopLoading
  } = (0, _use_polling.usePolling)(params => {
    return sqlService.fetchWithJobId(params, dataSourceMDSClientId.current);
  }, _constants.ASYNC_POLLING_INTERVAL);
  const onLoadingFailed = () => {
    setLoadStatus(_types.DirectQueryLoadingStatus.FAILED);
    updateToCache(null, loadCacheType, currentDataSourceName, currentDatabaseName, currentTableName, dataSourceMDSClientId.current);
  };
  const startLoading = ({
    dataSourceName,
    dataSourceMDSId,
    databaseName,
    tableName
  }) => {
    setLoadStatus(_types.DirectQueryLoadingStatus.SCHEDULED);
    setCurrentDataSourceName(dataSourceName);
    setCurrentDatabaseName(databaseName);
    setCurrentTableName(tableName);
    dataSourceMDSClientId.current = dataSourceMDSId || '';
    let requestPayload = {
      lang: 'sql',
      query: createLoadQuery(loadCacheType, dataSourceName, databaseName, tableName),
      datasource: dataSourceName
    };
    const sessionId = (0, _query_session_utils.getAsyncSessionId)(dataSourceName);
    if (sessionId) {
      requestPayload = {
        ...requestPayload,
        sessionId
      };
    }
    sqlService.fetch(requestPayload, dataSourceMDSId).then(result => {
      (0, _query_session_utils.setAsyncSessionId)(dataSourceName, (0, _shared.get)(result, 'sessionId', null));
      if (result.queryId) {
        startPolling({
          queryId: result.queryId
        });
      } else {
        // eslint-disable-next-line no-console
        console.error('No query id found in response');
        onLoadingFailed();
      }
    }).catch(e => {
      var _e$body;
      onLoadingFailed();
      const formattedError = (0, _shared.formatError)('', 'The query failed to execute and the operation could not be complete.', (_e$body = e.body) === null || _e$body === void 0 ? void 0 : _e$body.message);
      notifications.toasts.addError(formattedError, {
        title: 'Query Failed'
      });
      // eslint-disable-next-line no-console
      console.error(e);
    });
  };
  (0, _react.useEffect)(() => {
    // cancel direct query
    if (!pollingResult) return;
    const {
      status: anyCaseStatus,
      datarows,
      error
    } = pollingResult;
    const status = anyCaseStatus === null || anyCaseStatus === void 0 ? void 0 : anyCaseStatus.toLowerCase();
    if (status === _types.DirectQueryLoadingStatus.SUCCESS || datarows) {
      setLoadStatus(status);
      stopLoading();
      updateToCache(pollingResult, loadCacheType, currentDataSourceName, currentDatabaseName, currentTableName, dataSourceMDSClientId.current);
    } else if (status === _types.DirectQueryLoadingStatus.FAILED) {
      onLoadingFailed();
      stopLoading();
      const formattedError = (0, _shared.formatError)('', 'The query failed to execute and the operation could not be complete.', error);
      notifications.toasts.addError(formattedError, {
        title: 'Query Failed'
      });
    } else {
      setLoadStatus(status);
    }

    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [pollingResult, pollingError]);
  return {
    loadStatus,
    startLoading,
    stopLoading
  };
};
exports.useLoadToCache = useLoadToCache;
const useLoadDatabasesToCache = (http, notifications) => {
  const {
    loadStatus,
    startLoading,
    stopLoading
  } = useLoadToCache('databases', http, notifications);
  return {
    loadStatus,
    startLoading,
    stopLoading
  };
};
exports.useLoadDatabasesToCache = useLoadDatabasesToCache;
const useLoadTablesToCache = (http, notifications) => {
  const {
    loadStatus,
    startLoading,
    stopLoading
  } = useLoadToCache('tables', http, notifications);
  return {
    loadStatus,
    startLoading,
    stopLoading
  };
};
exports.useLoadTablesToCache = useLoadTablesToCache;
const useLoadTableColumnsToCache = (http, notifications) => {
  const {
    loadStatus,
    startLoading,
    stopLoading
  } = useLoadToCache('tableColumns', http, notifications);
  return {
    loadStatus,
    startLoading,
    stopLoading
  };
};
exports.useLoadTableColumnsToCache = useLoadTableColumnsToCache;
const useLoadAccelerationsToCache = (http, notifications) => {
  const {
    loadStatus,
    startLoading,
    stopLoading
  } = useLoadToCache('accelerations', http, notifications);
  return {
    loadStatus,
    startLoading,
    stopLoading
  };
};
exports.useLoadAccelerationsToCache = useLoadAccelerationsToCache;